# Copyright (C) 2007, 2010 Ian Zimmerman <itz@buug.org>

# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the conditions spelled out in
# the file LICENSE are met.

from __future__ import with_statement , absolute_import

import time
import os
import errno
import random
import logging
from contextlib import contextmanager
from socket import gethostname

class Error (Exception):
    """Base class for exceptions specific to the lock module."""

class LockError (Error):
    """Exception raised when a file locking operation times out."""
    def __init__ (self, filename, pid, host):
        """The exception constructor; filename is the name of the original file,
        i.e. without the .lock suffix.
        """
        self.filename = filename
        self.pid = pid
        self.host = host

def lock (f, wait = 5.0, tries = 4, callback = None, allow_recursive=False):
    """Attempt to create a lock file named after the file f.
    Raise LockError if unable to create the lock file after tries attempts;
    the interval between attempts, initially wait, increases exponentially
    with some random variation to avoid starvation scenarios.
    The function like object callback is called (passing the lock file name
    and the number of attempts), if not None, after each failed attempt.
    """
    lockfile = f + '.lock'
    current_sleep = wait
    pid , host = None, None
    for i in range (tries):
        try:
            lfd = os.open (lockfile, os.O_CREAT|os.O_WRONLY|os.O_EXCL, 0444)
            break
        except OSError as inst:
            if inst.errno != errno.EEXIST:
                raise
            try:
                with open(lockfile) as lfh:
                    pid = int(lfh.readline())
                    host = lfh.readline().strip()
            except:
                pass
            else:
                if host == gethostname():
                    if allow_recursive and pid == os.getpid():
                        return lockfile, True
            if callback is not None:
                callback (lockfile, i + 1)
            time.sleep (current_sleep)
            current_sleep = current_sleep * (1.5 + random.random ())
    else:
        logging.getLogger ('sortmail').error ('cannot create lockfile %s after %d tries',
                                                   lockfile, tries)
        raise LockError (f, pid, host)
    lock_string = '{0:>10}\n{1}\n'.format(os.getpid(), gethostname())
    with os.fdopen(lfd, 'w') as lfh:
        try:
            lfh.write(lock_string)
        except:
            os.remove(lockfile)
            raise
    return lockfile, False

def unlock (f):
    """Remove the lock file named after the file f, created by lock (f)."""
    os.remove (f + '.lock')


@contextmanager
def unlocking(f, wait=5.0, tries=4, callback=None):
    lockfile, recursive = lock(f, wait, tries, callback)
    try:
        yield recursive
    finally:
        if not recursive:
            os.remove(lockfile)

def with_lock (fn, fun, wait = 5.0, tries = 4, callback = None):
    """Call the function like object fun, locally locking the file fn using
    lock (fn, wait, tries, callback).  The lock file is removed when with_lock
    returns or propagates an exception.
    """
    with unlocking(fn, wait, tries, callback):
        return fun(fn)
