# Copyright (C) 2007, 2010 Ian Zimmerman <itz@buug.org>

# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the conditions spelled out in
# the file LICENSE are met.

from __future__ import absolute_import

import quopri
import base64
import re

class Error (Exception):
    """Base class for exceptions specific to the rfc2047 module."""

class InvalidEncodingError (Error):
    """Exception raised when an invalid encoding is passed to encode ()."""
    def __init__ (self, invalid):
        """The exception constructor; invalid is the string passed to encode ()."""
        self.invalid = invalid

_re_rfc2047 = re.compile (r'=\?([-a-zA-Z0-9]+)\?([qQbB])\?([^?]*)\?=')

def encode (s, charset, encoding):
    """Utility function to create "encoded words" in headers per RFC 2047."""
    if encoding == 'q' or encoding == 'Q':
        return '=?' + charset + '?' + encoding + '?' + quopri.encodestring (s, True) + '?='
    elif encoding == 'b' or encoding == 'B':
        return '=?' + charset + '?' + encoding + '?' + base64.standard_b64encode (s) + '?='
    else: raise InvalidEncodingError (encoding)

def make_string (pieces):
    """Make a single string from a list of (string, charset, encoding) triples.
    Plain strings in the list are interpolated as is.
    """
    result = ''
    for p in pieces:
        try:
            result += p
        except TypeError:
            (s, chs, enc) = p
            result += encode (s, chs, enc)
    return result

def decode (match):
    """Utility function to decode a single "encoded word" in headers per RFC 2047.
    match is a re.match object describing the encoded bit: match.group (1) is the
    charset, match.group (2) is the encoding, and match.group (3) the actual encoded text.
    Return a tuple of (charset, decoded_word).
    """
    charset = match.group (1)
    encoding = match.group (2)
    entext = match.group (3)
    if encoding == 'q' or encoding == 'Q':
        return (charset, quopri.decodestring (entext, True))
    elif encoding == 'b' or encoding == 'B':
        return (charset, base64.standard_b64decode (entext))

def decode_no_charset (match):
    """Like decode (), but only return the decoded word."""
    encoding = match.group (2)
    entext = match.group (3)
    if encoding == 'q' or encoding == 'Q':
        return quopri.decodestring (entext, True)
    elif encoding == 'b' or encoding == 'B':
        return base64.standard_b64decode (entext)

def decode_string (s):
    """Decode an entire contents of a header per RFC 2047.
    Return the decoded text.  This function is suitable for inclusion
    among sortmail.header_cxform.
    """
    return _re_rfc2047.sub (decode_no_charset, s)

