#! /usr/bin/python
# -*- coding: iso-8859-1 -*-

# Copyright (C) 2007, 2010 Ian Zimmerman <itz@buug.org>

# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the conditions spelled out in
# the file LICENSE are met.

from __future__ import with_statement

__author__ = 'Ian Zimmerman <itz@buug.org>'
__date__ = '2007-06-13'

import unittest
import tempfile
import sortmail
from sortmail import rfc2047, msg, destination, lock
import logging
import shutil
import os
import sys
import signal
from StringIO import StringIO
import difflib
import time
import re
from socket import gethostname

def lock_callback (lockfile, tries):
    if tries >= 2:
        os.write (1, ' ')

class TestSortmail (unittest.TestCase):

    def setUp (self):
        self.td = tempfile.mkdtemp ()
        self.maildir = self.td + '/maildir'
        os.mkdir (self.maildir)
        self.tmpdir = self.maildir + '/tmp'
        os.mkdir (self.tmpdir)
        self.newdir = self.maildir + '/new'
        os.mkdir (self.newdir)
        self.handler = logging.FileHandler (self.td + '/foo')
        self.handler.setFormatter (sortmail.log_formatter)
        logger = logging.getLogger ('sortmail')
        self.oldlevel = logger.getEffectiveLevel ()
        logger.addHandler (self.handler)
        logger.setLevel (logging.DEBUG)
        self.dbfile = self.td + '/db~'

    def tearDown (self):
        logger = logging.getLogger ('sortmail')
        logger.setLevel (self.oldlevel)
        logger.removeHandler (self.handler)
        shutil.rmtree (self.td)

class TestLock (TestSortmail):

    def test_explicit (self):
        fn = self.td + '/bar'
        lock.lock (fn)
        self.assert_ (os.access (fn + '.lock', os.F_OK))
        lock.unlock (fn)
        self.assert_ (not os.access (fn + '.lock', os.F_OK))

    def test_interlock (self):
        logger = logging.getLogger ('sortmail')
        fn = self.td + '/bar'
        lock.lock (fn)
        pr, pw = os.pipe ()
        kid = os.fork ()
        if kid != 0:
            # parent code
            os.close (pw)
            os.read (pr, 1)
        else:                           # child
            sys.stdout.close ()
            os.dup2 (pw, 1)
            lock.lock (fn, callback = lock_callback)
            logger.debug ('after child lock')
            lock.unlock (fn)
            os.write (1, ' ')
            os.kill (os.getpid (), signal.SIGKILL)
        # parent code after pipe read returns
        logger.debug ('before parent unlock')
        lock.unlock (fn)
        os.read (pr, 1)
        logfh = open (self.td + '/foo', 'r')
        loglines = logfh.readlines ()
        self.assert_ (loglines [0].find ('parent unlock') != -1)
        self.assert_ (loglines [1].find ('child lock') != -1)

test_headers = [('From', 'me_myself_I@localhost\n'),
                ('X-Strange', 'one two-three four_five six.seven@eight\n'),
                ('X-Multiline', 'line one\n line two\n'),
                ('To', 'you_vous_Vy@nowhere.one.org (You Fish)\n'),
                ('Sender', 'my_own_list@nowhere.two.org\n'),
                ('X-Strange', 'and a second incarnation\n'),
                ('Date', 'Sat, 30 Jun 2007 11:56:36 -0400 (EDT)\n'),
                ('Message-ID', '<wiet0Waenaichujo@unicorn.ahiker.homeip.net>\n')]

body_short = ("From Myself, blah blah blah, this is a really stupid test message.\n" +
              "Another line,\n" +
              "\n" +
              "and an empty one for fun.\n"
              )

test_data_short = (''.join (map (lambda (t, c): t + ': ' + c, test_headers)) +
                   "\n" + body_short
                   )

test_data_short_from = ('From me_myself_I@localhost bla bla\n' +
                        ''.join (map (lambda (t, c): t + ': ' + c, test_headers)) +
                        '\n>' + body_short
                        )

num_foo = 5000

test_data_long = test_data_short + 'foo\n' * num_foo

test_data_long_from = test_data_short_from + 'foo\n' * num_foo

ngs = ['comp.lang.functional', 'comp.lang.haskell', 'comp.lang.python', 'sci.math.research']

test_list_data = ('From: me_myself_I@localhost\n' +
                  'To: you_vous_Vy@nowhere.one.org (You Fish), funnylist@nowhere.two.org\n' +
                  'List-Id: Funny List <funnylist.nowhere.two.org>\n' +
                  'List-Post: <mailto:funnylist@nowhere.two.org>\n' +
                  'Newsgroups: ' + ngs [0] + ',' + ngs [1] + ', ' + ngs [2] + ',' + ngs [3] + ' \n' +
                  '\n' + body_short
                  )

test_list_address_exact = 'funnylist@nowhere.two.org'

test_list_address_case = test_list_address_exact.replace ('funnylist', 'FuNNyLiSt').replace ('two', 'TWO')

efb = 'Mdchen\tEffie Briest=tot'

efb_enc = 'iso-8859-1'

efb_q = 'M=E4dchen=09Effie=20Briest=3Dtot'

efb_b = 'TeRkY2hlbglFZmZpZSBCcmllc3Q9dG90'

efb_p = '<effie> '

calculus_enc = 'utf-8'

calculus = u'dvanhorn @ \u03bb-calcul.us'.encode(calculus_enc)

calculus_q = 'dvanhorn_=40_=CE=BB-calcul=2Eus'

calculus_p = ' <bozo@dev.null.invalid>'

expected_received_ips = {
    'dhl.mail': [
        ['198', '144', '194', '12'],
        ['127', '0', '0', '1'],
        ['122', '164', '50', '79'],
        ['174', '22', '157', '113']
        ],
    'frank_john.mail': [
        ['198', '144', '194', '12'],
        ['127', '0', '0', '1'],
        ['98', '139', '91', '198'],
        ['98', '139', '91', '63'],
        ['98', '139', '91', '9'],
        ['127', '0', '0', '1'],
        ['41', '82', '133', '243']
        ],
    'gentleman.mail': [
        ['198', '144', '194', '12'],
        ['127', '0', '0', '1'],
        ['122', '164', '30', '60'],
        ['158', '135', '182', '188'],
        ['183', '129', '159', '161']
        ]
    }

def xform ((t, foo)):
    return (t, foo.replace ('injected life', 'injected line'))

class TestConstruct (TestSortmail):
    
    def test_short_filehandle (self):
        sort = msg.Msg (source = StringIO (test_data_short))
        self.assertEqual (sort._envelope_from, os.getenv ('LOGNAME', 'nobody') + '@localhost')
        self.assertEqual (sort._headers, test_headers)
        self.assertEqual (sort._body,
                          body_short.splitlines (True))
        self.assertEqual (sort._tf, None)

    def test_short_filehandle_from (self):
        sort = msg.Msg (source = StringIO (test_data_short_from))
        self.assertEqual (sort._envelope_from, 'me_myself_I@localhost')
        self.assertEqual (sort._headers, test_headers)
        self.assertEqual (sort._body,
                          ('>' + body_short).splitlines (True))
        self.assertEqual (sort._tf, None)

    def test_long_filehandle (self):
        sort = msg.Msg (source = StringIO (test_data_long))
        self.assertEqual (sort._envelope_from, os.getenv ('LOGNAME', 'nobody') + '@localhost')
        self.assertEqual (sort._headers, test_headers)
        all_body = body_short + 'foo\n' * num_foo
        self.assertEqual (sort.body (), all_body)

    def test_long_filehandle_from (self):
        sort = msg.Msg (source = StringIO (test_data_long_from))
        self.assertEqual (sort._envelope_from, 'me_myself_I@localhost')
        self.assertEqual (sort._headers, test_headers)
        all_body = '>' + body_short + 'foo\n' * num_foo
        self.assertEqual (sort.body (), all_body)

    def test_short_string (self):
        sort = msg.Msg (source =  (test_data_short))
        self.assertEqual (sort._envelope_from, os.getenv ('LOGNAME', 'nobody') + '@localhost')
        self.assertEqual (sort._headers, test_headers)
        self.assertEqual (sort._body,
                          body_short.splitlines (True))
        self.assertEqual (sort._tf, None)

    def test_short_string_from (self):
        sort = msg.Msg (source =  (test_data_short_from))
        self.assertEqual (sort._envelope_from, 'me_myself_I@localhost')
        self.assertEqual (sort._headers, test_headers)
        self.assertEqual (sort._body,
                          ('>' + body_short).splitlines (True))
        self.assertEqual (sort._tf, None)

    def test_long_string (self):
        sort = msg.Msg (source =  (test_data_long))
        self.assertEqual (sort._envelope_from, os.getenv ('LOGNAME', 'nobody') + '@localhost')
        self.assertEqual (sort._headers, test_headers)
        all_body = body_short + 'foo\n' * num_foo
        self.assertEqual (sort.body (), all_body)

    def test_long_string_from (self):
        sort = msg.Msg (source =  (test_data_long_from))
        self.assertEqual (sort._envelope_from, 'me_myself_I@localhost')
        self.assertEqual (sort._headers, test_headers)
        all_body = '>' + body_short + 'foo\n' * num_foo
        self.assertEqual (sort.body (), all_body)

    def test_short_object (self):
        orig_msg = msg.Msg (test_data_short)
        sort = msg.Msg (source = orig_msg, envelope_from = orig_msg._envelope_from)
        self.assertEqual (sort._envelope_from, os.getenv ('LOGNAME', 'nobody') + '@localhost')
        self.assertEqual (sort._headers, test_headers)
        self.assertEqual (sort._body,
                          body_short.splitlines (True))
        self.assertEqual (sort._tf, None)

    def test_short_object_from (self):
        orig_msg = msg.Msg (test_data_short_from)
        sort = msg.Msg (source = orig_msg, envelope_from = orig_msg._envelope_from)
        self.assertEqual (sort._envelope_from, 'me_myself_I@localhost')
        self.assertEqual (sort._headers, test_headers)
        self.assertEqual (sort._body,
                          ('>' + body_short).splitlines (True))
        self.assertEqual (sort._tf, None)

    def test_long_object (self):
        orig_msg = msg.Msg (test_data_long)
        sort = msg.Msg (source = orig_msg, envelope_from = orig_msg._envelope_from)
        self.assertEqual (sort._envelope_from, os.getenv ('LOGNAME', 'nobody') + '@localhost')
        self.assertEqual (sort._headers, test_headers)
        all_body = body_short + 'foo\n' * num_foo
        self.assertEqual (sort.body (), all_body)

    def test_long_object_from (self):
        orig_msg = msg.Msg (test_data_long_from)
        sort = msg.Msg (source = orig_msg, envelope_from = orig_msg._envelope_from)
        self.assertEqual (sort._envelope_from, 'me_myself_I@localhost')
        self.assertEqual (sort._headers, test_headers)
        all_body = '>' + body_short + 'foo\n' * num_foo
        self.assertEqual (sort.body (), all_body)

class TestHeaders (TestSortmail):

    def test_matches (self):
        sort = msg.Msg (source = test_data_short)
        matches = sort.header_match ('from', '.*myself')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 0)
        matches = sort.header_match ('From', '.*Myself')
        self.assertEqual (len (matches), 0)
        matches = sort.header_match ('from', '.*m._(m.)')
        self.assertEqual (len (matches), 1)
        (i, (mt, mc)) = matches [0]
        self.assertEqual (i, 0)
        self.assertEqual (mc.group (1), 'my')
        matches = sort.header_start ('to', 'you')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 3)
        matches = sort.header_start ('to', 'vous')
        self.assertEqual (len (matches), 0)
        matches = sort.header_start ('x-multiline', 'line one')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 2)
        matches = sort.header_start ('x-multiline', 'line two')
        self.assertEqual (len (matches), 0)
        matches = sort.header_match ('x-multiline', '.*one\n line')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 2)
        matches = sort.destination_match ('.*(one)')
        self.assertEqual (len (matches), 1)
        (i, (mt, mc)) = matches [0]
        self.assertEqual (i, 3)
        self.assertEqual (mc.group (1), 'one')
        matches = sort.destination_match ('.*two')
        self.assertEqual (len (matches), 0)
        matches = sort.destination_address ('you_vous_Vy@nowhere.one.org')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 3)
        matches = sort.destination_address ('vous_Vy@nowhere.one.org')
        self.assertEqual (len (matches), 0)
        matches = sort.destination_word ('You')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 3)
        matches = sort.destination_word ('ou')
        self.assertEqual (len (matches), 0)
        matches = sort.sender_match (r'\s*(?:my_)?((own_)?list)')
        self.assertEqual (len (matches), 1)
        (i, (mt, mc)) = matches [0]
        self.assertEqual (i, 4)
        self.assertEqual (mc.group (1), 'own_list')
        self.assertEqual (mc.group (2), 'own_')
        self.assertEqual (len (mc.groups ()), 2)
        matches = sort.header_match ('x-strange')
        self.assertEqual (len (matches), 2)
        (i, _) = matches [0]
        (j, _) = matches [1]
        self.assertEqual (i, 1)
        self.assertEqual (j, 5)
        headers = sort.get_headers ([i for (i, _) in matches])
        self.assertEqual (len (headers), 2)
        self.assertEqual (headers [0], test_headers [1])
        self.assertEqual (headers [1], test_headers [5])

    def test_glob (self):
        sort = msg.Msg (source = test_data_short)
        matches = sort.header_glob ('from', '*[Mm]yself*')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 0)
        matches = sort.header_glob ('from', 'myself', kind = ':contains')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 0)
        matches = sort.header_glob ('from', 'myself', kind = ':is')
        self.assertEqual (len (matches), 0)
        matches = sort.header_glob ('from', 'Myself', kind = ':contains')
        self.assertEqual (len (matches), 0)
        matches = sort.header_glob ('From', '*Myself*')
        self.assertEqual (len (matches), 0)
        matches = sort.header_glob ('from', '*m?_m*')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 0)
        matches = sort.header_glob ('x-multiline', '*one\n line*')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 2)
        matches = sort.header_glob ('x-multiline', 'one\n line', kind = ':contains')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 2)
        matches = sort.header_glob ('x-multiline', 'one\n line', kind = ':is')
        self.assertEqual (len (matches), 0)
        matches = sort.header_glob ('x-multiline', 'one line', kind = ':contains')
        self.assertEqual (len (matches), 0)
        matches = sort.header_glob ('x-strange')
        self.assertEqual (len (matches), 2)
        (i, _) = matches [0]
        (j, _) = matches [1]
        self.assertEqual (i, 1)
        self.assertEqual (j, 5)
        headers = sort.get_headers ([i for (i, _) in matches])
        self.assertEqual (len (headers), 2)
        self.assertEqual (headers [0], test_headers [1])
        self.assertEqual (headers [1], test_headers [5])

    def test_trolls (self):
        msg_fh = open ('trolls.mail')
        sort = msg.Msg (source = msg_fh)
        msg_fh.close ()
        matches = sort.header_match ('subject', '.*Millionaire Dating')
        self.assertEqual (len (matches), 1)
        matches = sort.header_match ('newsgroups', '(?:.*[ \t,])?(' + re.escape ('comp.lang.functional') + ')(?:[ ,].*)?$', re.I|re.S)
        self.assertEqual (len (matches), 1)
        matches = sort.newsgroup_match ('comp.lang.functional')
        self.assertEqual (len (matches), 1)

    def test_received(self):
        for spam in expected_received_ips.keys():
            msg_fh = open(spam)
            sort = msg.Msg(source=msg_fh)
            msg_fh.close()
            all_received_ips = []
            for ip in sort.received_ip_generate():
                all_received_ips.append(ip)
            self.assertEqual(all_received_ips, expected_received_ips[spam])

    def test_add_delete (self):
        sort = msg.Msg (test_data_short)
        imp_tag, imp_content = 'X-Strange', 'an imposing line\n'
        sort.add_header_at (imp_tag, imp_content, 1)
        matches = sort.header_match ('x-strange')
        self.assertEqual (len (matches), 3)
        indices = [i for (i, _) in matches]
        self.assertEqual (indices, [1, 2, 6])
        headers = sort.get_headers (indices)
        self.assertEqual (len (headers), 3)
        self.assertEqual (headers [0], (imp_tag, imp_content))
        self.assertEqual (headers [1], test_headers [1])
        self.assertEqual (headers [2], test_headers [5])
        inc_tag, inc_content = 'X-Incubus', 'an injected line\n'
        sort.add_header_before (inc_tag, inc_content, 'x-strange', 1)
#        sort.add_header_at (inc_tag, inc_content, 2)
        matches = sort.header_match ('x-incubus')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 2)
        indices = [i for (i, _) in matches]
        headers = sort.get_headers (indices)
        self.assertEqual (len (headers), 1)
        self.assertEqual (headers [0], (inc_tag, inc_content))
        sort.delete_header_tag ('x-strange', 1)
        matches = sort.header_match ('x-strange')
        self.assertEqual (len (matches), 2)
        indices = [i for (i, _) in matches]
        self.assertEqual (indices, [1, 6])
        sort.replace_header_tag ('X-Incubus', 'an injected life\n', 'x-incubus', 0)
        matches = sort.header_match ('x-incubus')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 2)
        t, c = sort.get_header (i)
        self.assert_ (c.find ('injected life') != -1)
        sort.transform_header_tag (xform, 'x-incubus', 0)
        matches = sort.header_match ('x-incubus')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 2)
        t, c = sort.get_header (i)
        self.assert_ (c.find ('injected line') != -1)
        sort.append_header (inc_tag, inc_content)
        sort.delete_header_tag_all ('x-incubus')
        matches = sort.header_match ('x-incubus')
        self.assertEqual (matches, [])
        matches = sort.header_match ('sender')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        sender_tag, sender_content = 'Sender', 'woo@hoo\n'
        sort.append_header_and_rename (sender_tag, sender_content)
        matches = sort.header_match ('sender')
        self.assertEqual (len (matches), 1)
        (j, _) = matches [0]
        self.assertEqual (j, len (sort._headers) - 1)
        (hc, ht) = sort.get_header (j)
        self.assertEqual (ht, sender_content)
        matches = sort.header_match ('x-original-sender')
        self.assertEqual (len (matches), 1)
        (j, _) = matches [0]
        self.assertEqual (i, j)
        (hc, ht) = sort.get_header (i)
        self.assertEqual (ht, test_headers [4][1])


class TestRFC2047 (TestSortmail):
    def test_encode (self):
        self.assertEqual (rfc2047.encode (efb, efb_enc, 'q'), '=?' + efb_enc + '?q?' + efb_q + '?=')
        self.assertEqual (rfc2047.encode (efb, efb_enc, 'b'), '=?' + efb_enc + '?b?' + efb_b + '?=')

    def test_make_string (self):
        hybrid = rfc2047.make_string ([efb_p, (efb, efb_enc, 'q')])
        self.assertEqual (hybrid, efb_p + '=?' + efb_enc + '?q?' + efb_q + '?=')
        hybrid = rfc2047.make_string ([efb_p, (efb, efb_enc, 'b')])
        self.assertEqual (hybrid, efb_p + '=?' + efb_enc + '?b?' + efb_b + '?=')

    def test_decode (self):
        matchobj = rfc2047._re_rfc2047.match ('=?' + efb_enc + '?q?' + efb_q + '?=')
        self.assertEqual (rfc2047.decode (matchobj), (efb_enc, efb))
        self.assertEqual (rfc2047.decode_no_charset (matchobj), efb)
        matchobj = rfc2047._re_rfc2047.match ('=?' + efb_enc + '?b?' + efb_b + '?=')
        self.assertEqual (rfc2047.decode (matchobj), (efb_enc, efb))
        self.assertEqual (rfc2047.decode_no_charset (matchobj), efb)
        matchobj = rfc2047._re_rfc2047.match ('=?' + calculus_enc + '?q?' + calculus_q + '?=')
        self.assertEqual (rfc2047.decode (matchobj), (calculus_enc, calculus))
        self.assertEqual (rfc2047.decode_no_charset (matchobj), calculus)

    def test_decode_string (self):
        hybrid = efb_p + '=?' + efb_enc + '?q?' + efb_q + '?='
        self.assertEqual (rfc2047.decode_string (hybrid), efb_p + efb)
        hybrid = efb_p + '=?' + efb_enc + '?b?' + efb_b + '?='
        self.assertEqual (rfc2047.decode_string (hybrid), efb_p + efb)
        hybrid = '=?' + calculus_enc + '?q?' + calculus_q + '?=' + calculus_p
        self.assertEqual (rfc2047.decode_string (hybrid), calculus + calculus_p)

    def test_cxform (self):
        sort = msg.Msg (test_data_short)
        sort.append_header ('Resent-To', rfc2047.make_string ([efb_p, (efb, efb_enc, 'q')]))
        matches = sort.destination_word ('Mdchen')
        self.assertEqual (len (matches), 0)
        with sort.restoring_cxform([rfc2047.decode_string]):
            matches = sort.destination_word ('Mdchen')
        self.assertEqual (len (matches), 1)
        sort.append_header ('X-Foo-Bar', rfc2047.make_string ([(calculus, calculus_enc, 'q'), calculus_p]))
        with sort.restoring_cxform([rfc2047.decode_string]):
            matches = sort.header_match ('X-Foo-Bar', '.*-calcul\.us')
        self.assertEqual (len (matches), 1)

class TestList (TestSortmail):

    def test_list_exact (self):
        sort = msg.Msg (test_list_data)
        matches = sort.list_match (test_list_address_exact)
        self.assertEqual (len (matches), 1)
        (i, (mt, _)) = matches [0]
        self.assertEqual (i, 1)
        self.assertEqual (mt.group (0), 'To')
        sort.delete_header_at (i)
        matches = sort.list_match (test_list_address_exact)
        self.assertEqual (len (matches), 1)
        (i, (mt, _)) = matches [0]
        self.assertEqual (i, 2)
        self.assertEqual (mt.group (0), 'List-Post')
        sort.delete_header_at (i)
        matches = sort.list_match (test_list_address_exact)
        self.assertEqual (len (matches), 1)
        (i, (mt, _)) = matches [0]
        self.assertEqual (i, 1)
        self.assertEqual (mt.group (0), 'List-Id')
        sort.delete_header_at (i)
        matches = sort.list_match (test_list_address_exact)
        self.assertEqual (matches, [])

    def test_list_case (self):
        sort = msg.Msg (test_list_data)
        matches = sort.list_match (test_list_address_case)
        self.assertEqual (len (matches), 1)
        (i, (mt, _)) = matches [0]
        self.assertEqual (i, 1)
        self.assertEqual (mt.group (0), 'To')
        sort.delete_header_at (i)
        matches = sort.list_match (test_list_address_case)
        self.assertEqual (len (matches), 1)
        (i, (mt, _)) = matches [0]
        self.assertEqual (i, 2)
        self.assertEqual (mt.group (0), 'List-Post')
        sort.delete_header_at (i)
        matches = sort.list_match (test_list_address_case)
        self.assertEqual (len (matches), 1)
        (i, (mt, _)) = matches [0]
        self.assertEqual (i, 1)
        self.assertEqual (mt.group (0), 'List-Id')
        sort.delete_header_at (i)
        matches = sort.list_match (test_list_address_case)
        self.assertEqual (matches, [])

    def test_newsgroup_yes (self):
        sort = msg.Msg (test_list_data)
        for i in range (4):
            matches = sort.newsgroup_match (ngs [i])
            self.assertEqual (len (matches), 1)
            (_, (mt, mc)) = matches [0]
            self.assertEqual (mc.group (1),  ngs [i])

    def test_newsgroup_no (self):
        sort = msg.Msg (test_list_data)
        matches = sort.newsgroup_match ('comp.text.sgml')
        self.assertEqual (matches, [])
        matches = sort.newsgroup_match (ngs [0].replace ('.', '@'))
        self.assertEqual (matches, [])

class TestProcess (TestSortmail):

    def test_filter (self):
        sort = msg.Msg (test_data_short)
        sort.filtermsg (['tail', '-n', '+5'])
        self.assertEqual (len (sort._headers), 5)
        self.assertEqual (sort.body (),
                          body_short)
        self.assertEqual (sort._tf, None)
        matches = sort.header_match ('x-strange')
        self.assertEqual (len (matches), 1)
        (i, _) = matches [0]
        self.assertEqual (i, 2)
        headers = sort.get_headers ([i for (i, _) in matches])
        self.assertEqual (len (headers), 1)
        self.assertEqual (headers [0], test_headers [5])

    def test_mbox_short (self):
        mbox = destination.MBox (self.td + '/test~')
        sort = msg.Msg (test_data_short)
        mbox.accept (sort)
        fh = open (self.td + '/test~', 'r')
        fromline = fh.readline ()
        self.assert_ (fromline.startswith ('From ' + os.getenv ('LOGNAME', 'nobody') + '@localhost '))
        (_, stamp) = fromline.split ('@localhost ')
        brokentime = time.strptime (stamp.rstrip (), '%a %b %d %H:%M:%S %Y')
        stamptime = time.mktime (brokentime)
        nowtime = time.mktime (time.localtime ())
        self.assert_ (nowtime - stamptime < 60.0)
        rest = fh.read ()
        fh.close ()
        self.assertEqual (len (rest), len (test_data_short) + 2)
        sm = difflib.SequenceMatcher (None, rest, test_data_short)
        diffs = sm.get_matching_blocks ()
        self.assertEqual (len (diffs), 3)
        i1, j1, n1 = diffs [0]
        i2, j2, n2 = diffs [1]
        self.assertEqual (j1 + n1, j2)
        self.assertEqual (i1 + n1 + 1, i2)
        self.assertEqual (rest [i1 + n1], '>')
        self.assertEqual (j2 + n2, len (test_data_short))
        self.assertEqual (i2 + n2 + 1, len (rest))
        self.assertEqual (rest [i2 + n2], '\n')

    def test_mbox_long (self):
        mbox = destination.MBox (self.td + '/test~')
        sort = msg.Msg (test_data_long)
        mbox.accept (sort)
        fh = open (self.td + '/test~', 'r')
        fromline = fh.readline ()
        self.assert_ (fromline.startswith ('From ' + os.getenv ('LOGNAME', 'nobody') + '@localhost '))
        (_, stamp) = fromline.split ('@localhost ')
        brokentime = time.strptime (stamp.rstrip (), '%a %b %d %H:%M:%S %Y')
        stamptime = time.mktime (brokentime)
        nowtime = time.mktime (time.localtime ())
        self.assert_ (nowtime - stamptime < 60.0)
        rest = fh.read ()
        fh.close ()
        self.assertEqual (len (rest), len (test_data_long) + 2)
        sm = difflib.SequenceMatcher (None, rest, test_data_long)
        diffs = sm.get_matching_blocks ()
        self.assertEqual (len (diffs), 3)
        i1, j1, n1 = diffs [0]
        i2, j2, n2 = diffs [1]
        self.assertEqual (j1 + n1, j2)
        self.assertEqual (i1 + n1 + 1, i2)
        self.assertEqual (rest [i1 + n1], '>')
        self.assertEqual (j2 + n2, len (test_data_long))
        self.assertEqual (i2 + n2 + 1, len (rest))
        self.assertEqual (rest [i2 + n2], '\n')

    def test_maildir_short (self):
        mdir = destination.MDir (self.maildir)
        sort = msg.Msg (test_data_short)
        mdir.accept (sort)
        tmpfiles = os.listdir (self.tmpdir)
        self.assertEqual (len (tmpfiles), 0)
        newfiles = os.listdir (self.newdir)
        self.assertEqual (len (newfiles), 1)
        newname = self.newdir + '/' + newfiles [0]
        self.assert_ (re.match (r'.*/[0-9]+\.M[0-9]+P[0-9]+Q[0-9]+\.' + re.escape (gethostname ()), newname))
        fh = open (newname, 'r')
        rest = fh.read ()
        fh.close ()
        self.assertEqual (rest, test_data_short)

    def test_maildir_long (self):
        mdir = destination.MDir (self.maildir)
        sort = msg.Msg (test_data_long)
        mdir.accept (sort)
        tmpfiles = os.listdir (self.tmpdir)
        self.assertEqual (len (tmpfiles), 0)
        newfiles = os.listdir (self.newdir)
        self.assertEqual (len (newfiles), 1)
        newname = self.newdir + '/' + newfiles [0]
        self.assert_ (re.match (r'.*/[0-9]+\.M[0-9]+P[0-9]+Q[0-9]+\.' + re.escape (gethostname ()), newname))
        fh = open (newname, 'r')
        rest = fh.read ()
        fh.close ()
        self.assertEqual (rest, test_data_long)

    def test_pipe_short (self):
        mpipe = destination.MPipe (['sh', '-c', 'cat >' + self.td + '/test~'])
        sort = msg.Msg (test_data_short)
        mpipe.accept (sort)
        fh = open (self.td + '/test~', 'r')
        contents = fh.read ()
        fh.close ()
        self.assertEqual (contents, test_data_short)

    def test_pipe_long (self):
        mpipe = destination.MPipe (['sh', '-c', 'cat >' + self.td + '/test~'])
        sort = msg.Msg (test_data_long)
        mpipe.accept (sort)
        fh = open (self.td + '/test~', 'r')
        contents = fh.read ()
        fh.close ()
        self.assertEqual (contents, test_data_long)

    def test_mbox_short_from (self):
        mbox = destination.MBox (self.td + '/test~')
        sort = msg.Msg (test_data_short_from)
        mbox.accept (sort)
        fh = open (self.td + '/test~', 'r')
        fromline = fh.readline ()
        self.assert_ (fromline.startswith ('From me_myself_I@localhost '))
        (_, stamp) = fromline.split ('@localhost ')
        brokentime = time.strptime (stamp.rstrip (), '%a %b %d %H:%M:%S %Y')
        stamptime = time.mktime (brokentime)
        nowtime = time.mktime (time.localtime ())
        self.assert_ (nowtime - stamptime < 60.0)
        rest = fh.read ()
        fh.close ()
        self.assertEqual (len (rest), len (test_data_short) + 2)
        sm = difflib.SequenceMatcher (None, rest, test_data_short)
        diffs = sm.get_matching_blocks ()
        self.assertEqual (len (diffs), 3)
        i1, j1, n1 = diffs [0]
        i2, j2, n2 = diffs [1]
        self.assertEqual (j1 + n1, j2)
        self.assertEqual (i1 + n1 + 1, i2)
        self.assertEqual (rest [i1 + n1], '>')
        self.assertEqual (j2 + n2, len (test_data_short))
        self.assertEqual (i2 + n2 + 1, len (rest))
        self.assertEqual (rest [i2 + n2], '\n')

    def test_mbox_long_from (self):
        mbox = destination.MBox (self.td + '/test~')
        sort = msg.Msg (test_data_long_from)
        mbox.accept (sort)
        fh = open (self.td + '/test~', 'r')
        fromline = fh.readline ()
        self.assert_ (fromline.startswith ('From me_myself_I@localhost '))
        (_, stamp) = fromline.split ('@localhost ')
        brokentime = time.strptime (stamp.rstrip (), '%a %b %d %H:%M:%S %Y')
        stamptime = time.mktime (brokentime)
        nowtime = time.mktime (time.localtime ())
        self.assert_ (nowtime - stamptime < 60.0)
        rest = fh.read ()
        fh.close ()
        self.assertEqual (len (rest), len (test_data_long) + 2)
        sm = difflib.SequenceMatcher (None, rest, test_data_long)
        diffs = sm.get_matching_blocks ()
        self.assertEqual (len (diffs), 3)
        i1, j1, n1 = diffs [0]
        i2, j2, n2 = diffs [1]
        self.assertEqual (j1 + n1, j2)
        self.assertEqual (i1 + n1 + 1, i2)
        self.assertEqual (rest [i1 + n1], '>')
        self.assertEqual (j2 + n2, len (test_data_long))
        self.assertEqual (i2 + n2 + 1, len (rest))
        self.assertEqual (rest [i2 + n2], '\n')

    def test_pipe_short_from (self):
        mpipe = destination.MPipe (['sh', '-c', 'cat >' + self.td + '/test~'])
        sort = msg.Msg (test_data_short_from)
        mpipe.accept (sort)
        fh = open (self.td + '/test~', 'r')
        contents = fh.read ()
        fh.close ()
        self.assertEqual (len (contents), len (test_data_short) + 1)
        sm = difflib.SequenceMatcher (None, contents, test_data_short)
        diffs = sm.get_matching_blocks ()
        self.assertEqual (len (diffs), 3)
        i1, j1, n1 = diffs [0]
        i2, j2, n2 = diffs [1]
        self.assertEqual (j1 + n1, j2)
        self.assertEqual (i1 + n1 + 1, i2)
        self.assertEqual (contents [i1 + n1], '>')
        self.assertEqual (j2 + n2, len (test_data_short))
        self.assertEqual (i2 + n2, len (contents))

    def test_pipe_long_from (self):
        mpipe = destination.MPipe (['sh', '-c', 'cat >' + self.td + '/test~'])
        sort = msg.Msg (test_data_long_from)
        mpipe.accept (sort)
        fh = open (self.td + '/test~', 'r')
        contents = fh.read ()
        fh.close ()
        self.assertEqual (len (contents), len (test_data_long) + 1)
        sm = difflib.SequenceMatcher (None, contents, test_data_long)
        diffs = sm.get_matching_blocks ()
        self.assertEqual (len (diffs), 3)
        i1, j1, n1 = diffs [0]
        i2, j2, n2 = diffs [1]
        self.assertEqual (j1 + n1, j2)
        self.assertEqual (i1 + n1 + 1, i2)
        self.assertEqual (contents [i1 + n1], '>')
        self.assertEqual (j2 + n2, len (test_data_long))
        self.assertEqual (i2 + n2, len (contents))


class TestDupes (TestSortmail):
    def test_single_mbox (self):
        mbox = destination.MBox (self.td + '/test~')
        sort = msg.Msg (source = test_data_short)
        sort.deliver ([mbox], self.dbfile)
        fh = open (self.td + '/test~', 'r')
        fromline = fh.readline ()
        self.assert_ (fromline.startswith ('From ' + os.getenv ('LOGNAME', 'nobody') + '@localhost '))
        (_, stamp) = fromline.split ('@localhost ')
        brokentime = time.strptime (stamp.rstrip (), '%a %b %d %H:%M:%S %Y')
        stamptime = time.mktime (brokentime)
        nowtime = time.mktime (time.localtime ())
        self.assert_ (nowtime - stamptime < 60.0)
        rest = fh.read ()
        fh.close ()
        self.assertEqual (len (rest), len (test_data_short) + 2)
        sm = difflib.SequenceMatcher (None, rest, test_data_short)
        diffs = sm.get_matching_blocks ()
        self.assertEqual (len (diffs), 3)
        i1, j1, n1 = diffs [0]
        i2, j2, n2 = diffs [1]
        self.assertEqual (j1 + n1, j2)
        self.assertEqual (i1 + n1 + 1, i2)
        self.assertEqual (rest [i1 + n1], '>')
        self.assertEqual (j2 + n2, len (test_data_short))
        self.assertEqual (i2 + n2 + 1, len (rest))
        self.assertEqual (rest [i2 + n2], '\n')
        fh = open (self.td + '/foo', 'r')
        loglines = fh.readlines ()
        self.assert_ (loglines [-1].endswith ('success\n'))
        fh.close ()
        
    def test_dupe_mbox (self):
        mbox = destination.MBox (self.td + '/test~')
        sort = msg.Msg (source = test_data_short)
        sort.deliver ([mbox], self.dbfile)
        fh = open (self.td + '/foo', 'r')
        loglines = fh.readlines ()
        self.assert_ (loglines [-1].endswith ('success\n'))
        fh.close ()
        sort2 = msg.Msg (source = test_data_short)
        sort2.deliver ([mbox], self.dbfile)
        fh = open (self.td + '/test~', 'r')
        fromline = fh.readline ()
        self.assert_ (fromline.startswith ('From ' + os.getenv ('LOGNAME', 'nobody') + '@localhost '))
        (_, stamp) = fromline.split ('@localhost ')
        brokentime = time.strptime (stamp.rstrip (), '%a %b %d %H:%M:%S %Y')
        stamptime = time.mktime (brokentime)
        nowtime = time.mktime (time.localtime ())
        self.assert_ (nowtime - stamptime < 60.0)
        rest = fh.read ()
        fh.close ()
        self.assertEqual (len (rest), len (test_data_short) + 2)
        sm = difflib.SequenceMatcher (None, rest, test_data_short)
        diffs = sm.get_matching_blocks ()
        self.assertEqual (len (diffs), 3)
        i1, j1, n1 = diffs [0]
        i2, j2, n2 = diffs [1]
        self.assertEqual (j1 + n1, j2)
        self.assertEqual (i1 + n1 + 1, i2)
        self.assertEqual (rest [i1 + n1], '>')
        self.assertEqual (j2 + n2, len (test_data_short))
        self.assertEqual (i2 + n2 + 1, len (rest))
        self.assertEqual (rest [i2 + n2], '\n')
        fh = open (self.td + '/foo', 'r')
        loglines = fh.readlines ()
        self.assert_ (loglines [-3].endswith ('detected duplicate ID\n'))
        fh.close ()

if __name__ == '__main__':
    lock_suite = unittest.makeSuite (TestLock)
    construct_suite = unittest.makeSuite (TestConstruct)
    header_suite = unittest.makeSuite (TestHeaders)
    rfc2047_suite = unittest.makeSuite (TestRFC2047)
    list_suite = unittest.makeSuite (TestList)
    process_suite = unittest.makeSuite (TestProcess)
    dupe_suite = unittest.makeSuite (TestDupes)
    all_suite = unittest.TestSuite ((lock_suite, construct_suite, header_suite, rfc2047_suite, list_suite, process_suite, dupe_suite))
    unittest.TextTestRunner (verbosity = 3).run (all_suite)
