# Copyright (C) 2007, 2010 Ian Zimmerman <itz@buug.org>

# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the conditions spelled out in
# the file LICENSE are met.

from __future__ import with_statement , absolute_import

import logging
import subprocess
from subprocess import PIPE
import os
from os import O_WRONLY, O_EXCL, O_CREAT
from errno import EEXIST
import signal
from .lock import unlocking
import time
import math
from socket import gethostname
import random
from contextlib import closing

sendmails = [p for p in ['/usr/sbin/sendmail', '/usr/lib/sendmail', '/sbin/sendmail', '/lib/sendmail']
             if os.access (p, os.X_OK)]
sendmail = len (sendmails) > 0 and sendmails [0] or None

class Error (Exception):
    """Base class for exceptions specific to the destination module."""

class ProcessError (Error):
    """Exception raised when an external process exits with non-zero status."""
    def __init__ (self, argv, status):
        """The exception constructor; argv is the list of command name arguments
        of the failed process, status is the exit status.
        """
        self.argv = argv
        self.status = status


class MBox(object):
    """Class representing traditional Unix mbox style mailbox file."""
    def __init__ (self, fn):
        """The constructor; fn is the mailbox file name."""
        self._fn = fn

    def accept (self, msg):
        """Delivers message msg to this mailbox."""
        logger = logging.getLogger ('sortmail')
        logger.info ('(deliver) MBOX: %s', self._fn)
        with unlocking(self._fn):
            msg._append_to_box(self._fn)


class MDir(object):
    """Class representing a maildir (qmail style mailbox store)."""
    # the total number of attempted maildir deliveries in this process
    _deliveries = 0

    def _uniq_name (self):
        time_frac, time_int = math.modf (time.time ())
        time_sec = int (time_int)
        time_usec = int (math.floor (time_frac * 1000000.0))
        uname = (str (time_sec)
                 + '.M' + str (time_usec) + 'P' + str (os.getpid ()) + 'Q' + str (MDir._deliveries)
                 + '.' + gethostname ())
        MDir._deliveries += 1
        return uname

    def __init__ (self, fn):
        """The constructor; fn is the directory name for the maildir.
        This directory must already have the conventional tmp and new subdirectories."""
        self._fn = fn
        
    def accept (self, msg):
        """Delivers msg to this maildir."""
        logger = logging.getLogger ('sortmail')
        logger.info ('(deliver) MDIR: %s', self._fn)
        sleep_time = 2.0
        fd = None
        while fd is None:
            uname = self._uniq_name ()
            tname = self._fn + '/tmp/' + uname
            try:
                fd = os.open(tname, O_WRONLY|O_EXCL|O_CREAT, 0600)
            except OSError as e:
                if e.errno != EEXIST:
                    raise
                time.sleep (sleep_time)
                sleep_time = sleep_time * (1.5 + random.random ())
        with os.fdopen (fd, 'w') as tfh:
            msg._print_to_fh (tfh)
        os.link (tname, self._fn + '/new/' + uname)
        os.unlink (tname)

class MPipe(object):
    """Class representing a command subprocess to pipe messages into."""
    def __init__ (self, argv, env=None):
        """The constructor; argv is the list of command line arguments for the process."""
        self._argv = argv
        self._env = env

    def accept (self, msg):
        """Delivers message msg to this pipe."""
        logger = logging.getLogger ('sortmail')
        logger.info ('(deliver) PIPE: %s', ' '.join (self._argv))
        with open('/dev/null', 'w') as devnull:
            pipe = subprocess.Popen (self._argv, stdin = PIPE, stdout = devnull, env=self._env)
            with closing(pipe.stdin):
                if not msg._test_only:
                    msg._print_to_fh (pipe.stdin)
        status = pipe.wait ()
        if status > 0 or (status < 0 and -status != signal.SIGPIPE):
            logger.error ('delivery subprocess exited with status %d', status)
            raise ProcessError (self._argv, status)

class MForward(object):
    """Class representing an email address to forward messages to."""
    def __init__ (self, addy):
        """The constructor; addy is the forwarding address."""
        self._addy = addy

    def accept (self, msg):
        """Forwards message msg to this address."""
        logger = logging.getLogger ('sortmail')
        logger.info ('(deliver) FORWARD: %s', self._addy)
        argv = (sendmail, '-i', self._addy)
        pipe = subprocess.Popen (argv, stdin = PIPE)
        with closing(pipe.stdin):
            if not msg._test_only:
                msg._print_to_fh (pipe.stdin)
        status = pipe.wait ()
        if status > 0 or (status < 0 and -status != signal.SIGPIPE):
            logger.error ('sendmail exited with status %d', status)
            raise ProcessError (argv, status)
