# Author: Jonathan Slenders, City Live

from django.conf import settings
from django.db import models
from django.utils.translation import ugettext_lazy as _
from hashlib import md5

from sparky.parser import POEntry, POEntryReference

class Application(models.Model):
    name = models.CharField(verbose_name=_('Application name'), max_length=32)
    description = models.CharField(verbose_name=_('Application description'), max_length=1024)

    class Meta:
        verbose_name = _('Application')
        verbose_name_plural = _('Applications')

    def __unicode__(self):
        return self.name

class Reference(models.Model):
    filename = models.TextField(verbose_name=_('Filename'))
    line = models.IntegerField(verbose_name=_('Line number'))
    content = models.TextField(verbose_name=_('Content'))

    class Meta:
        verbose_name = _('Reference')
        verbose_name_plural = _('References')

    def __unicode__(self):
        return '%s:%s' % (self.filename, self.line)

    @property
    def first_line(self):
        return max(0, self.line - 4)

    @property
    def is_html(self):
        return self.filename.endswith('.html')

    @property
    def is_python(self):
        return self.filename.endswith('.py')


class Entry(models.Model):
    msgid = models.TextField(verbose_name=_('Original text'))  # Should actually be unique. But unique cannot have length>255
    checksum = models.CharField(verbose_name=_('md5 of msgid'), max_length=32, unique=True)

    # NOTE: because we can't get msgid case sensitive in some MySQL
    # configurations and get it to be a primary key.  The md5 checksum of that
    # field will do this instead.

    # TODO: make this generic -- Ingo
    en_be = models.TextField(verbose_name=_('Belgium English'), blank=True)
    en_nl = models.TextField(verbose_name=_('Netherlands English'), blank=True)
    nl_be = models.TextField(verbose_name=_('Belgium Dutch'), blank=True)
    nl_nl = models.TextField(verbose_name=_('Netherlands Dutch'), blank=True)
    fr_be = models.TextField(verbose_name=_('Belgium French'), blank=True)

    fuzzy_en_be = models.BooleanField(verbose_name=_('Fuzzy Belgium English'))
    fuzzy_en_nl = models.BooleanField(verbose_name=_('Fuzzy Netherlands English'))
    fuzzy_nl_be = models.BooleanField(verbose_name=_('Fuzzy Belgium Dutch'))
    fuzzy_nl_nl = models.BooleanField(verbose_name=_('Fuzzy Netherlands Dutch'))
    fuzzy_fr_be = models.BooleanField(verbose_name=_('Fuzzy Belgium French'))

    comments = models.TextField(blank=True)  # Also used for dumping collision information, etc... things to debug and notes.
    references = models.ManyToManyField(Reference)  # List of references
    applications = models.ManyToManyField(Application)  # List of applications where this string is used.

    created = models.DateTimeField(auto_now_add=True)
    modified = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = _('Entry')
        verbose_name_plural = _('Entries')

    def __unicode__(self):
        if len(self.msgid) > 50:
            return '%s: %s...' % (_('Entry'), self.msgid[:50])
        else:
            return '%s: %s' % (_('Entry'), self.msgid)

    def get_parser_entry(self, lang):
        if lang in settings.AVAILABLE_LANGUAGES:
            entry = POEntry(self.msgid, self.get_translation(lang))
            entry.fuzzy = self.is_fuzzy(lang)
        entry.comments = self.comments
        for r in self.references.all():
            entry.references.append(POEntryReference(r.filename, r.line))
        return entry

    def get_translation(self, lang):
        return getattr(self, lang)

    def set_translation(self, lang, translation):
        setattr(self, lang, translation)

    def is_fuzzy(self, lang):
        return getattr(self, 'fuzzy_%s' % lang)

    def set_fuzzy(self, lang, fuzzy):
        setattr(self, 'fuzzy_%s' % lang, fuzzy)

    def is_ignored(self, project):
        from sparky.project.models import ProjectIgnore
        return ProjectIgnore.objects.filter(project=project, entry=self).exists()

    def set_ignored(self, project, ignore):
        from sparky.project.models import ProjectIgnore
        try:
            if ignore:
                ProjectIgnore.objects.create(project=project, entry=self)
            else:
                ProjectIgnore.objects.filter(project=project, entry=self).delete()
        except Exception, e:
            print e


class Conflict(models.Model):
    language = models.CharField(verbose_name=_('Language'), max_length=5)
    msgid = models.TextField(verbose_name=_('Original string'))
    existing_string = models.TextField(verbose_name=_('Existing string'))
    new_string = models.TextField(verbose_name=_('New string'))
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = _('Conflict')
        verbose_name_plural = _('Conflicts')

    def __unicode__(self):
        if len(self.msgid) > 50:
            return '%s (%s): %s...' % (_('Conflict'), self.language, self.msgid[:50])
        else:
            return '%s (%s): %s' % (_('Conflict'), self.language, self.msgid)

    @property
    def references(self):
        """ Look for an 'Entry' with the same msgid, and returns his references """
        refs = []

        for entry in Entry.objects.filter(checksum=checksum(self.msgid)):
            refs += list(entry.references.all())
        return refs


def checksum(msgid):
    assert isinstance(msgid, unicode)
    return md5(msgid.encode('utf-8')).hexdigest()
