# Author: Jonathan Slenders, City Live

# NOTE: code is not yet really DRY

import datetime

from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.contrib.auth import login as auth_login, logout as auth_logout, authenticate
from django.core.urlresolvers import reverse
from django.db.models import Q
from django.http import HttpResponse, HttpResponseRedirect
from django.views.decorators.http import require_POST

from templatable_view import templatable_view
from django_ajax.pagination import paginate

from sparky.models import Entry, Conflict, Application, checksum, Reference
from sparky.forms import SearchForm
from sparky.parser import Parser, POEntry, ParseError
from sparky.project.models import Project


# ================[ Authentication views ]==================

@templatable_view('login.html')
def login(request):
    authenticated = False
    if request.method == 'POST':
        user = authenticate(username=request.POST['username'], password=request.POST['password'])
        if user is not None and user.is_active:
            auth_login(request, user)
            if 'next' in request.REQUEST:
                return HttpResponseRedirect(request.REQUEST['next'])
            else:
                return HttpResponseRedirect('/')
    return {
            'error': not request.user.is_authenticated() and request.method == 'POST',
        }


def logout(request):
    auth_logout(request)
    return HttpResponseRedirect('/')


# ================[ API views ]==================

def _save_conflict_if_not_exist(conflict):
    if Conflict.objects.filter(
                    language=conflict.language,
                    msgid=conflict.msgid,
                    existing_string=conflict.existing_string,
                    new_string=conflict.new_string).count() == 0:
        conflict.save()


@require_POST
def upload_entry(request):
    """
    Used by the client script, to upload a single entry into the database
    """
    # ====== Parameters =====
    # Project parameter
    # project_slug = request.POST['project_slug']
    # try:
    #     project = Project.objects.get(slug=project_slug)
    # except Project.DoesNotExist:
    #     project = None
    # assert project is not None

    # Language parameter
    language = request.POST['language']
    assert language in settings.AVAILABLE_LANGUAGES

    # Replace comments/references parameter
    replace_comments = bool(request.POST.get('replace_comments', ''))
    replace_references = bool(request.POST.get('replace_references', ''))

    # Application parameter
    application = request.POST.get('application', None)
    if application:
        application, created = Application.objects.get_or_create(name=application)

    # Comments parameter
    comments = request.POST.get('comments', '')

    # msgid and msgstr parameter
    msgid = request.POST.get('msgid')
    msgstr = request.POST.get('msgstr')
    fuzzy = bool(request.POST.get('fuzzy', ''))

    # ====== Create entry =====
    s = checksum(msgid)
    # e, created = Entry.objects.get_or_create(checksum=s, project=project)
    e, created = Entry.objects.get_or_create(checksum=s)
    e.msgid = msgid

    #print '=='
    #print ' *%s*' % e.msgid

    # When translation was given, save into database, but never
    # overwrite existing values, (unless the existing value is marked
    # fuzzy)
    if not e.get_translation(language) or e.is_fuzzy(language):
        e.set_translation(language, msgstr)
        e.set_fuzzy(language, fuzzy)

    error = ''
    if msgstr and e.get_translation(language) != msgstr and not fuzzy:
        _save_conflict_if_not_exist(Conflict(language=language, msgid=e.msgid, existing_string=e.get_translation(language), new_string=msgstr))
        error = 'ERROR: conflicting strings\n  en="%s"\n  %s="%s"\n  existing="%s"\n' % (msgid, language, msgstr, e.get_translation(language))

    # Application
    if application:
        e.applications.add(application)

    # References
    if replace_references:
        for r in e.references.all():
            r.delete()
    for k, v in request.POST.items():
        if k.startswith('reference_'):
            filename = request.POST['o_%s_filename' % k]
            line = int(request.POST['o_%s_line' % k])
            r = Reference.objects.create(content=v, filename=filename, line=line)
            e.references.add(r)

    # Save comments
    if replace_comments:
        e.comments = comments

    # Save entry
    e.save()

    # Answer
    if error:
        return HttpResponse(error, status=500)
    else:
        return HttpResponse('Entry saved (language=%s, application=%s)\n' % (language, application))


### @login_required
@templatable_view('upload.html')
def upload_po(request):
    if request.method == 'POST':
        # Language parameter
        language = request.POST['language']
        assert language in settings.AVAILABLE_LANGUAGES

        # Replace comments parameter
        replace_comments = bool(request.POST.get('replace_comments', ''))

        # Application parameter
        application = request.POST.get('application', None)
        if application:
            application, created = Application.objects.get_or_create(name=application)

        try:
            # Parse PO file
            parser = Parser(request.FILES['file'].read())

            # Errors
            errors = []

            # Save entries
            for entry in parser.entries:
                s = checksum(entry.msgid)
                e, created = Entry.objects.get_or_create(checksum=s)
                e.msgid = entry.msgid

                # When translation was given, save into database, but never
                # overwrite existing values, (unless the existing value is marked
                # fuzzy)
                if not e.get_translation(language) or e.is_fuzzy(language):
                    e.set_translation(language, entry.msgstr)
                    e.set_fuzzy(language, entry.fuzzy)

                if entry.msgstr and e.get_translation(language) != entry.msgstr and not entry.fuzzy:
                    _save_conflict_if_not_exist(Conflict(language=language, msgid=e.msgid, existing_string=e.get_translation(language), new_string=entry.msgstr))
                    errors.append('ERROR: conflicting strings\n  en="%s"\n  %s="%s"\n  existing="%s"\n' % (entry.msgid, language, entry.msgstr, e.get_translation(language)))

                # Application
                if application:
                    e.applications.add(application)

                # Save comments
                if replace_comments:
                    e.comments = entry.comments

                # Save entry
                e.save()

            # Add entries to database
            return HttpResponse("""%s entries in PO-file processed and saved into database:
                Replace comments: %s
                Language: %s
                Application: %s
                %s errors: %s
        """ % (len(parser.entries), replace_comments, language, application, len(errors), ''.join(errors)),
                        content_type='text/plain'
                )
        except ParseError, e:
            return HttpResponse("PARSE ERROR: %s\n" % str(e))
    else:
        return { 'languages': settings.AVAILABLE_LANGUAGES, 'language_names': settings.LANGUAGE_NAMES }


@require_POST
def complete_po(request):
    # Language parameter
    language = request.POST['language']
    assert language in settings.AVAILABLE_LANGUAGES

    # Parse PO file
    parser = Parser(request.FILES['file'].read())

    # Loop through entries
    for entry in parser.entries:
        try:
            s = checksum(entry.msgid)
            e = Entry.objects.get(checksum=s)

            # Replace msgstr when the original entry is fuzzy or not defined.
            if 'update_all' in request.GET or entry.fuzzy or not entry.msgstr:
                if e.get_translation(language):
                    entry.msgstr = e.get_translation(language)
                    entry.fuzzy = e.is_fuzzy(language)
        except Entry.DoesNotExist:
            pass

    return HttpResponse(parser.to_string(), content_type='text/plain; charset=utf-8')


def download_po(request):
    parser = Parser()

    # Language
    lang = request.GET['lang']

    entries = Entry.objects.all()

    header = 'Language: %s\n' % settings.CLIENT_LANGUAGES[lang.replace('_', '-')]

    # Filter applications from GET parameters
    if 'application' in request.GET:
        print [int(a) for a in request.GET.getlist('application')]
        app_ids = [int(a) for a in request.GET.getlist('application')]
        entries = entries.filter(applications__id__in=app_ids)

        # Header
        header += 'Applications: %s' % ', '.join([a.name for a in Application.objects.filter(id__in=app_ids) ])

    entries = entries.distinct()
    for entry in entries:
        parser.entries.append(entry.get_parser_entry(lang))

    return HttpResponse(parser.to_string(header=header), content_type='text/plain; charset=utf-8')


def download_backup(request):
    """
    Creates a zipfile.
    """
    from zipfile import ZipFile, ZIP_DEFLATED

    parsers = {}
    for lang in settings.AVAILABLE_LANGUAGES:
        parsers[lang] = Parser()

    for entry in Entry.objects.all():
        for lang, parser in parsers.items():
            parser.entries.append(entry.get_parser_entry(lang))

    # Create zipfile
    response = HttpResponse(content_type='application/zip')
    response['Content-Disposition'] = 'attachment; filename=i18n-database-%s.zip' % datetime.datetime.today().date().isoformat()
    zipfile = ZipFile(response, 'w', ZIP_DEFLATED)
    for lang, parser in parsers.items():
        lang = settings.CLIENT_LANGUAGES[lang.replace('_', '-')]
        zipfile.writestr('%s/django.po' % lang, parser.to_string(header='Backup / Language: %s\n' % lang))
    zipfile.close()

    return response


# ================[ Web UI ]==================

@templatable_view('download-overview.html')
@login_required
def download_overview(request):
    # Stats
    stats = {}
    for p in Project.objects.all():
        project_stats = []

        for a in p.applications:
            entries = Entry.objects.filter(applications=a)
            application_stats = {
                'application': a.name,
                'application_id': a.id,
                'total': entries.count()
            }
            for l in p.languages:
                lang = l.language.lower()
                application_stats[lang] = entries.filter(**{lang: ''}).count()
                application_stats['%s_fuzzy' % lang] = entries.filter(**{'fuzzy_%s' % lang: True}).count()

            project_stats.append(application_stats)

        stats[p] = project_stats

    return {
        'stats': stats,
        # 'languages': settings.AVAILABLE_LANGUAGES,
        'language_names': settings.LANGUAGE_NAMES,
    }


@login_required
@templatable_view('conflicts.html')
def conflicts(request):
    conflicts = Conflict.objects.all().order_by('-created')

    # Ordering
    order = request.GET.get('order', '')
    if order == 'existing_string':
        conflicts = conflicts.order_by('existing_string')
    elif order == 'new_string':
        conflicts = conflicts.order_by('new_string')
    elif order == 'msgid':
        conflicts = conflicts.order_by('msgid')
    elif order == 'lang':
        conflicts = conflicts.order_by('language')

    paginated_conflicts = paginate(request, conflicts,
                num_per_page=20, use_get_parameters=True)

    return {
            'conflicts': paginated_conflicts,
            'conflict_count': conflicts.count(),
            'order': order
            }


@login_required
def solve_conflict(request, conflict_id):
    """
    Solve conflict, take either the new or the old string.
    """
    conflict = Conflict.objects.get(id=int(conflict_id))
    replace = request.GET.get('replace', '')

    if replace:
        s = checksum(conflict.msgid)
        entry = Entry.objects.get(checksum=s)

        entry.set_translation(conflict.language, conflict.new_string)
        entry.save()

    conflict.delete()

    return HttpResponseRedirect(reverse('conflicts'))


@login_required
@require_POST
def ajax_save_translations(request):
    """
    Save translations in DB
    """
    # post params look like: 'fr_be_283=translation'
    for k, translation in request.POST.iteritems():
        language = k[0:5]
        id = int(k[6:])

        entry = Entry.objects.get(id=id)
        if language in settings.AVAILABLE_LANGUAGES:
            entry.set_translation(language, translation)
        else:
            raise 'Unknown language'
        entry.save()

    return HttpResponse('OK')


@login_required
@require_POST
def ajax_save_fuzzy(request):
    """
    Save 'fuzzy' parameter in DB
    """
    # post parameters look like fuzzy_fr_be_283=fuzzy or fuzzy_fr_be_283=not_fuzzy
    for k, value in request.POST.iteritems():
        language = k[6:11]
        id = int(k[12:])

        assert value in ('fuzzy', 'not_fuzzy')
        fuzzy = (value == 'fuzzy')

        entry = Entry.objects.get(id=id)
        if language in settings.AVAILABLE_LANGUAGES:
            entry.set_fuzzy(language, fuzzy)
        else:
            raise 'Unknown language'
        entry.save()

    return HttpResponse('OK')


@login_required
@require_POST
def ajax_save_ignore(request):
    """
    Save 'ignore' parameter in DB
    """
    # post parameters look like ignore_12_283=ignore or ignore_12_283=not_ignore
    for k, value in request.POST.iteritems():
        rest, project_id, entry_id = k.split('_')

        assert value in ('ignore', 'not_ignore')
        ignore = (value == 'ignore')

        entry = Entry.objects.get(id=entry_id)
        entry.set_ignored(project=Project.objects.get(id=project_id), ignore=ignore)

    return HttpResponse('OK')


@login_required
@require_POST
def ajax_save_translation_comment(request):
    # Save translation comment in DB
    pass
