#ifndef TESTFRAMEWORK_H
#define TESTFRAMEWORK_H

/*
Planarity-Related Graph Algorithms Project
Copyright (c) 1997-2010, John M. Boyer
All rights reserved. Includes a reference implementation of the following:

* John M. Boyer. "Simplified O(n) Algorithms for Planar Graph Embedding,
  Kuratowski Subgraph Isolation, and Related Problems". Ph.D. Dissertation,
  University of Victoria, 2001.

* John M. Boyer and Wendy J. Myrvold. "On the Cutting Edge: Simplified O(n)
  Planarity by Edge Addition". Journal of Graph Algorithms and Applications,
  Vol. 8, No. 3, pp. 241-273, 2004.

* John M. Boyer. "A New Method for Efficiently Generating Planar Graph
  Visibility Representations". In P. Eades and P. Healy, editors,
  Proceedings of the 13th International Conference on Graph Drawing 2005,
  Lecture Notes Comput. Sci., Volume 3843, pp. 508-511, Springer-Verlag, 2006.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

* Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright notice, this
  list of conditions and the following disclaimer in the documentation and/or
  other materials provided with the distribution.

* Neither the name of the Planarity-Related Graph Algorithms Project nor the names
  of its contributors may be used to endorse or promote products derived from this
  software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "../graph.h"

#ifdef __cplusplus
extern "C" {
#endif

#define NUMCOMMANDSTOTEST	7
extern char *commands;

// numGraphs: the number of graphs that met the test criteria
//            (e.g. were generated and had a specific number of edges)
// numOKs: the number of graphs on which the test produced OK
//         (as opposed to NONEMBEDDABLE (or NOTOK))
typedef struct
{
	unsigned long numGraphs;
	unsigned long numOKs;
} baseTestResultStruct;

// command: Indicates the algorithm being tested
// result: Cumulative result of tests on graphs of all sizes
//         (i.e. numbers of edges; the number of vertices is fixed
//          within a given test)
// edgeResults*: Array to accumulateresult of tests on graphs of a
//               each fixed size (i.e. fixed number of edges)
// theGraph: a graph data structure on which each test is performed
//           The graph is preconfigured with a number of vertices
//           and a specific algorithm extension.
// origGraph: a copy of the graph being tested before the algorithm
//            is run on it. This is needed for integrity checking of
//            an algorithm's result produced in theGraph
//            The origGraph from algResult[0] in th test framework
//            receives the edges from the adjacency matrix graph
//            generated by Nauty.  Then, gp_CopyAdjacencyLists() is
//            used to copy the edges of this graph into the origGraph
//            of each other algorithm testResult (if any).
typedef struct
{
	char command;
	baseTestResultStruct result;
	baseTestResultStruct *edgeResults;
	int edgeResultsSize;
	graphP theGraph, origGraph;
} testResultStruct;

typedef testResultStruct * testResultP;

// algResults: An array of testResults for each algorithm being tested
// algResultsSize: the number of algorithms being tested
typedef struct
{
	testResultStruct *algResults;
	int algResultsSize;
} testResultFrameworkStruct;

typedef testResultFrameworkStruct * testResultFrameworkP;

// Returns a pointer to the testResult for the given command character
// or NULL if there is no test result for the algorithm
testResultP tf_GetTestResult(testResultFrameworkP framework, char command);

// Allocate the test framework for the given command. The number of vertices
// and maximum number of edges in the graphs to be tested are also given.
testResultFrameworkP tf_AllocateTestFramework(char command, int n, int maxe);

// Free the test framework.
void tf_FreeTestFramework(testResultFrameworkP *pTestFramework);

#ifdef __cplusplus
}
#endif

#endif
