"""
    sphinxcontrib.httpdomain
    ~~~~~~~~~~~~~~~~~~~~~~~~

    The HTTP domain for documenting RESTful HTTP APIs.

    :copyright: Copyright 2011 by Hong Minhee
    :license: BSD, see LICENSE for details.

"""

import re

from docutils import nodes
from docutils.parsers.rst.roles import set_classes

from sphinx import addnodes
from sphinx.roles import XRefRole
from sphinx.domains import Domain, ObjType, Index
from sphinx.directives import ObjectDescription
from sphinx.util.nodes import make_refnode
from sphinx.util.docfields import Field, GroupedField, TypedField


HTTP_STATUS_CODES = {
    100: 'Continue',
    101: 'Switching Protocols',
    102: 'Processing',
    200: 'OK',
    201: 'Created',
    202: 'Accepted',
    203: 'Non Authoritative Information',
    204: 'No Content',
    205: 'Reset Content',
    206: 'Partial Content',
    207: 'Multi Status',
    226: 'IM Used',              # see RFC 3229
    300: 'Multiple Choices',
    301: 'Moved Permanently',
    302: 'Found',
    303: 'See Other',
    304: 'Not Modified',
    305: 'Use Proxy',
    307: 'Temporary Redirect',
    400: 'Bad Request',
    401: 'Unauthorized',
    402: 'Payment Required',     # unused
    403: 'Forbidden',
    404: 'Not Found',
    405: 'Method Not Allowed',
    406: 'Not Acceptable',
    407: 'Proxy Authentication Required',
    408: 'Request Timeout',
    409: 'Conflict',
    410: 'Gone',
    411: 'Length Required',
    412: 'Precondition Failed',
    413: 'Request Entity Too Large',
    414: 'Request URI Too Long',
    415: 'Unsupported Media Type',
    416: 'Requested Range Not Satisfiable',
    417: 'Expectation Failed',
    418: "I'm a teapot",        # see RFC 2324
    422: 'Unprocessable Entity',
    423: 'Locked',
    424: 'Failed Dependency',
    426: 'Upgrade Required',
    449: 'Retry With',           # proprietary MS extension
    500: 'Internal Server Error',
    501: 'Not Implemented',
    502: 'Bad Gateway',
    503: 'Service Unavailable',
    504: 'Gateway Timeout',
    505: 'HTTP Version Not Supported',
    507: 'Insufficient Storage',
    510: 'Not Extended'
}

http_sig_param_re = re.compile(r'\((?:(?P<type>[^:)]+):)?(?P<name>[\w_]+)\)',
                               re.VERBOSE)


def http_resource_anchor(method, path):
    path = re.sub(r'[<>:/]', '-', path)
    return method.lower() + '-' + path


class HTTPResource(ObjectDescription):

    doc_field_types = [
        TypedField('parameter', label='Parameters',
                   names=('param', 'parameter', 'arg', 'argument'),
                   typerolename='obj', typenames=('paramtype', 'type')),
        GroupedField('queryparameter', label='Query Parameters',
                     names=('queryparameter', 'queryparam', 'qparam', 'query')),
        GroupedField('formparameter', label='Form Parameters',
                     names=('formparameter', 'formparam', 'fparam', 'form')),
        GroupedField('statuscode', label='Status Codes',
                     rolename='statuscode',
                     names=('statuscode', 'status', 'code'))
    ]

    method = NotImplemented

    def handle_signature(self, sig, signode):
        method = self.method.upper() + ' '
        signode += addnodes.desc_name(method, method)
        offset = 0
        for match in http_sig_param_re.finditer(sig):
            path = sig[offset:match.start()]
            signode += addnodes.desc_name(path, path)
            params = addnodes.desc_parameterlist()
            typ = match.group('type')
            if typ:
                typ = typ + ': '
                params += addnodes.desc_annotation(typ, typ)
            name = match.group('name')
            params += addnodes.desc_parameter(name, name)
            signode += params
            offset = match.end()
        if offset < len(sig):
            path = sig[offset:len(sig)]
            signode += addnodes.desc_name(path, path)
        fullname = self.method.upper() + ' ' + path
        signode['method'] = self.method
        signode['path'] = sig
        signode['fullname'] = fullname
        return (fullname, self.method, sig)

    def needs_arglist(self):
        return False

    def add_target_and_index(self, name_cls, sig, signode):
        signode['ids'].append(http_resource_anchor(*name_cls[1:]))
        self.env.domaindata['http'][self.method][sig] = (self.env.docname, '')

    def get_index_text(self, modname, name):
        return ''


class HTTPOptions(HTTPResource):

    method = 'options'


class HTTPHead(HTTPResource):

    method = 'head'


class HTTPPost(HTTPResource):

    method = 'post'


class HTTPGet(HTTPResource):

    method = 'get'


class HTTPPut(HTTPResource):

    method = 'put'


class HTTPDelete(HTTPResource):

    method = 'delete'


class HTTPTrace(HTTPResource):

    method = 'trace'


def http_statuscode_role(name, rawtext, text, lineno, inliner,
                         options={}, content=[]):
    if text.isdigit():
        code = int(text)
        try:
            status = HTTP_STATUS_CODES[code]
        except KeyError:
            msg = inliner.reporter.error('%d is invalid HTTP status code'
                                         % code, lineno=lineno)
            prb = inliner.problematic(rawtext, rawtext, msg)
            return [prb], [msg]
    else:
        try:
            code, status = re.split(r'\s', text.strip(), 1)
            code = int(code)
        except Value:
            msg = inliner.reporter.error(
                'HTTP status code must be an integer (e.g. `200`) or '
                'start with an integer (e.g. `200 OK`); %r is invalid' %
                text,
                line=lineno
            )
            prb = inliner.problematic(rawtext, rawtext, msg)
            return [prb], [msg]
    nodes.reference(rawtext)
    if code == 226:
        url = 'http://www.ietf.org/rfc/rfc3229.txt'
    if code == 418:
        url = 'http://www.ietf.org/rfc/rfc2324.txt'
    if code == 449:
        url = 'http://msdn.microsoft.com/en-us/library' \
              '/dd891478(v=prot.10).aspx'
    elif code in HTTP_STATUS_CODES:
        url = 'http://www.w3.org/Protocols/rfc2616/rfc2616-sec10.html' \
              '#sec10.' + ('%d.%d' % (code // 100, 1 + code % 100))
    else:
        url = ''
    set_classes(options)
    node = nodes.reference(rawtext, '%d %s' % (code, status),
                           refuri=url, **options)
    return [node], []


def http_method_role(name, rawtext, text, lineno, inliner,
                     options={}, content=[]):
    references = {
        'options': '9.2',
        'get': '9.3',
        'head': '9.4',
        'post': '9.5',
        'put': '9.6',
        'delete': '9.7',
        'trace': '9.8',
        'connect': '9.9'
    }
    method = str(text).lower()
    umethod = method.upper()
    try:
        sec = references[method]
    except KeyError:
        msg = inliner.reporter.error('%s is not valid HTTP method' % umethod,
                                     lineno=lineno)
        prb = inliner.problematic(rawtext, rawtext, msg)
        return [prb], [msg]
    url = 'http://www.w3.org/Protocols/rfc2616/rfc2616-sec9.html#sec' + sec
    node = nodes.reference(rawtext, umethod, refuri=url, **options)
    return [node], []


class HTTPXRefRole(XRefRole):

    def __init__(self, method, **kwargs):
        XRefRole.__init__(self, **kwargs)
        self.method = method

    def process_link(self, env, refnode, has_explicit_title, title, target):
        if not target.startswith('/'):
            pass
        if not has_explicit_title:
            title = self.method.upper() + ' ' + title
        return title, target


class HTTPIndex(Index):

    name = 'routingtable'
    localname = 'HTTP Routing Table'
    shortname = 'routing table'

    def generate(self, docnames=None):
        content = {}
        for method, routes in self.domain.routes.iteritems():
            for path, info in routes.iteritems():
                letter = path.split('/', 2)
                try:
                    first_letter = letter[1]
                except IndexError:
                    first_letter = letter[0]
                entries = content.setdefault('/' + first_letter, [])
                entries.append([
                    method.upper() + ' ' + path, 0, info[0],
                    http_resource_anchor(method, path), '', '', info[1]
                ])
        content = content.items()
        content.sort(key=lambda (k, v): k)
        return (content, True)


class HTTPDomain(Domain):
    """HTTP domain."""

    name = 'http'
    label = 'HTTP'

    object_types = {
        'options': ObjType('options', 'options', 'obj'),
        'head': ObjType('head', 'head', 'obj'),
        'post': ObjType('post', 'post', 'obj'),
        'get': ObjType('get', 'get', 'obj'),
        'put': ObjType('put', 'put', 'obj'),
        'delete': ObjType('delete', 'delete', 'obj'),
        'trace': ObjType('trace', 'trace', 'obj')
    }

    directives = {
        'options': HTTPOptions,
        'head': HTTPHead,
        'post': HTTPPost,
        'get': HTTPGet,
        'put': HTTPPut,
        'delete': HTTPDelete,
        'trace': HTTPTrace
    }

    roles = {
        'options': HTTPXRefRole('options'),
        'head': HTTPXRefRole('head'),
        'post': HTTPXRefRole('post'),
        'get': HTTPXRefRole('get'),
        'put': HTTPXRefRole('put'),
        'delete': HTTPXRefRole('delete'),
        'trace': HTTPXRefRole('trace'),
        'statuscode': http_statuscode_role,
        'method': http_method_role
    }

    initial_data = {
        'options': {}, # path: (docname, synopsis)
        'head': {},
        'post': {},
        'get': {},
        'put': {},
        'delete': {},
        'trace': {}
    }

    indices = [HTTPIndex]

    @property
    def routes(self):
        return dict((key, self.data[key]) for key in self.object_types)

    def clear_doc(self, docname):
        for typ, routes in self.routes.iteritems():
            for path, info in routes.items():
                if info[0] == docname:
                    del routes[path]

    def resolve_xref(self, env, fromdocname, builder, typ, target,
                     node, contnode):
        try:
            info = self.data[str(typ)][target]
        except KeyError:
            return
        else:
            anchor = http_resource_anchor(typ, target)
            title = typ.upper() + ' ' + target
            return make_refnode(builder, fromdocname, info[0], anchor,
                                contnode, title)

    def get_objects(self):
        for method, routes in self.routes.iteritems():
            for path, info in routes.iteritems():
                anchor = http_resource_anchor(method, path)
                yield (path, path, method, info[0], anchor, 1)


def setup(app):
    app.add_domain(HTTPDomain)

