#-*- coding: utf-8 -*-



import unittest
from stalker.core.models import Entity, Status, StatusList#, Project






########################################################################
class StatusListTest(unittest.TestCase):
    """testing the StatusList class
    """
    
    
    
    #----------------------------------------------------------------------
    def setUp(self):
        """let's create proper values for the tests
        """
        
        self.kwargs = {
            "name": "a status list",
            "description": "this is a status list for testing purposes",
            "statuses": [
                Status(name="Not Available", code="N/A"),
                Status(name="Waiting To Start", code="WSTRT"),
                Status(name="Started", code="STRT"),
                Status(name="Waiting For Approve", code="WAPPR"),
                Status(name="Approved", code="APPR"),
                Status(name="Finished", code="FNSH"),
                Status(name="On Hold", code="OH"),
                ],
            "target_entity_type": "Project",
            }
        
        self.test_status_list = StatusList(**self.kwargs)
    
    
    
    #----------------------------------------------------------------------
    def test_statuses_argument_accepts_statuses_only(self):
        """testing if statuses list argument accepts list of statuses only
        """
        
        # the statuses argument should be a list of statuses
        # can be empty?
        #
        
        test_values = ["a str", {}, 1, 1.0]
        
        for test_value in test_values:
            #----------------------------------------
            # it should only accept lists of statuses
            
            self.kwargs["statuses"] = test_value
            
            self.assertRaises(TypeError, StatusList, **self.kwargs)
    
    
    
    #----------------------------------------------------------------------
    def test_statuses_attribute_accepting_only_statuses(self):
        """testing the status_list attribute accepting Status objects only
        """
        
        test_values = ["1", ["1"], 1, [1, "w"]]
        
        # check the attribute
        for test_value in test_values:
            self.assertRaises(
                TypeError,
                setattr,
                self.test_status_list,
                "statuses",
                test_value
            )
    
    
    
    ##----------------------------------------------------------------------
    #def test_statuses_argument_being_empty(self):
        #"""testing status_list against being empty
        #"""
        
        ##----------------------------------------------------------------------
        ## the list couldn't be empty
        #self.kwargs["statuses"] = []
        
        #self.assertRaises(ValueError, StatusList, **self.kwargs)
    
    
    
    ##----------------------------------------------------------------------
    #def test_statuses_attribute_is_set_to_empty_list(self):
        #"""testing status_list against being empty
        #"""
        
        ##----------------------------------------------------------------------
        ## the list couldn't be empty
        #self.assertRasises(ValueError, self.test_status_list, "statuses", [])
    
    
    
    #----------------------------------------------------------------------
    def test_statuses_argument_elements_being_status_objects(self):
        """testing status_list elements against not being derived from Status
        class
        """
        
        #------------------------------------------------------
        # every element should be an object derived from Status
        a_fake_status_list = [1, 2, "a string", u"a unicode", 4.5]
        
        self.kwargs["statuses"] = a_fake_status_list
        
        self.assertRaises(TypeError, StatusList, **self.kwargs)
    
    
    
    #----------------------------------------------------------------------
    def test_statuses_attribute_works_properly(self):
        """testing if status_list attribute is working properly
        """
        
        new_list_of_statutes = [
            Status(name="New Status", code="NSTS")
        ]
        
        self.test_status_list.statuses = new_list_of_statutes
        self.assertEqual( self.test_status_list.statuses,
                          new_list_of_statutes)
    
    
    
    #----------------------------------------------------------------------
    def test_statuses_attributes_elements_changed_to_none_status_objects(self):
        """testing if a TypeError will be raised when trying to set an
        individual element in the statuses list to an object which is not a
        Status instance
        """
        
        self.assertRaises(
            TypeError,
            self.test_status_list.statuses.__setitem__,
            0,
            0
        )
    
    
    
    #def test_statuses_attribute_is_empty_list_when_statuses_argument_is_None(self):
        #"""testing if the statuses attribute will be an empty list when the
        #statuses argument is given as None
        #"""
        
        #self.kwargs["statuses"] = None
        #new_statusList = StatusList(**self.kwargs)
        #self.assertEqual(new_statusList.statuses, [])
    
    
    
    #def test_statuses_attribute_is_empty_list_when_it_is_set_to_None(self):
        #"""testing if the statuses attribute will be an empty list when it is
        #set to None
        #"""
        
        #self.test_status_list.statuses = None
        #self.assertEqual(self.test_status_list.statuses, [])
    
    
    
    #----------------------------------------------------------------------
    def test_equality_operator(self):
        """testing equality of two status list object
        """
        
        status_list1 = StatusList(**self.kwargs)
        status_list2 = StatusList(**self.kwargs)
        
        
        self.kwargs["target_entity_type"] = "SomeOtherClass"
        
        status_list3 = StatusList(**self.kwargs)
        
        self.kwargs["statuses"] = [
            Status(name="Started", code="STRT"),
            Status(name="Waiting For Approve", code="WAPPR"),
            Status(name="Approved", code="APPR"),
            Status(name="Finished", code="FNSH"),
            ]
        
        status_list4 = StatusList(**self.kwargs)
        
        self.assertTrue(status_list1==status_list2)
        self.assertFalse(status_list1==status_list3)
        self.assertFalse(status_list1==status_list4)
    
    
    
    #----------------------------------------------------------------------
    def test_inequality_operator(self):
        """testing equality of two status list object
        """
        
        status_list1 = StatusList(**self.kwargs)
        status_list2 = StatusList(**self.kwargs)
        
        self.kwargs["target_entity_type"] = "SomeOtherClass"
        
        status_list3 = StatusList(**self.kwargs)
        
        self.kwargs["statuses"] = [
            Status(name="Started", code="STRT"),
            Status(name="Waiting For Approve", code="WAPPR"),
            Status(name="Approved", code="APPR"),
            Status(name="Finished", code="FNSH"),
            ]
        
        status_list4 = StatusList(**self.kwargs)
        
        self.assertFalse(status_list1!=status_list2)
        self.assertTrue(status_list1!=status_list3)
        self.assertTrue(status_list1!=status_list4)
    
    
    
    #----------------------------------------------------------------------
    def test_indexing_get(self):
        """testing indexing of statuses in the statusList, get
        """
        # first try indexing
        
        # this shouldn't raise a TypeError
        status1 = self.test_status_list[0]
        
        # check the equality
        self.assertEqual(self.test_status_list.statuses[0], status1)
    
    
    
    #----------------------------------------------------------------------
    def test_indexing_get_string_indexes(self):
        """testing indexing of statuses in the statusList, get with string
        """
        
        status1 = Status(name="Complete", code="CMPLT")
        status2 = Status(name="Work in Progress", code="WIP")
        status3 = Status(name="Pending Review", code="PRev")
        
        a_status_list = StatusList(name="Asset Status List",
                                   statuses=[status1, status2, status3],
                                   target_entity_type="Asset")
        
        self.assertEqual(a_status_list[0], a_status_list["complete"])
        self.assertEqual(a_status_list[1], a_status_list["wip"])
    
    
    
    #----------------------------------------------------------------------
    def test_indexing_set(self):
        """testing indexing of statuses in the statusList, set
        """
        # first try indexing
        
        # this shouldn't raise a TypeError
        status1 = self.test_status_list[0]
        
        #self.test_status_list[-1] = status1
        
        # check the equality
        self.assertEqual(self.test_status_list.statuses[0], status1)
    
    
    
    #----------------------------------------------------------------------
    def test_indexing_del(self):
        """testing indexing of statuses in the statusList, del
        """
        
        # first get the lenght
        len_statuses = len(self.test_status_list.statuses)
        
        del self.test_status_list[-1]
        
        self.assertEqual(len(self.test_status_list.statuses),
                         len_statuses-1)
    
    
    
    #----------------------------------------------------------------------
    def test_indexing_len(self):
        """testing indexing of statuses in the statusList, len
        """
        
        # get the len and compare it wiht len(statuses)
        self.assertEqual(len(self.test_status_list.statuses),
                         len(self.test_status_list))
    
    
    
    #----------------------------------------------------------------------
    def test_target_entity_type_argument_being_empty_string(self):
        """testing if a ValueError will be raised when the target_entity_type
        argument is given as None
        """
        
        self.kwargs["target_entity_type"] = ""
        self.assertRaises(ValueError, StatusList, **self.kwargs)
    
    
    
    #----------------------------------------------------------------------
    def test_target_entity_type_argument_being_None(self):
        """testing if a TypeError will be raised when the target_entity_type
        argument is given as None
        """
        
        self.kwargs["target_entity_type"] = None
        self.assertRaises(TypeError, StatusList, **self.kwargs)
    
    
    
    #----------------------------------------------------------------------
    def test_target_entity_type_attribute_is_read_only(self):
        """testing if a AttributeError will be raised when the
        target_entity_type argment is tried to be set
        """
        
        # try to set the target_entity_type attribute and expect AttributeError
        self.assertRaises(
            AttributeError,
            setattr,
            self.test_status_list,
            "target_entity_type",
            "Sequence"
        )
    
    
    
    #----------------------------------------------------------------------
    def test_target_entity_type_argument_accepts_classes(self):
        """testing if the target_entity_type argument accepts classes
        """
        
        class TestClass(object):
            pass
        
        self.kwargs["target_entity_type"] = TestClass
        new_status_list = StatusList(**self.kwargs)
        
        self.assertEqual(new_status_list.target_entity_type, "TestClass")
    
    
    
    ##----------------------------------------------------------------------
    #def test_plural_name(self):
        #"""testing the plural name of Status class
        #"""
        
        #self.assertTrue(Status.plural_name, "Statuses")
