import copy
import logging
import logging.config
from io import BytesIO
import traceback
import sys

__author__ = 'd9pouces'
__all__ = ['ColorizedHandler', 'traceback', 'CONSOLE']


class ColorizedHandler(logging.StreamHandler):
    """Basic :class:`logging.StreamHandler` modified to colorize its output
    according to the record level.
    """

    def emit(self, record):
        """If a formatter is specified, it is used to format the record. The
        record is the written to the stream with a new line terminator.
        If exception information is present, it is formatted using
        :param record: record to emit
        :func:`traceback.print_exception()` and appended to the stream.
        The message is colorized according to the level of the record:

        * :attr:`logging.debug` => pink
        * :attr:`logging.info` => green
        * :attr:`logging.warning` => yellow
        * :attr:`logging.error` => red
        """
        myrecord = copy.copy(record)
        levelno = myrecord.levelno
        if levelno >= 50:
            color = '\x1b[31m'  # red
        elif levelno >= 40:
            color = '\x1b[31m'  # red
        elif levelno >= 30:
            color = '\x1b[33m'  # yellow
        elif levelno >= 20:
            color = '\x1b[32m'  # green
        elif levelno >= 10:
            color = '\x1b[35m'  # pink
        else:
            color = '\x1b[0m'  # normal
        myrecord.msg = color + myrecord.msg + '\x1b[0m'  # normal
        return logging.StreamHandler.emit(self, myrecord)


def log_traceback(error, msg=None):
    """
    Log a traceback
    :param error: error
    :param msg: message to prefix tracebacks with
    """
    if msg is not None:
        logging.error(msg)
    out_buf = BytesIO()
    exc_traceback = sys.exc_info()[2]
    traceback.print_tb(exc_traceback, file=out_buf)
    logging.error('{0}: {1}'.format(error.__class__.__name__, error))
    logging.error(out_buf.getvalue())


CONSOLE = {
    'version': 1,
    'disable_existing_loggers': False,
    'handlers': {
        'color': {
            'level': 'DEBUG',
            'filters': [],
            'class': 'starterpyth.log.ColorizedHandler'
        }
    },
    'root': {
        'handlers': ['color'],
        'level': 'WARNING',
        'propagate': True,
    }
}


def dict_config(config):
    if not hasattr(logging.config, 'dictConfig'):
        f_in = BytesIO()

        def write_value(config_key, values, value_key, default):
            value = values.get(value_key, default)
            if isinstance(value, list):
                value = ",".join(value)
            elif isinstance(value, dict):
                value = ",".join([x for x in value])
            f_in.write("{0}={1}\n".format(config_key, value))
        f_in.write("[loggers]\n")
        values = ['root'] + list(config.get('loggers', {}).keys())
        f_in.write("keys={0}\n".format(",".join(values)))
        for index_name in ('handlers', 'formatters'):
            f_in.write("[{0}]\n".format(index_name))
            write_value('keys', config, index_name, [])
        for h_name in config.get('handlers', []):
            f_in.write("[handler_{0}]\n".format(h_name))
            write_value('class', config['handlers'][h_name], 'class', '')
            write_value('level', config['handlers'][h_name], 'level', 'NOTSET')
            write_value('args', config['handlers'][h_name], 'args', '()')
        for l_name in config.get('loggers', []):
            f_in.write("[logger_{0}]\n".format(l_name))
            write_value('handlers', config['loggers'][l_name], 'handlers', [])
            write_value('level', config['loggers'][l_name], 'level', 'NOTSET')
            write_value('propagate', config['loggers'][l_name], 'propagate', 1)
        f_in.write("[logger_root]\n")
        write_value('handlers', config['root'], 'handlers', [])
        write_value('level', config['root'], 'level', 'NOTSET')
        write_value('propagate', config['root'], 'propagate', 1)
        f_in.seek(0)
        f_in.seek(0)
        logging.config.fileConfig(f_in)
    else:
        logging.config.dictConfig(config)


if __name__ == '__main__':
    import doctest
    doctest.testmod()
