
from itertools import chain
import re
import subprocess
import tempfile

from ..utils import *

class _JSStreamLicense(object):
    def __init__(self, stream):
        self.stream = stream
        self.buffer = []
        self.bufferlen = 0

    def __iter__(self):
        for chunk in self.stream:
            if self.bufferlen < 1024:
                self.buffer.append(chunk)
                self.bufferlen += len(chunk)
            yield chunk

    def get_license_text(self):
        header = b"".join(self.buffer)
        m = re.search(b'^\s*\/\*\!.*?\*\/', header, re.S)
        if m is not None:
            return m.group(0)

    def close(self):
        try:
            self.stream.close()
        except:
            pass

class UglifyJS(object):
    def __init__(self, closure_jar=None, yui_compressor="yui-compressor", java="java"):
        self.closure_jar = closure_jar
        self.yui_compressor = yui_compressor
        self.java = java

    def __call__(self, ctx, stream):
        if self.closure_jar is not None and len(self.closure_jar) > 0:
            with tempfile.NamedTemporaryFile() as fin:
                stream = _JSStreamLicense(stream)
                for chunk in stream:
                    fin.write(chunk)
                try:
                    stream.close()
                except:
                    pass
                fin.seek(0)

                fout = TmpFileIterator()
                subprocess.call([self.java, "-jar", self.closure_jar,
                                    "--js", fin.name,
                                    "--js_output_file", fout.name])

                license_text = stream.get_license_text()
                if license_text is not None:
                    return chain([license_text], fout)
                return fout

        return CmdIterator([self.yui_compressor, "--type", "js"], stream)

class UglifyCSS(object):
    def __init__(self, closure_jar=None, yui_compressor="yui-compressor", java="java"):
        self.yui_compressor = yui_compressor

    def __call__(self, ctx, stream):
        return CmdIterator([self.yui_compressor, "--type", "css"], stream)

