

import gzip
import mimetypes
import posixpath
import re
import os
import shutil
import sys

from webob import Response as WebObResponse

from .utils import iter_file_chunks, lookup, mkdir_p

#
# Response
#

class StaticIterator(object):
    CHUNKSIZE = 2**14

    def __init__(self, filename, *args, **kwargs):
        self.filename = filename
        self.fileobj = open(filename, *args, **kwargs)

    def __iter__(self):
        return iter_file_chunks(self.fileobj, self.CHUNKSIZE)

    def accelerate(self, environ):
        if "wsgi.file_wrapper" in environ:
            return environ['wsgi.file_wrapper'](self.fileobj, self.CHUNKSIZE)
        else:
            return self

    def close(self):
        try:
            self.fileobj.close()
        except:
            pass
        self.fileobj = None

class Response(WebObResponse):
    def __init__(self, iterator, path):
        content_type, charset = mimetypes.guess_type(path)

        super().__init__(app_iter=iterator, content_type=content_type)

        if isinstance(iterator, StaticIterator):
            try:
                self.last_modified = os.path.getmtime(iterator.filename)
            except OSError:
                pass

    def __call__(self, environ, start_response):
        iterobj = self.conditional_response_app(environ, start_response)
        if isinstance(iterobj, StaticIterator):
            return iterobj.accelerate(environ)
        return iterobj

#
# Request/View
#

class Context(object):
    def __init__(self, parent, path):
        self.root = parent.root
        self.path = path

    def resolve(self, filename):
        f = os.path.join(self.root, filename)
        if os.path.isfile(f):
            return f

    def static_iterator(self, filename, *args, **kwargs):
        virtual = kwargs.pop('virtual', True)
        if virtual:
            filename = self.resolve(filename)

        if filename is None:
            return None

        return StaticIterator(filename, *args, **kwargs)

    def open(self, filename, *args, **kwargs):
        resolved_filename = self.resolve(filename)
        if resolved_filename is None:
            raise FileNotFoundError("Not found: %s"%filename)
        return open(resolved_filename, *args, **kwargs)

class StaticFunc(object):
    def __init__(self, rule, kwargs):
        self._func = None
        self.rule = rule
        self.kwargs = kwargs

    def _init(self):
        cls = self.rule
        if isinstance(cls, str):
            cls = lookup(cls)
        return cls(**self.kwargs)

    def __call__(self, *args, **kwargs):
        if self._func is None:
            self._func = self._init()
        return self._func(*args, **kwargs)

class StaticRule(object):
    RE_FMT = re.compile(r'{(\w+)}')

    def __init__(self, rule, args, input, output, priority, exclude):
        self._func = StaticFunc(rule, args)

        self.exclude_re = re.compile(exclude) if exclude is not None else None
        self.input_re, self.input_fmt = self._mk_re_fmt(input)
        self.output_re, self.output_fmt = self._mk_re_fmt(output)

        self.priority = priority

    def __call__(self, *args, **kwargs):
        return self._func(*args, **kwargs)

    def _mk_re_fmt(self, s):
        if s is None:
            return None, None

        regex = []
        fmt = []

        last = 0
        for m in self.RE_FMT.finditer(s):
            regex.append(re.escape(s[last:m.start()]))
            fmt.append(s[last:m.start()])

            regex.append('(?P<%s>.*?)'%m.group(1))
            fmt.append("%%(%s)s"%m.group(1))

            last = m.end()

        regex.append(re.escape(s[last:]))
        fmt.append(s[last:])

        return re.compile('^%s$'%"".join(regex)), "".join(fmt)

class StaticFiles(object):
    def __init__(self, root, pathvar=None):
        self.root = os.path.abspath(root)
        self.pathvar = pathvar

        self.rules = []
        self.postrules = []

    #
    # As Request
    #

    def __call__(self, request, path=None):
        if path is None:
            if self.pathvar is not None:
                path = request.urlvars[self.pathvar]
            else:
                path = request.path_info

            path = posixpath.normpath(path)
            while path[0] == '/':
                path = path[1:]

        content_type = mimetypes.guess_type(path)
        iterator = self.get_response(path)
        if iterator is None:
            return None
        return Response(iterator, path)

    def get_response(self, path):
        ctx = Context(self, path)

        for rule in self.rules:
            resp = None
            if rule.output_re is None:
                resp = rule(ctx, path)
            else:
                m = rule.output_re.match(os.path.basename(path))
                if m is not None:
                    outp = os.path.join(os.path.dirname(path), rule.input_fmt%(m.groupdict()))
                    resp = rule(ctx, outp)

            if resp is not None:
                return resp

    #
    # Setup
    #

    def _register(self, list_, entry):
        for i, ei in enumerate(list_):
            if ei.priority < entry.priority:
                list_.insert(i, entry)
                return
        list_.append(entry)

    def register(self, rule, args={}, input=None, output=None, priority=0, exclude=None):
        if (input is None) != (output is None):
            raise ValueError("input and output must both be included or neither")

        entry = StaticRule(rule, args, input, output, priority, exclude)
        return self._register(self.rules, entry)

    def register_post(self, rule, args={}, name="{f}", priority=0, exclude=None):
        entry = StaticRule(rule, args, name, name, priority, exclude)
        return self._register(self.postrules, entry)

    #
    # Installer
    #

    def install(self, destdir, **kwargs):
        installer = StaticInstaller(self, **kwargs)
        installer.install_tree(self.root, ".", os.path.abspath(destdir))

#
# Installer Actual
#

class StaticInstaller(object):
    def __init__(self, parent, verbose=False, gzip=False, clean=False):
        self.parent = parent
        self.verbose = verbose
        self.gzip = gzip
        self.clean = clean

    def get_response(self, inpath, outpath):
        ctx = Context(self.parent, inpath)
        responses = {}

        for rule in self.parent.rules:
            if rule.exclude_re is not None:
                if rule.exclude_re.search(outpath):
                    continue

            outp = None
            resp = None
            if rule.output_re is None:
                resp = rule(ctx, inpath)
                outp = outpath
            else:
                m = rule.input_re.match(os.path.basename(inpath))
                if m is not None:
                    outp = os.path.join(os.path.dirname(outpath), rule.output_fmt%(m.groupdict()))
                    resp = rule(ctx, inpath)

            if resp is not None:
                responses[outp] = resp

        return responses

    def get_post_response(self, filename, stream):
        ctx = Context(self.parent, filename)
        for rule in self.parent.postrules:
            if rule.exclude_re is not None:
                if rule.exclude_re.search(filename):
                    continue

            if rule.input_re is None:
                stream = rule(ctx, filename)
            else:
                m = rule.input_re.match(os.path.basename(filename))
                if m is not None:
                    stream = rule(ctx, stream)
        return stream

    def install_tree(self, root, frompath, topath, at_root=True):
        if at_root and self.clean:
            shutil.rmtree(topath)
        mkdir_p(topath)

        for name in os.listdir(os.path.join(root, frompath)):
            frompath_ = os.path.join(frompath, name)
            topath_ = os.path.join(topath, name)
            if os.path.isdir(os.path.join(root, frompath_)):
                mkdir_p(topath_)
                self.install_tree(root, frompath_, topath_, at_root=False)
            else:
                if self.verbose:
                    print(topath_)
                    sys.stdout.flush()

                responses = self.get_response(frompath_, topath_)
                for filename, stream in responses.items():
                    stream = self.get_post_response(filename, stream)

                    gz = None

                    try:
                        if self.gzip:
                            gz = gzip.GzipFile(filename+".gz", "wb")

                        with open(filename, "wb") as f:
                            for chunk in stream:
                                f.write(chunk)
                                if gz is not None:
                                    gz.write(chunk)
                    finally:
                        if gz is not None:
                            gz.close()
