
import errno
import os
import subprocess
import sys
import tempfile

__all__ = ['iter_file_chunks', 'lookup', 'mkdir_p', 'CmdIterator', 'TmpFileIterator']

def iter_file_chunks(filelike, chunksize=2**14):
    while True:
        chunk = filelike.read(chunksize)
        if chunk == b'':
            break
        yield chunk

def lookup(name):
    # we only look up classes and functions, but both are callable
    if callable(name):
        return name

    module_name, func_name = name.split(':', 1)
    __import__(module_name)

    module = sys.modules[module_name]
    return getattr(module, func_name)

def mkdir_p(path):
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else: raise
    except Exception as exc:
        if isinstance(exc, FileExistsError):
            if not os.path.isdir(path):
                raise
            else:
                return
        raise

#
# Iterator
#

class CmdIterator(object):
    # TODO don't buffer?

    def __init__(self, cmd, stdin=()):
        self.stdin_orig = stdin
        self.stdin = tempfile.TemporaryFile()
        self.stdout = tempfile.TemporaryFile()

        for chunk in stdin:
            self.stdin.write(chunk)
        self.stdin.seek(0)

        self.proc = subprocess.Popen(cmd, stdout=self.stdout, stdin=self.stdin)

    def __iter__(self):
        self.proc.wait()
        self.stdout.seek(0)
        return iter_file_chunks(self.stdout)

    def close(self):
        try:
            self.stdin_orig.close()
        except:
            pass
        self.stdout.close()
        self.stdin.close()

class TmpFileIterator(object):
    def __init__(self):
        self.file = tempfile.NamedTemporaryFile()

    @property
    def name(self):
        return self.file.name

    def __iter__(self):
        self.file.seek(0)
        return iter_file_chunks(self.file)

    def close(self):
        self.file.close()
