# -*- coding: utf-8 -*-
# vi:si:et:sw=4:sts=4:ts=4

##
## Copyright (C) 2012 Async Open Source <http://www.async.com.br>
## All rights reserved
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., or visit: http://www.gnu.org/.
##
## Author(s): Stoq Team <stoq-devel@async.com.br>
##

from kiwi.currency import currency

from stoqlib.domain.payment.method import PaymentMethod
from stoqlib.domain.payment.payment import Payment
from stoqlib.gui.editors.crediteditor import CreditEditor
from stoqlib.gui.test.uitestutils import GUITest


class TestCreditEditor(GUITest):
    def test_credit_editor(self):
        client = self.create_client()
        self.assertEquals(client.credit_account_balance, 0)

        editor = CreditEditor(self.store, client)
        self.assertNotSensitive(editor.main_dialog, ['ok_button'])
        self.check_editor(editor, 'editor-client-credit')

        editor.description.set_text('Desc')
        editor.value.set_text('4')
        self.assertSensitive(editor.main_dialog, ['ok_button'])
        editor.confirm()
        self.assertEquals(client.credit_account_balance, 4)

        editor = CreditEditor(self.store, client)
        editor.description.set_text('Desc 2')
        editor.value.set_text('-3')
        # We need the payment generated by the editor to perform some tests. We
        # won't confirm the dialog to avoid creating repeated payments.
        payment = editor._create_payment()

        self.assertEquals(payment.value, currency(3))
        self.assertEquals(payment.payment_type, Payment.TYPE_IN)
        self.assertEquals(client.credit_account_balance, 1)

    def test_credit_editor_cancel(self):
        method = PaymentMethod.get_by_name(self.store, u'credit')
        client = self.create_client()
        group = self.create_payment_group(payer=client.person)

        payment = self.create_payment(method=method,
                                      payment_type=Payment.TYPE_OUT, value=6,
                                      group=group)
        payment.set_pending()
        payment.pay()

        editor = CreditEditor(self.store, client)
        editor.description.set_text('Desc')

        editor.value.set_text('-4')
        # Canceling the dialog here does not roll back the transaction because
        # this is done in the caller.
        editor.cancel()
        self.assertEquals(client.credit_account_balance, currency(6))

    def test_negative_credit(self):
        method = PaymentMethod.get_by_name(self.store, u'credit')
        client = self.create_client()
        group = self.create_payment_group(payer=client.person)

        payment = self.create_payment(method=method,
                                      payment_type=Payment.TYPE_OUT, value=6,
                                      group=group)
        payment.set_pending()
        payment.pay()

        editor = CreditEditor(self.store, client)
        editor.description.set_text('Desc')

        editor.value.set_text('-5')
        self.assertValid(editor, ['value'])
        self.assertSensitive(editor.main_dialog, ['ok_button'])

        editor.value.set_text('-6')
        self.assertValid(editor, ['value'])
        self.assertSensitive(editor.main_dialog, ['ok_button'])

        editor.value.set_text('-7')
        self.assertInvalid(editor, ['value'])
        self.assertNotSensitive(editor.main_dialog, ['ok_button'])
