# coding: UTF-8

'''Module to operate with different kinds of streams.'''

# Define global imports
import os
import re
import sys
import codecs
import shutil
import inspect
import tempfile
import platform
import __main__
from io import BytesIO
from io import StringIO
from libc.stdlib cimport *

# Repair Windows Unicode
codecs.register(lambda name:
  name == 'cp65001' and codecs.lookup('UTF-8') or None)

# Define module information
__author__ = 'Dmitriy Selyutin'
__license__ = 'GNU GPL v3'
__console__ = sys.getfilesystemencoding()
if not hasattr(__main__, '__file__'):
  __encoding__ = sys.stdout.encoding
else: # if hasattr(__main__, '__file__'):
  __encoding__ = 'UTF-8'
__system__ = platform.system().lower()
__version__ = 0.2
__test__ = {}

# Define global functions
def _screen_(path):
  '''_screen_(path) -> ustream
  
  Return screened version of the path to use with pstream.'''
  path = ustream(path)
  if __system__ == 'linux':
    path = ustream("'%s'" % path)
  elif __system__ == 'windows':
    path = path.replace('^', '^^')
    path = path.replace(' ', '^ ')
    path = path.replace('&', '^&')
    path = path.replace('|', '^|')
    path = path.replace('(', '^(')
    path = path.replace(')', '^)')
    path = path.replace('<', '^<')
    path = path.replace('>', '^>')
  return(ustream(path))

# Define global constants
NoneType = type(None)
STREAM_BEG = 0
STREAM_CUR = 1
STREAM_END = 2
TABLE_GR = \
[ # Greek table
  ['α', 'ἀ', 'ἁ', 'ά', 'ὰ', 'ᾶ', 'ἄ', 'ἅ', 'ἂ', 'ἅ', 'ἆ', 'ἇ', 'ά',
    'ᾳ', 'ᾀ', 'ᾁ', 'ᾴ', 'ᾲ', 'ᾷ', 'ᾄ', 'ᾅ', 'ᾂ', 'ᾅ', 'ᾆ', 'ᾇ'],
  ['η', 'ἠ', 'ἡ', 'ή', 'ὴ', 'ῆ', 'ἤ', 'ἥ', 'ἢ', 'ἣ', 'ἦ', 'ἧ', 'ή',
    'ῃ', 'ᾐ', 'ᾑ', 'ῄ', 'ῂ', 'ῇ', 'ᾔ', 'ᾕ', 'ᾒ', 'ᾓ', 'ᾖ', 'ᾗ'],
  ['ω', 'ὠ', 'ὡ', 'ώ', 'ὼ', 'ῶ', 'ὤ', 'ὥ', 'ὢ', 'ὣ', 'ὦ', 'ὧ', 'ώ',
    'ῳ', 'ᾠ', 'ᾡ', 'ῴ', 'ῲ', 'ῷ', 'ᾤ', 'ᾥ', 'ᾢ', 'ᾣ', 'ᾦ', 'ᾧ'],
  ['ι', 'ἰ', 'ἱ', 'ί', 'ὶ', 'ῖ', 'ἴ', 'ἵ', 'ἲ', 'ἳ', 'ἶ', 'ἷ', 'ί',
    'ῐ', 'ῑ', 'ϊ', 'ῒ', 'ΐ', 'ῗ', 'ΐ'],
  ['υ', 'ὐ', 'ὑ', 'ύ', 'ὺ', 'ῦ', 'ὔ', 'ὕ', 'ὒ', 'ὓ', 'ὖ', 'ὗ', 'ύ',
    'ῠ', 'ῡ', 'ϋ', 'ῢ', 'ΰ', 'ῧ', 'ΰ'],
  ['ε', 'ἐ', 'ἑ', 'έ', 'ὲ', 'ἔ', 'ἕ', 'ἒ', 'ἓ', 'έ'],
  ['ο', 'ὀ', 'ὁ', 'ό', 'ὸ', 'ὄ', 'ὅ', 'ὂ', 'ὃ', 'ό'],
  ['β', 'ϐ'],
  ['θ', 'ϑ'],
  ['κ', 'ϰ'],
  ['π', 'ϖ'],
  ['ρ', 'ϱ', 'ϼ'],
  ['σ', 'ς', 'ϲ'],
  ['ϕ', 'φ'],
] # Greek table
TABLE_LA = \
[ # Latin table
  ['a', 'ă', 'ā'],
  ['e', 'ĕ', 'ē'],
  ['o', 'ŏ', 'ō'],
  ['i', 'ĭ', 'ī', 'j'],
  ['u', 'ŭ', 'ū', 'v'],
] # Latin table
TABLE_RU = \
[ # Russian table
  ['ф', 'ѳ'],
  ['е', 'ѣ', 'ё'],
  ['и', 'і', 'ѵ'],
] # Russian table
TABLE_DE = \
[ # German table
  ['a', 'ä', 'ae'],
  ['o', 'ö', 'oe'],
  ['u', 'ü', 'ue'],
  ['ss', 'ß'],
] # German table
TABLE_GR = [[ustream(item) for item in row] for row in TABLE_GR]
TABLE_LA = [[ustream(item) for item in row] for row in TABLE_LA]
TABLE_RU = [[ustream(item) for item in row] for row in TABLE_RU]
TABLE_DE = [[ustream(item) for item in row] for row in TABLE_DE]
TABLES = \
{
  'GR': TABLE_GR,
  'LA': TABLE_LA,
  'RU': TABLE_RU,
  'DE': TABLE_DE,
}

# Delete temporary variables
del __test__; del row; del item; del sys; del platform; del __main__;

# Define global binaries
_GZIP_EXE_ = \
'''DISABLED'''
_BZIP2_EXE_ = \
'''DISABLED'''
_BZIP2_LIB_ = \
'''DISABLED'''
_LZOP_EXE_ = \
'''eJxEtQdQE2oTNpqQQiAJCT30IL13pErvHQFBuiCgREFMAOkhtBgCAaUoRRBUrDTFhAiEIoSjKCJi
EKUfBUFARIog3PN/979zd+aZ3Xefnd19Z3ZnM+1c7YFAIAjwf4ULAAIA/9NG0yyAPuD/F30A9j9O
+X828P9zoln/g+QE4H+A/M/3//LJyP9AhcL+D/6/cOj/5Q2bMBK+nv6yq9xIxP9x1d0B/g/X/7OD
/tcDAJBpEbmX+X96+19HyJYs8MyF9i8wfe7KBgk4VkwZgAQgzI+C2gAacqDVm0AA5Ihpui+IBsjU
QbBQGMj2vw8lX4hcJ2lAZbz+xUfy/vd2vcwN5vkv9ZH/f/E/X/1Xx4kXkLl/eORL/vfkEi8AYEns
l7gNBPDSwEBvbGBw/9H64VEAeYE8S/6HaksD6/zw9VWN+TwGgcVczwLDPE8erU0xvdR+K27RABRJ
+X5J0iosyzxMDno0fXgEA+BRK0bURACJBW6AYVFHY/VQCjp3rCc9dYewQgcAcll4tCYipu6YyNK/
Gfs7zS8P90kD6EGIKXC4n+Yr51m4lvTkb81pYh+pvjJIX9j7PCAmUnGCoRRt/LD+KUJKInBteEdS
4Kksmbzt1/Nl+uIpHZYPxReRh4fzkv5woW4dtu5ss6hgyyXqU0Xjfgw5eJcSjMis359RhJEJB7lT
FAPJzmTeSdIcCtXxdydzjgh4DyDtcmfokWa5GWgES5psK7+zk9ej7ioPJgvJm8DiyVJ65A2KU9ga
4bgNeTWsI1RXY+pv0hpd3BgGXnlF9pFH0IFSneCt1c9Dn5CiQ2D5Y8fckdUluZNlNIAS18URjdTQ
ng7FHjvYt4nudXbuNkVKu0ffJ4oBVRI7Vwt09VEy0BZNfWPOiTgXpXeTN7X/Df3fm3qZqeBVGiCQ
JkTur0+T+/w3F1WHyh0FNVOEHbHIMVTO1xMTnHSyqRQFiZF3itj/MgrTeYkwhYihig32GVV3zG2J
f/QhxhBV+NSFzlAHvFJxMEMTTHf/bHpG8ASikxX2r8NW8rB8xAzJ3BWoleCxFlYLwsgLBQmdkzPk
0JX7p9pOCcn3tK43N/DqjNELyZ86wRTn1O3WGVhDa+7wY0mKNcKM68F3DmE9zkfNGbGMDMlP7dS+
0vJvarRhrg+jCEF2bd5TBBvopBUzL7cLZrA6nW8gdEx0EI47CmoXXwBzAjBOH3UFua5jYaF6SpCq
xo/111eHsSOutoEZJ3fWA3UDDeVdGswZ4DNxQxwy8vcYUdgVJTupCRfsAsEIPEeegQLuoigEKgcx
6MdWOCcT9Nd8/6OGpPFohu9VKbRdHwPRmMpNKcmfauArlGIPsy9LoZFWUiLHlL4EoBA+V0bFuyMl
LEHHHyEjTRoXhrSlCSxLTbqbLc0qGWzxyLZUk4PruON6Iv+amnGkoTbCnC3EAZfrYPmp+M4g/7hX
1BdKb+2C0axljZcYa/Mqe8drBvfvXPGzlLSprf3yJI5ugZO0nZGQoLnaqHTIq3RaKUa0XKAxc4Pa
paNCRTZCUf73dzpLWNSeuOr6yvo40GN/82rDHEewRddz81gBy6M55yVbFb2+5VpFwmRro6ytyCLt
lW1nso9uV7m26crDDJlbT9VJ1UnBSBiCx7h0/XM8Z7YFvkhzIbatr2VoajePchNXl7ipbmjDPSQf
pYZao7x7dGR9CHovcRLg/mLQbgeAPeh/lYUXCyTryx95draozWnTtqXGH9Et+W543borG6GNfhEK
pOcQU+lHgLhejLii9Dn1M7PVGIyEgGtmXHI7DCMgtxX06dk3B8Pq9IqCpGQuKbEIl1vDwf09MOga
7ZpyJYxPskarNZaz7aGLl6yyRugsBR28GnbkSuQeJrP7M8uHcRd4VvE3MH1wid4a+GWXYSDWQk5O
R73InGcY9RawHSKxWlUwvD5Ehp4UCvvOYYRanPKzkgziFzf8zKSkI9rI+Jq0U5efy00psX1sUPlI
3hipwb0fp1boZb1wKSkXhBNp72sbKSqPQxDHRGLTD+R/ZnjVN+ny/qDm7dpWNDsOzUNOlIxHAWJH
5lfA9XVh4JQnyAWFDTEEAvdXOtlgtG5DwixLlaPUkjFTNVVVvCEJJ1dsncpDkouMDbe24ZUnEc4y
SRHR7ZIBfUbCaVVsZYRh4z55EB8ZBHinmRyxTITDbSRLA0NCQ1sS+7VcOw/6DStJ6Z1IR5544R6d
7bPSIvnxtEpCNd819rPP+9v0oRk5AoQO/7O9edmlsnBHCLztB+6fa+E1PIrbtN+Za2Z9uphHLbaj
C9lvdGfhqeW7MKlO3Xpi6+qXTlPWxIWAc8B3aMWVj28/ndW2RhHShdolFCVkhPgug+2/Ig4yonvk
co/EEU1LIATy+4iiSa0L17zSrMr4c8fLtIpxphhwYDzIBWMKN+OK1AbkVvDx7FDcRFg6UxvJOff6
ycRPbjd2psi0FFt3+5+Vfk2GgdpC5AIB/HkLNZ/EYvdnt5r+efAofV3KsP7KWCoG1qvLgS+cF8Cr
KILsy98aVsLAqJMQQaoVZlIcylHNt4CzL5e6I/05hGKJNyDkgwn4l3o50pdU5EIRQhqGuvZLRBJ8
H6J14GrA1+yrffybsONllrZpOl+YHDyhUAd6Dn7hiF4+deFur06E7/lqZq4WJrRNl4q0vJ+6nTwU
0/twRSFQYQhp6He69uGwop3jafvea0rjr9p2/O3nxnXOprJzP3/XbJJDvVho17hxBUF3nvnttuE2
JFLY+29JYkAoU74q2YKBASpND4yLBs7KgvNlKBfW4T+OmY4jbcWQ6fci2712FfMGYqouuyQjwBeM
YuTT4wKuTwQN+G4gG7X8wP7oMKneQw6ekt1A6dwKvIFOq2mVekcpEWHc9jG3NeK5bBBToRFdWlLy
+IYFje7TaBy5SCyx9Rlmb8ZZpyCyKen5FfKiCYY+Joy0yNUbqxaOKVfoFrRSODMyjdhpq42j40ws
4m0YOT59EbKvQe78jvHrP5z1iX1iXwiFWAmb3I3irxjXeTsZjwarup/i9wwvDHYsrFHchX6jrrNq
Oza4+26kniMyETyjTfmqpUC8L/i0vn8KOknrPl6I8R6Xl1dNPuZMoXDJmsrO8AuYqzvSqZJN+cZa
93B6d9w8tETQbNMcZOSZSgx6aeaiggMp7KTh04EVONeBxvQglzJHuYDL2onw60HS675e35XWFyvc
NE1iwBEXrABm6UzYDe1RP35RW4hbVArhQTpIsqRePMo1VrysR96lDdpC3B2v0H4onni6Z+bhju/z
QXLAd6aYRyDlUUBfAcimAOl05Zu7LOFfi7vSUJcddWu+D6qITewI1wdHuvoHW471EmmlYgWZWbyO
YvP/AXHZCnMKQOr+cXFFzldWr/4hfH9bXA67UgQLKVqvjm14AThOKSEdAhMvEXWmo+moUEfbgpyp
Kc2nWYcvCVsFuTAovPjvPvSKEXsFIWNtVCgstV0g2JX7qN7mskqoeo7wDq9G9YiMrfCNtbOIC+tH
sAK2O6GDOBDdE0xX93heKFMkFOzGyWG37Qe/LV7j/ArmD9nNjQRaZV84CyKr9WTy4zt6WsvQZSgX
QmeOUKjCw7D3PS+yymfHAc4GjxV4b6JaG29YdnIhTzxb/i33cIcilXB0shV84gDOeFIgP5Eb835W
M5reLngVwR1DiBp2zwPrhzzpSQoPZelf78xZNGNGbXxGBmnansrqfG2/ExZ2NiZ4+IjXf9//Ivb4
6/RgC3/dzTBh3sR6D8aJ5Jm109Zo5gWGpx3TYDkEJd30FGbg1x91jxXRGffog2lnatdefwtjqDRd
v19XiYn5iJHlDgtdBNu/OTxtKc35YNhJz+J44xxfClroiBfQcoawpyuf5Vx2TGqJ5Tp5T4BPy7pV
0PmRorZEeXpyX5+/E7NOSNHDIkKEKSFhQh/95FPnqUrL5Y2rDK2IdIw7BZCEnbJQjJYzlxU9HlIU
Jofvlp/WxduHbrhkbq4trYgWXJda2nGwsSTuH375c8cFkMQdgs4ZIyzJUqkiWpX7GPrCbfkCna+t
fslYFro7XJ0MDrwxlboXKi6TczU2A9t7ZDydKJuVccRLkJhxNOYMnl2Pm4EkQrOSxfGbVHcg3TRA
Zq2v5wRec9DSs3P9iizz7YhRzjrH7NI5DH/y40gU8K+9WcTreIFCNWFm37rOfvcT1+oTjHYq7ecx
EMJUDeNv+sZMQcRx5gSoj/8oqD6WhYmJS6/KUQcuUglPqTtku6dYiTUnxe2w+Q5lryXzh2MPMNPc
3c81qsDTOt3KhvqxNO2kFh3558gnLoY6yVICJxSb98erZnmyjABADBtLFxt92rjSHT8NwwcFYQQT
SrTbEd2yYsPQlmc6y9m9306KPp0hMgO5GL8YlS+sco/C5DUyjfZAeP6O+J+DZHZ+75LM9+2+wJ+i
QXjZvw3q8sSYn+JPlA9+8w0MPqk1V4hpwmsc4LHjPxIF9MbbxDbTKgorBh6N09tYns8hJkcXLGBO
w4THFGGlEdVLtxsGYhVX8sarLl/c0ZVNCHt37QNETJXWHLj96eJpQbaCXmsZ6rN2HQLyAACKxMGS
gsGEyZ2o61f0aKFGiC9x2D6lMOyFf6Ocznv3WlouyEb082lfLGI9x03i3Ff93T5Grrb8qaeC8z6m
WPE2m94Sm/Hv2iEuHHmr/0RWdLJkE8ealvvrXYmT4yqKXuMoMToRvIb0cDYvnpzKYvi6Fb5v+hWb
NoMmBDFez4CTbVfP3F9niLPl5JBj+BPqk6SRcivwd4hv5BDovqjnp5XHfBtQ+zZv1HjfMKyiW+qY
pO8lmGxf+NOsVIVCiH/AsrNWspMM+eMcTd8ZVgQ0gO/qzAtWSMXApcSymLutN7v4KSSowGWMzS5F
2LZnd9rXooXzkJ1WdeS6DJtbWtpTkwq5Z3e1NwyN3m1jScWebMgySZfGuHy2nI2Wdenc6HN9yO7Q
XjgNsk/4NR62cW/9sJ+HA3TogKpK0kEfwnWKNiXPaZW9eKVAGn/6+RczqPMygS3v037BmtDmMzsw
8QuQg2yD53GzJLdztPty1hFq7nED3Tkbq9Qmg/Jj0qYj3blvOQZO21L8QHXkUdXqGbcADynE8aFC
nifI0QRA0r0UvTA6jw7rQ6aObnp2VZyxGlN/ZoZWajvhqjAjPMZxYjNdpIZLf/O1V9Dae2xzBuHe
Pm8a82RUITXBDQWI/lsOBzEDT18zQPRPjdxdRPP0cFmbjn2VpVzlYS1gm6isDZfOsHoLiqfw1rYk
BT3webkrQNtUx9zFLSys97iWufZx2g8tNbCFlhPSUpNr5lsjatDOGVuQRW+srAknT+IzXGiiFcdy
tc/wGNQc0SZDDEruo9HDBlHp3kPiQueYmjNTQeG0HFNhbe3LLcR0Jij7+ejyJTzl4q9vy/+ND+K0
Ms5Yn6ewrNCgSzJOUvZkx5GsGY8a9prsenIcOPkwrHv90B+q5JaIwGck34mML81WuMG2HKt5cjRs
doqXyC2r3AULBy2CMPpHWwtzHYnxdS8xBaNWHeuOh7Rx7jEUByoWSfs8u7g07zkShVwNvtsf/1TG
o1NSFpX3MevLG8JoVkqS9Ba8zq7140KNlPyNaPEVvrNNjNWfAUidpIX0KdcG9frbMj4Kx8UiTZ+L
+SuggsLWS6ygkTENngr6ShVrJbiGeHs1Ft4++b86rfYiaVk5Su2M8gtggUIZB22MCIopZ1A5PSHX
CLp7ukgdJDdS5zDadxKx1DTTNTFm5EkGUUCWGr2GaDhm3QWjjkAW01yT3ss5fjeZmDM3rcWqkjmw
iyLnGOAsX62gAiz99NuhqBn4lQSG7GYoELdc0CrYuTwg08yaA095ii65CiYkQxZ4TnvVmMuVzzgb
YOJOSz17VB5TvIZmun0+VrczakHcV8tEmUyA/pb15fbhLX64LVCkpwxZ7fYd7O7ky5o3uNSRfy8B
ayP4yz8jgfI89rdb+Ifj0RnqKYY1owl05ce7fdmuIqOuZ5ehXNVXbh6506bDBdZanG4EBQE4jcvu
YMSN67NOdRbKZIJYnFdd4NX0Aw0bcvDWX7lhHuihgaIxRbQI9EUAW8CjDjVv3clx0C597v9IUk6u
h5mUBrw1m1Kx6B8UWzhkaPzLST2rp7BbxAW68hcPpT5HNjG9byWzFGOwkT1AIbEVMFcDowEPy+dB
aAT7G40ZxKreMjeI6chJ647bVmxYp/D1wcUB6eLNnTpZkps4ZpdPkOmjdoqHgu8//fzW0nxzQ7U6
P/rkHk5UqKUxWPbnuYiIHQOUvsx9b6vzKD+q79a6vD7XDqGKAi0UWNAWq/HNA77M3Absg/HeAJkl
VIfd9tG/mbE4KGEH1eH7u3cW5rbFM9g7j+ZJIfEfTfTOYXjekPVctmdZKotXbim3rGRz3F/n3Woe
bz3ASaC2dR8X2B2dgUEO88GFhX8PRBXWchxFbt1SP+/4u66/Xn5QX5yZkbirWlmPfY84T7HHvDFX
YPJN3wZQzQFZBzpPjL6z20CaPaYvB7jSplrkuVAEhn7sF0vdCj/sJ5K5ZOCUhqiHnTZaQuz5t6nd
2/sbxCFPHs3UgjUz4SouXPCygqaAzdTJjuna7z03ClyQ9ZLeQBEhMdbDv9ajN4UhABgAxBc3lLke
IyILJbHBcF80yQR485nU5JJK5QaaIEjI0XEkeIgXixRYCIWsfIqOFCW6kCnELGjr9+aHQSqDFHfM
p24D+NSjSj8MvyeKR7J3V2bdMe5WtUoTHJvVnYa3ISaz7/BqhW3IqTwtCDUAWWLuGO2ISXjxFIr/
KzNrPiPF16cTqzowQxy4MfdJMCSgecimudgsemZqVnkXdReM4pA+5rGewxDM9SJo+iiUKaAimc9Y
yiMGoaz3qV1L2Z+U6DKLaffY5NxF4S2Voy8kjnGW3ASCvEcBknrBVFpR3OHU0SFgZ1S2jzyqkIVf
Is5noXveGfJejEZjuVmkSZCKd2bc6h5x722+STg1Vf9tmNKVZJmBEGwS98cVifBVCjV1WC3mmbG2
Ma3qXoVpTz9XmqvhGFxpy+KORl/5iEHJt6nSy/qpjctuF454TnWirZQUt4idZtVFn5Xc6ghGVuRa
Ngb8yQvW+yrvddjRlgx/DQVFTd9sKXD71btsTVslQDYPjaSupvelUvIsE9cdycjB9dWuUzhsuV8P
l3Z7woj798pz7lkSASnPbVjBG5rzPzeCRBwv34hJC9qsgZBuY4timNEfBHQRX+QdOOzEDti7tZj1
tXcJ2AjBohgE3XtbUT9t6N3rKciZ8Ygq4VBE5A9v0Hb4QUHwjyFwt2IkTt269rUk8NQmzSatB2h3
cc1cLcYfqR4SuUoNg8+NcnFpD7WBZiRMJJmXXscdeIH3KB5oriWybWEh1WzlO2R5CbQsgjXjgN/n
R6aGvwdvW/LDtMQWKhe0P4lo94AlMBgJ4/DZtoV5yFzQWatBpdlt8IU4OPGn5PDvzyWtJT8eLs9M
Q778U6qQ8/mt+/YiLa269nWFJAOMW2w8i+UApj5BLvOER05youCn5CXC5G1LJvILz9pa+CJIfZrJ
jk9ZvrLv4ijG+TQHaVTFR9k/5FebYyz7VLbQ1Cvifr6eobT2k4yP99KMHIdN2j4+YVxh+YfrQX89
AQ4tPb4WnhG6m85eDtKNSowv9gfT8tYAlNmgFJ91f0eNmYRtG3705OGFQXy68MBoge/73u8wyNjC
f2v2bnQk1K4Vjjg+Bi8vcpT9Uio0HzSXGg2qMRtMe72XHImLfPsGMvJ6hXjzsz7z7tMZT7oRCIWx
Kp3u17KnIFXGxiyFji9ai+rrGrY5ffsK+XdRhL1T3v2J5UorWdM9FfSy20wMHH48UaPML5N6wl95
hJOedm4g39U+E0C2G70xuzlRW8BrYEahUAoLNV9IUP6b94/814qe6bZGQjwib4XVQ7qqATaoWuEv
dx6yEnJ1rwjyPItPm9ec1f1HpYnzlbddvXdBsVdUZIPypSBqBR6alWZhVpNwQPB2vLLdfMsXKHGf
J19PSHawLiw69rf+q7Ub7eUN20nDBu7ZBkJZjk6IoMqo51kNlSijwkrXMs57Hg2hM8xNATbK47lo
jJ8u1V54l99LP8rp099bFPAz1k/74R5T6xYtjvqsk+1P+w2LS+cOeEd7VqkEzq+7qAqOTBgTU8o6
YVSqX9nDL2aeaNo7GSqNPwbr6ZrR8pbY1JrUePA2AjTxj37lEt+BDhWO3AgsJLEgPb3egz2vNd4c
DzPmVs0JE9Nu73r9N2RQFnyFr1StD/Ry8QYNzgse1lx01uR5A6slLTq/B3EUU79bQ8UChiYT2771
SUFMg4U3per59QURhGaXHuOMJAac14wHjzYjf+Lvo7qyQCaftiDFb9x0PkI/t5kwrgQzBkE9Fe1A
rXNpG7qVJTR1Y5qLMr3FlpzsHMmWdzlnf0JKSxf/tmm7B68F2xwhxEe3KVx7N9zT0t2AD3vIvaRT
oCX03vJhkqDETTfMGlDYET56ozU4NZgfWRUH8SzwAWMKVQXX9YVpbi8Tk3iuW4omXoIXgS/GiAA7
f0iVhuYyIZ2b0GPj33tmsFaf608YQzzwwi7ZO2jEnubhrPpSyMQYr+Uo7xlbmv/Yyp0e1i3cth0G
yL2K1yRqRv+1nmBaWqMAefmGv5sfK04JwniEpMyYrafm/aRWNBNRGqar5qtDduNaRv9I8SC0drrw
0QirvOMhdI1lntnfCcUyjySd0uxf3wh4zov+yTLdf36kFZsknQAxtUNfFqWPvAbsLFDBHVT94xlq
dKvxFZB6Ujc5uIXTbSuUW9ayc7Yl5+k8NaYEfH0G1byXtv94DlMl9RS4p3PExwFOkvu8BIO2/lz5
jJgWeIdLFdNLanqWfEKyWuxo6miNWrn1mR4g+vwdA/DPTdNc+nu3M2jyx9712+8MfyLP7aCvqPwS
DOzU6fAPghYtjeHD7TrMcQHBYIfN15IDqzTSxHSnVcXF9ueRk1EBl9BCKvc5p+X4hWT6/+wjXQeQ
XBM8PwmiJze++IXNlgoJ6ei809ngSRthRKBM0GqKOJX+zTSLvkTjbo1t35bw4F8/dUZzRQxJ6S/F
ZTj3mPyz7lsOj1MfnBQRGsZY0OeQ5LzIxeOmUb32tPOrtwJxGbXOlaC3iCQ52GXJMA6v1XDw+GuE
XW3PeClwm0wYZyj5x5igxwnx46Lsv/3WSc/0no4OfdgE9gN/sbcKW7ReE9LQ9W3aCpL3PrLPvr5/
4gBGDKBDdaYMbOxP1MoLEPApx7F9dvQNb98X4UpCqC1xUyFS9+kdyb6t71q+mHOSTQjIKREtW+2U
fwFzG4N6dA1pw8+YOBwzYlvKyM3w96uQDXNQoBBDfTb+TdkcP83JV7tDwjDxjwo+YNIoxVq1FjWw
9M5hVkjA8eMl/ctBJNv6ORbWIuzC4PlKOwnLNURJ+/yzIOR9zkDG4ydXeEZNtZ+ZGhTD3vm/mFF2
RJmbZZAZIsLg+cqYK1CK+KLQSlAVJutgWl3ipmzIbI6n1Gyn0qp/macbx8wGI8iyPs9JspHhkF7m
ITrbzDIcGS94JA+XKH7adJp2XNyLBw+PHVMXUo5ojTG5CMV0FuNF/oaDaw2Nsswrc7SBGEq0eXOI
Nka20MpvA0OCJEGs3PwTR3VijtsgPbukfLTVK6+EDd7msoEdq27bG94xksQvnrjje3IBnS8Y8Bds
ZI1rbZIdeG8uY36+9+/x7gqvNc7m5+D+PNfXjqaVnXApNhtZfGsWtjyU/HxemtzdKPnkcQF3b1k8
dwtT7DEbN7OW7t/JnzsIgWlAE9kT0AlAD7OcVTrS/fZW54OlNnx81kjToTM9WSHmUeioXkJ3Hf6Y
2Usip/wCNYjhMfmju8L2QZOksW0n7TyjM9gezee6Wy9+g1aUZVzfKxh9HAf4LJtpqowT0+C9PNaQ
KQqX6xzN1MryAFMUdE5wCHqf2gIR7TBHBY9bilW4E9PKjOHr+sOukbhOK0ddXZ8NaEujCbB/aVnH
8aSEa/0Dbn/XWf7bft9/Ao51qQuoCkhHrEV6ODOCCZEvejgeX+g8c6J92QCwisw6W+GL+tObDFdo
4LwUMk2gnooOG77oV1+BdF1tzYt8f6evU2RucbDE9sUKsRnnpbyzcmaihZvADWPD0sJelJkLgsex
tfFPb1qxc7Pw7bEGgUxNihUGWdg4zO70ASq/KberHVvij9uxoVkVMI1c6eaPYzTVTZFKdnisQ4YV
tB4lZGStXP5qXmlx7jfkDAIt6tZrWTgYXj6j7uobYmTF6TbTSbXDWPkg2icE2tBwZbnf116uAnrj
zjKb6Jt5cjuFowLMRzhbxMX/7lRO7k90NkopfdPK2kbXrSlSxenaryz9p1xdM36GdEGm1+ovUO9f
6YeOoXRXU4z16gb4o4GkomCl2Ytg/8EW5sndoGXlDJcEDab96SCd7TwQcmPbEuP0YFfnBjp5aMTs
ApAqa2M/JV8w4JSYFjBn8e0OrT8qSJAZlpCOOI0gfZG3/Hj3Y6TejCDbmGOaM3sxWq8yVzu0RDU8
p02+Psh+cHkXb9qN92dmoXe1d0ME4nxbup3Y73tO3hu1MqxvtAqwsoFhMsFG8t9rbe1UFbve0IN5
DEZsBtRSXjiIUMrydm6Gwny+mJhCvzaozYHWj5AufCL501eX1FPm/lZsWFHg9Axl8j+9ywjivBOI
KY61mUN9QEBRxuFyWIpL1dSNa63GPm/0iF/Null546+NL8MYl1Y97F03etcQCglfUf7benl/wk6i
6fnVvOzAtgBLDKMssGk61aNhVLIPSY8rlPxAPN+sos1VTlCwZaM5y4pKT+dMBA2WXAAyFBshBRFy
OqLa9vqdMV7V5SqN5TAtJlNWRxVx5gNXczwr34PlE7y1QNjNqkI6n+rJOsKjYa4G2rHNJnIxI90J
MkekHZjhdgNW4OS8t4wVb26JedC/crvpcwZhEGSkWAzqhHF2g/QrBy6V9cM5n4a4xHjzY1joKYYZ
Qa+NcXQxtvNiXNJF5olK0+S0d66k1CiyVBZUejA5ekC+wHuOQ+eJLU5+VlLIyBiHWw+6Vq0wLM0n
2fxwwJ2lNR5MFfWe5teh86BJaeTdgm5Xn5bQW3XGAIppuMNrUaWtyPjOJS2gIU3I4TiAmfvIbFeE
OLdB3APsdYQW1FXp5W4TPF9Uis7XroEwf3jMvo6z3n4XbrBb/W1n6VnQJdQwQjY9hBTc0MwDqBVi
g/pjPYl/c0/k+PeD1A+GHM3rmP9IyfGri5C8JFt8uu7NsoQapOuliLUqDqsijlIJCNg9k03Hc2Sj
Plwf3qpu7Mnm6iRTnlQvbWHn6Zn9LfszzDxv91Os2aKM3PxTb2vDV7kvYfNn/aH+SwovbDcf3S2w
X0xr6pDhnyJEph0UEk7zvqOqsxDF/jdvyb3aGXxRmQNrB7350VKoDl3mvq3eV1oiFfx0hSsRsSZy
TuvRym25aUE0TfTmFD0sKWtnrlt5T0aZIRXS+5xWIcO+aJ2DlfnUAxLw0JHFSoJtZVG9vCLdCjjJ
8mOGpHnKDUBUSUzCOVlmaJES5RYtyP5MPVv+gN4A2RNJRrwaFTB7v4FeQQidq0R93mgJ/zZMf0UA
r+rVXiSqiquPvYBHitJ8PH1Whcni/sV4cfnvjLw6Ox5DWvNdmPDliWmpAlvTNLRn/bYnKf6FNC5/
IaNb2nZuuz65hCU2sBOUZZ8DkUoYiNmQ3XYJ/2ArK/rHp8+5rRctnwcHr7gCkFYbuni547pf8Nww
0zCW3Ko/69fwf8pwfjW9BRCi7LBUv2udpN720+XR/SU/dQb7gmDYOG0d0P2qzWvdfpW4oreBMjRz
0HbREmA0cpZHmXLFg8RB1HGJMaG/7zSH6JqPbZUA+mcqDYARxy0j8jd6U/QBK/bN2VLKvYibXTWR
sZBnGx4f60gsfa1X/0IFXK/azsfRT0GeC+pibZ2o2fhwrEj4TgKZdP3mIIn19wJYZMPj6O8fflLJ
8p6WQ9qqEum5emUCtM5uTrtQ3higAWrnkV+Hr5+LaYqmGLjfzoNUiVsMOngBr4pNkV4dalc0em57
OdeeDDNHTRWQl/6MObVoc2RN7xidCFvGetqXf3vv1tPtn8bEx7UAzwVVmXV7uYrsuSfYDeBzzvSj
6W5WkEEP8ZV4tsZC9eqwdnylnat4z4uvyTy9U10btX3OG4SWae2xyy53HQeAZKUSdw1Vz1tOem2T
NwPmtxl413liS6CPyylpj4cuKHjIncWsHqBHZtHq+ZU6pbL+lFmRMvdnB5z4/mb37dWuoprVD8Jt
d0hHOR58ppESUzAuaic/CCr+YoK96TzNPbgZKqQVmUqKPML7TmVv9RwMATUTwpMGrcAHXrkRYq8H
Pxk+PUJGbtECLyJ0cQpehrup+cfeHhNaD6DoYIUxe9/KJ+590zmUcvqHnkL6BGL+0MdZ9HyHkLMH
bcWLlzmV9aHXs28Yio05iYXwVtt/I/ZZjXzurW0lNB4+bupqoKMjeqiDe66eUi45U3zY3mOGGGeK
roxLwUfp7290XXbxkaKYOjtyn7zH96TrV32YMkrJMtsQylBYquVbvgckI37NMH7jTS8JirZCxwZ6
Nmk+wMXh/CpRbXMDpvw6TN6MzBN9nfPINby9webi+XjsKGR8VcZt2GkqqNux5xiE7GDjeHtjyaHY
/161vUfFVSwu2L52sGh+h3dSPS4F4sanVp98wq28m3kFfwkqvsHlepiudYbrHjd67/jVs3cf4Xqe
zo8o4gtkngpGNpTj5t17FkKV7GaJ7kn1LXUx8HYtYdltB34s4q8w7iDUyCBnrOAdAb5yn2aDplyt
iC8c/muataeVpMaoshglfOPhFSZ89b6xODkeg6+fp9vWycND3049vHYtaufLV5X3PMvaiB9jd8Lj
S3087IEtky5hXJpkur+E2chFjlaZs2e57TBSB8qlYOHS9WopvHR+KrP7JuTSIx6heLc3dx5QaKH8
739ZcjdEaOLFHiWjs8/y41GmR0vLfKJbkvW5koKR5ZBW7WKBMZErWT1gzU43qPbycWF/PAM2SLQp
1Xz6EaAVrkV0AGNXrtq9KwlqLLl/DX/tnenbbO75Ojvz/BuaYv+cXvvtFSP38wFdRtZPKb00tqH8
hQy1DPdX0mRwqrYmBlJrm6wYPqycoeQLDi1T7+Eb1dFaicj0uXVvuiZXDXME0V+aOlsd/8ULZ8Ms
onY1FRvwuDsUQxfuYneBX+TA5GccScoZOmDw2POGMilXTxzXvxdDDqtzWw6pz2oHc5/BZ19GfRh3
vPPLqQw1miPSH6rvS28HPFeA/HQfzzHjSr9YOP3olKWKBylJrPc9kSuqQTxUGz5NufbwA/P5T5h7
49lq6g3fA4Pn0Ge3GpR652A5hnXPE59t5cJqd49//ulXaW6OouqBd6aoJ49If6ztD2V9Q2EUcA0T
dUHsWxCvyHlm7xdmCuojz3YuIxyMdyotT7hGzokCLG4kkezi/fjf5iKQWV3oBL2rny8gPk/7/ZD0
Q0Le17SH7wgb2VxzsiPozvTiCn08x+Rivvnj2swjpxIWkWd1+AaM8JBCIE0BrBblvow/EA9ZgDsq
DrcMWXpXur2AeyT1AUhdESm9nsWzP2XfEQdq7TMG269pCaP3MZTW3t5lmdbf5u/4HrDZ3k9ZLxda
KEaRXjGgeozI4pTxdVYQTt2kfPaZv/tl4RYyMzV0hZ/NrgBelxaOwQ5wJ4vce1YVStXuYXcIO/Fd
HVU9O0FT61PHDTYfO+IbVDt7qr7ypONT+VcReUunz7GE3S4hU+sjwe/3dTH6Lctyif8g90OkCNHj
sJC2+ADeXZZP8vvPOawVHsheXighjjL6yqE8cU1DQvVcx90YgopxIwNwP/JHpw8x32snZO2ngk4I
5xEXA2RGWvp7AZULenFbdFAELtaYX50VlI3gO+fLPvMqvfrjZWv3GmKXhXtLaKVC/qM7/kY9NAL2
xs3II+sEXSwIfbNq4pqcTYs2+uSEO8VoKjYNnaelswyBYxM5WdIi06T+qzfy7Kr5Z8Ie7sfaDq4N
PvxO/kVZxTXOyvTuywA3+D/lNIhNBvAGiRZLHWgXvEekIe+zifETQ4gXBlJV+iNpQfrLlK71t5dW
ygB45MjdgtynvzsmtXhYe+E9n9jXPV+Ihn91JNLgVWOv1AZrgaukQ/CIVHckvu7tWb3zbGE6WHfY
XoBwVagcTZE0ramAPitrvXh0a+PNK85XNM+v3m8Ynv3tIZMa+v3KoBNVuWHzNsNhqKl/C5t8ZhG5
UmWI5voZzmtJNhiZm2lsvBk4U1NUVkyNLv3XuHDh02ZBaDTpTy3dJTSsOs0jKK+L9U3qX0aBrjxp
w3VQkkQv1PM1UNN3OHHPcvDq32eUGvKlQaxSwZtnue1buB+126vThw8Geu/K2Js3h6HMZcBJwmZv
85vxokfgMzts4clFsT06kGRLErj+orEi4ija8sW3H2VLuOAFg+13OcS0zKFrrCfuYnonqGVN9++c
VSNKgK3fZljbsgrx92A+ubZecoHOlpJUvKzZIwV5J+NDoLLiuzPtTMGmowyVguvmRqRBfSZg1x4E
cijnshMUB+ndY/UE/1sUX4jujq1XzI3UWsC5pEia2aSxhBW0FsER7xMWF9kZXZnpz9Jaty7xvW3o
nHmECZBYl0Pw7Jh2TdW+Lga1l5AC1J/G6jjWRPPlDJoV4RNPqMIqVShL9wHovCJhI+WoxtjO3cNb
yNpeJ9LcV2oAclmZlkKzRyyksXp+Czhcun3KLy80xjKkvAlA7v4IUmow8i2OObnWl6Wo1JPL6RLv
Omn5isJ7+hNfYyI/nCe4HH38xuOu5ikfdd7pgRwjDfgvArVrIlWG65LvTYfcxr8y3OesZ8LrYiRa
Z6QD9kcb6yZMCuWkjgXYZjHUhSEYsDOfBNedaVrbshZOny+cEPIsZy8qOSGS5riC16UVSjJc8ZIL
hQ5deb7556m5lDZWVTUKNtDAAkPRSvoXk/vaLOWn2YmB/XjsDe0ftCisktKp6/al10p8nLFKNKto
dEqH6YvPAVt5XYtfq0qYQUNIclcQtasSZ8eWjXKVicw5E2fetptnVh/9m+9ZlX+A4NVz4rjGymfv
BUMSbBzX7zG7HwPfXw1iAKgfbg2WAYqCc6rtrgE7MT7Qyd0cv36dbdmBjwJsJR7Zc1cp+nfkPVv1
YjWN/IsBQCjBPCtD/10Qt5HW2FQgv/fKS9Wszk5StUqf7Quxs5vmz5TAf6wLohPBeOvbIOVb+vHy
JB9S//tcrlGg96LiQv97+ZsIR27G2KkwpUXX4EG1IFnAoxze0HZNMTBfwIljDFcFkpS1f4Ia07jF
hh0UCDnFTJHN15EIOi2TlevjKSHKgrFRY+fF9ApOhTWfIqUGvUwA5Aj2+XMA3WHHTXXxpgu/bSO6
17hDPXU6p/Z9fj5AKB4fObmQ2v2Zm2CmrLWI9/mp3BaujRHMFSOZ6pX1YkO3Qv/wFp7ChiOxf05h
f4bv2YSZNFkSuL0Lj3n5fXxbo4dpGN72pj+SeS1dnXzGsk2Igb0iez6lR10YJpb2MUXWoag6tetK
TfIrq0nsC/QjlTDQL6snoycBm8uzuPEY11YgNYl4EirpmnVu4WWE0rNjgO7I3Mbu4mMRNTwO6DDH
+CcG37DbF124V427Ot9Z8WHSraKvj5xHyXQpsPz/Ftu8iXcOYctKnne/du2aWsLlSyVWmrR4kH9K
gFfp+ImoJrQr+/EhfkGni0e/9NKHDkNrwxHrc5/JX0DSipNdEWhf1o/tTibGVZGSGRSE6eJS6Yvu
aUgePJYcE4nrMbej9WVff8FFzMH9E1MGI54WS/mUFu2VFvWDvJ0ZpvRenFkBOIhNVvD2Nxbm06//
WeS31jKfNF3WvzMl++tlrtoItjyb/ljxzN8qdKHZ0hICByMOANlgagjHMGzyDIAbP5r35kgerTRq
GJ9XbevAMjF+2KAdCI0vyMlS/YNWbCEvNK6e36Slg03U98c6cBLkicevzbP7hAb6tGO8zbzW8tLT
+j8xc8WNpviOJHFYftTSIAQkPmd5C4mwQEeBgTNcVtcA+tecgO+Sx8Ss8ApAa8O376uZUsMXCj9B
a2IWX/96+6laeWJlK1aa0+KtWwca1nA6L9792UgNmNN0laccDfeDHV/+0n0mfFtJIYyd07LB+2nf
uQxy86S02IsmeeQmXi3cehGxhqkdc4RGjL8P2mNNZtvdM/GqVIf3YSpcD531JmXBmWB3mHOUc7ez
YqPOEzc+ck7Q8re8nII+XI1ujf6eKsZSeEYo2e8lQM34VxPsB59vXFO9Xhv4xlq4z2i28givnVYU
MJG7UyBI1FhUk79zzHjidxggPzTOkix5qU1Ido9/0QfjaSziF18YfzjC241cqkBcchyEvLzcbCcb
HmMpNtUOe6GLR4kjjG+1Tphfek9rVA33GIvjrEOMsfFGg3Lcj6Jf1JdB4mKQl27Otx672INXvN19
8EN9SR++nqq+zjjGtwR9WyXZ3OK5IHCl7gGvoYmKaqIO2uyylS5ApumiAke0EasPr774fB1u7YLs
CVpGxedkO50dTQDNTLfk2xGQ1v2XXIDGP9PR16J4T51dLg9kraIslKsWSlmpubhY2IXp5kYj3K07
nYfT2HqkUUOy48ub9PhTBfHMB3F72mu2WFcRq7XzYFXDZT7Zqzlv74VG3kRdhR4KJ2UUh5tVhjFO
BiLPbU/Tzz659st7kvmaGr4c0G47PD38pDPCu93Fvi/qenv1U57Vmp/ONEFGoJ6Q3Eh+J4oRQfeB
zeF4aqeE0JuuWfGnaHEnIypbkoDLfqzRYXu2VZzPmpGOyHC4alaG7CI2TatA/M6C3MOl8IC4+FCF
M9DR+sZfHzSuBrLwkLt4lVKsheOwEAx7OTX6Ave5foK6+qKgBNs3nR/ruKbXGXKnSPoCPu53S6jQ
8so9EcI/3Cs54Jv4Tgjd7DVo/PAyPpJx7uKJ5F7k5mfYYFqz+NM1UFTzI3+BBNK9FgPacFD21Tpe
vgMH+zgC/kTDBDYX7/ABnxk6q9rNod1mR/pdBAMFKq4h1rFfeK6ALgPFjWxUE4LD33os0f4mvuFu
uDLc/y93fbbRYF7gyjx4ZSoX/tLxqIX05ysavT8nbCqMyICSuR/VbFT28HTO8sfBl4SexsddjJdX
hL0gHM9rlNNOLLRSVDCFnBQbENnO5kNkYrJMgBJFo7eiP79x6TGT98lQWGX4j0mORa5nuwp5e17e
GgJ+U04ovDqVk+NGGe5cCLNqRWtr/j4hnJxkVHn8CJLEd1wOGnFruayFB/5nzEwD3xT2RuYf6+EA
3PtgQkOO7cwRTZZkNiPgrP2LoFiYI8khrX/fCgop5P+5jffvd/WGOMy/enMl60RYYhN3znfMrx8L
evraXeyL0LnfBrtXUk1/ih/LETXTIDQTeVzkudcXEmk9rqxE8gy+E7j4WrT0q49rTY7i0/pPwtuw
4EqdLu3t7D+VRX+aiAsJ4ANI9z7AMkFC+E9DKzV+5lAk2azGsmy6dOBgHaJen3MYi9OPchUZ+4W5
BHNiawMneQ1Fc+IwisXk6DLYQKSlUDweaKwLElrEwXI00NleXQNR7We4mK3Vi0TLVOXFxJ6q22dt
m08LG52/ftvpWHC9ZOIllbM9ddYZho9kjWnI35z1oKbx0wBdK/zx6euFb0+sZJlqptng6+XcmAvc
+F2PTwV2V6Tr7kaMHr7u3sM3/ZBz23yelf5LYHrtbqxirHLbFVT36OCU+o8o+WOl9pXQufr6uUb/
oUdS390QmbROpFmydBWFW2Mhytn75xE1GUjuIx1hkzBd67z+t2dBFzr+yWXHDyWw8Jo5/D25BN2b
QjxJ2QKjVe8klsPO60vxXkPieuUeODjz/jmTp4h9+7jj1IDMFVxHR2YeMpY8lNaWUpx5Rsg9rcPj
L29qU3nKQz6LNOZZ2gVRubQsPsfi89MnXqXH2WTdMLBH5S69q/PPbj9iPmJn7duZp0FCo/cOPmvR
iH/ib9qnxqZiz75ixyXypW3R46U8ykt+ZFhHXnSIo/gXxU8QEW5+SM3IdcsHRm6V4yyJCksXCBhA
UBSxayt3mih1kEf7O6nhKg1is+8t//a9Qk/NrxQ+VdvELzZ5Jq4BLTEawVG/9XYzzd8nMzy90nNF
KnIvtepkPCJTcWF05iFH05rigLiyn6Uge4J+e5GrOe2rtTr+FC9X0rc0vjaqDD9XpmP6IGeSrw2Q
pQoHZi3m/c0624DpSZOcqxDST2siHZak8zNPqvj/o4RMITQ1mckeS92Q63UfR0yCK1ovLE6eBuZm
PDPdSc72/2DV8Vfcj9h7i8ildEZFuTFQ5K3C5p/Y7bM42yFv+uujU+q3bDpWkd4ZsXAqB5dvAkcQ
GohdI2Ecg5gb69QYfEifYQdV5xpvpGYQLAozqBkevo0YN+5Wfkij+Ou02jWmaV0NkN+Q6+y7X+k1
Ey2zVajHpQZT7Oap9tKVRqunLubhnoafMo1cosCqAaJfmsO3m7ozQFRPoJNkoRSWO0sdixBbJeL+
mJQfJnaQ1KNjsljwc7IL+G/qFZB8aD+roxXopI2EuWqIrVbsc9NhSnwp9cq+Vnw4MZKfHy9fG40G
vw4iJsaIj03Ced43onaD4SIxoMC5bbA9jKABYBTeuy4P97HulWPZhiIOZphZcOX17l93T7HPumxk
J0Akxy7lpmFfleK56CD5V1YdjXxZl1GNr6D1v4Q8z4nKOrZfOAmLleW5QdkS9FMAZ8IZEGFk4nmt
seDj0uEweQmgSQO+dpBfrFpck0dIeo65l4JvmIkRsMkykwb8opySMrJsl3e0OQIDHLHRo66rBdSX
asb1q1dSw9/2ydcAfjtmD/OjfF/J+qlrx8z5IBs27usT8fHbXB4IlNzSUfssSOKrhRGRKs9yhBQm
P1NPMZu8CmILw7LtnL3u8VbHzhaOb1+NVGXGSRkhxuaqVZfvsmEeb85AO7nLlflKhXEJb0ov2MZy
zHGZQDWDeFnR+JrXyA0gcoJlBOX+B/bZLUH2QdMYNB1ab3VeR3pNnN2QV/Ygkr8iuTXF+lH7Lgn4
kd8cG3aPoq06qMiSVZAXeVhd4gBErBTa1ISJihCegt9Mj59kFA80+wuK41a3Cbyrqi/DgHgfu6wC
WOJS2RHyu/RuF81DjIfO25A8R9vwZkdW22f6iBljJQCZz7PHOucWHN4V9PR8E+a9zZe1w2TzKp6Y
Fbg5sjf9Y92UdmdPCAZGCCMt0QXTJvf5axb97jzdyJo3+3kiXJa8oXB0OsZg9JwftalR5njRv5Wj
b6yo/u/3UecuMsQqtcs5i+birBHFNP2/JxiwfLuwp+ZoOCiEbZaS8YEb+J0jUN/dR75KfN+mEyby
z48lzdVxUTHeDyMiPJICD1hKcEl5MLC7TG29WcC1zoH3OijINz97fxyJaCmobtR7zRf2kiL4CL0s
rH9bEehw9IUH63fdibnhmk0CWDKEfnO9e5TCLH3vDyTirHrugz6hqL6IU1T5qCDUkjyRSVrc6GQy
lmR9gQ+ThnnrKLMxBUUI5V9JkbNIEwABOgQofLcQJpLORe+4LWMZWtBQvPvvuJhvW1v7bkFX9ug7
eNiTMaDssbcfiuLXCfe/XIYHylzvXXhHT7euqRScqTvGcnTJXs44KLBSQQlKTC6QzPHCoj1h1eAJ
UcFEkUJed1t1YEJFoX6nD0uYmzivGSDUjeFv8Cesd508zsORIMjbR384tdVZhP18LJFLu5tb0vC4
d7pDBFHB7IgHk5Vjc48rja5uhZnUBllif0jeuHvyXZ7j7ZIt7ahooWe6PS0A/Qjvpzzulqs7JHMb
vDK8vhTwdAPl83MlSkiDEjc7afZS++D12HWiEL4JSJviC3nPWNvuec6EyfOrpNipZno2LgEWrvG9
w/lJi/Jmh3jlrL/QOenZBn/gl/Eh2ErFwF9hVSpIGH8ze2eK9/vrEiFzgsGFb4Z7qG09vL10DiAG
wOJ+ZPBr6sCenAwRTmvI4pUqPC2PUP5qbnKxurGdWnQBst/VeVEOsAHezsE+OwC1zt3V767bsuza
TeRu/PQuRh4vFWqGJbs4HOBBBdwO1ulc64MhSb7BcfW6lw2LQxDU5NCWfZcj4t6hWpJ4cK5dRalV
fC65ljRIen72DxoPpUA1y6i4my+NRSQo2QY1S3zVCHaIZzPbS9j35lUpUCAX48fbplXIWxWbDd6S
u7QC3wo7vdSAFlwnVDM5WdOOg7iCf1MCXYZdWhmiMCUn/nJfC1rMmaHUHNiaQuRCuuvQq+nnDt72
66VwU3JfBEZ8dX1P48ToVwYRCv7ZVd+etDFO+0AxRaYrHaRz+BN9LXj0Z02DnbGz/hjqGQgLERVP
NRnjz/udCb0OT3I7ks8N24HEGMvqy8a6lqmu7F3VSsT00M3h4a0GqQaZXoaYlAq30e/bz+XlDbAQ
Pj8CTCqbINX4wOoqXugCuMhLJoOXlD5ME+JamjXiLum/MCqqzLGavcBuKlf5BCW+vfAqx/IB3cBI
JJuJXZumuKVvMHSfn7rOS93a+tQqQJUtau7Hc93Nl9LmNaPQK/kycq/hRA6K2UEWeKUgzed1v5Bf
LXeOw37nHM+pv2ZcEjsC4HNcLxFrYnzGFjmLflV22oFBUTG2sIJ9/Oa4r5X/z7HOdVmOawlvMid5
/tlHMTBvg7cGG5HQxhcUFlNyP6RVGvvXC50RcayhmCB5soX/RNXZNDrGklYmAbY9Vo7jif8DTKE/
oFJhBgzq18gPBj469zHCKrPyEBxB5t757fPaHlJ6XJ9AehC+uuV7WS61qoh6lbqwD6VgB5vb5Xd5
Cw/kXkr4fiupoGgBk5CFWcNNb/7SJjTRoX2EHp+d++rQZC9AQNp329k92JZwsqvxGUltNzOvGZhd
3hNxhCPtMjs339g8Laohy/6/+xuPvKKPLbRxaoIRTI+yUt02aOs9AeTb5uoYA3+clNSnheMqMXPL
b0oln+mFCGSNFNXXTv8tnkgX0/5J/kMBt9sPnW8yfdaYCznApH1z/IMJEJcAuyFFQM3GYb/bzsF2
za4Te2IF4Ljlgv0CGYp9lNboB7DDPfAHEwdHqAcgETormd/C6D6wMTGEfBRCzOG9WaK0NMe5Xu2m
WqQlTOniYfWIyaV8LYdIuYFetDbce9EjpRPQ4isYWLZyyyHLaQQSStvqoxw6vxxaBNmsTF44Zr8i
IyPCNSJHcqDNmnDZN9lgv1QUm6YixcXlQ87v6n5oenVVXEusm8vN+wn8AB/s1KQXNimZ6d0MTjye
d6yo+HiCSEybdE5j7nXmH8qo8qrAVb7BSwNInWzP1sIfKB3PnwIFXOeogvW31Yh8ld18aNWQASG+
ONUrT3rSlnHWH+L8JNrIQhkbJ21jJn4cnUQcTINqnakyRB0eEWXtTVq+cLJoZWe8OVYs7Ot1n5IU
xBzZn4Se7d3PUCYL/CkW3suCpLXvKQSSZPNxs1c16nA/DTXk5t2W1XRLNH49+XipqItioL+drJDG
fKmSYBRxk0n5UaJpDBTrZi4CMqIbDaalVMryYOV28dLpXMfk0UbBBdWR6G6Q/NeRGDXgE638cg05
HxcEov2PqTfp1gdJGxV2fZpVilPdo8ufEluhDWbTaWOrhZWSbPg5wNUkJpt7VNr230gptmivcVls
FIgRpOpOa5G2/m9ZqnKFEoxgrHNeXxu5h2VPUMa9ybOyzdd1yfc0ZSvgo2TLHTtYPhsaUtgsbM5W
v/NgXYTyDSjSbf5CveDUnXmtSjP/64G+gMNYBQq3Z9E7OUtKGSBw9m9b0W/1nEUuF7xvacRjnits
sutMkGZXLmE1wkTyuT0ZtSJZxxFDGl9P3s+1EHmzxqbWgCHGuSzi4NvtjHQXCh/2GUUyjx+7s8vR
sHuZlQqU4DFkCCt+WjjzVeh7AM7wyDDYK+2Z979vcYdnv4K7sb/+H4rOOx4Kx43jbh933Nl77+wI
h2xHZGWTkayMiDuj7DPunIsyUtlS0rL3OmQlIyMzIxUO0bDj9/09fzzP6/P35/U8r/fnr2eTxQT/
Tep8mXlZK972QOS78mKO1LlmJuwTWw5jQWR62sxWx8o89K/w30OKa4X6XL9R0+W/SerTyjGvB/Zz
MnXyXP06hX4L1b9UH/DKfjN8Id4KP/R1OWH6peWU1vX0zwZSEinMx86WYPQTgGwFrIcIMuku3oNc
q7Pczozv43rw8EZAHYzbzueUIp8WUib7qU0zQzqpMrUnp+T3p4Rui0hR2U72OvaOkWWjvYGvgsNC
9HRZoYtbuiw9+f1Z971qEy97dg7d6hTi30NHK+9eNdDViD/DyfdpwKk2rZ86eWyv9MnfilbnaSZd
oWRdjYdx3WvnfKnEYBfPuNz70oHV5uFfAbD0oTrKj2jqWHTucOyUNLvOIEla+v3qLR+4TuvRyJ8+
y0OSGTgMMmiDptw9/YwXYPZ/ZxeGdr8Y0svsx4jG22lDqNXAK88Okvm8muUfH0gPH/h5BFUZkt2N
trukclmaisj4Q+TmPmCcEtV5D+kfSTKY2LYpbyjBUS4jlWTHBamV6UdNZINtoSkLC16ZnQHDmX/T
ZvxgIEYRFHqFfI8bCWCCLuYqDKEiGL7mKnN7TWw1wA8zWxxaqxIg1FZFBZ+dx0b26C9yakfEu9pX
ou+VygtHkkE00juH7oOZJ0Ex7A6o8YpYGB0i6FOEUzoSkZCokBWvX19sXS0JVFXpmizsjo+LOmEp
PqKPkwIDcGbQWAmK3pFWAR8Y5XxRbI8tesplVijd5QHTFBP2P6NK79f7KoSP2U/ghzNXbdqK8mm1
5+XLb5nhsdbI+K8s/bmN6mAgLhpkjpecna8X7S+w89JjnFPug9aft8bRV9H36ylms58NaNFzdx2M
EZZRjjO4lB1hTk9FqTVQ1Vn5XRf2BZ8EjGVJ+IeEKLAgevQfhgbvvqYAViwtRDBY9D0cp1WOmvg/
jRyvCQZUAWhZ7wS1Nt96qbxpxCmSda5O8UbISJOtOSYSPtwduzr2iCqfMERnyprBzcHyDixF+ADg
iVPzF7ceAolskpzmuJkNpX4W8fWXeiBx7sH+iDjw/QogXni/V5AeDGWhVhOfOKYjmFWGO5vm48O1
LeQfHYYcyR5NAPfnGJPgiU60NxLdMcT6DrrGq6/CqTnsyHIcQvaJfBEE9vTqa2gETP3IIwQd3eXz
0eYvWAooaJGzXr3+L0i9hZ3wFeBLYvL5iJNftxa2npU06mwWFe7ZF/KFaqlWlTcw46K+fOZ3rHJF
sAk+cjRBmFMFQrtow+eXKFLoLa2pk29OmdoR9CfPbdVuXkudQVzJEqmhY+QO/NhU5Q75eIP+4ePN
i/skD6/WT4cGurwRDV4vnnzVbLb8krDbrKO6o9Z6FQvPRWss6jQzznz+aJHWE33VQZheK+FJ1rsn
iPvsSR+nO5ZgsQKvbUcTRZsf16r+ES3dQ/yyeKxEm5rOLoWKFYEmtFOTxNMDjeCDG1UXLPaQnXHA
8O0CrZIL4rJPF2UM2+5rXpIvZSnY5r+x+8hoOZJxwbpe3Gh0dHTVKITHccxNTKp8nbeXRJ09GCd9
SBErxPv9FB+WJPrefXHzvMiebIyu+FqYh4bDbXviiRr92p8+oIyRWrKFflpANRktVvDzahCqqSNQ
TzEh/mnofIymb9guhn2rCZUoUnAYSrhJXacrWPr1R5R/70ygIFJyLjTAfwesOTDo8/hBQUXBgEBL
9baL9EyY9D4PsPQfI+/0Oz4n5qKbt4qy3R+ZA01e2YtzVs47+Oezum6Nmm/pJnPSddrVMZWk+Ems
qUjf5r0Q3QXBSOO4830kV+80v/7HtvDu5P0FhQwU4dr7Q2C3vHf7l/ylJW66OiG+O9MdarKSi1Lr
DXxfAwa/LBikCL5dtxoPq2lsmYzjEsirEvrwTV/22aHtkW3SgdxayJFMeQrqlRS9IsffFuEkrOF6
smroyp642JJYybgmLsTYb1fKbk3ZyEundsVFrW3mdm5mcOztqIuAnybLf0TNM90awxb9HBgn7JQG
jfhcgn5KH34TMmJz8+bSRAB/BnoGls/dEs5bEd/svZmaH58665tpHmEz66tuUil31N1Fgk5m5o0G
vnjKxtzgbR6iblxId3Uz75P6vmLKCLqAS9F1jd90uLhFAmWY2SC2JMnWw7dhHXjqA98uGp5N9PLH
/ZyBronW8WtOztoN04+1cwVCEpn/LIEhTSgmnp6rb4o7hBg5m51+ITJWw9N16jW/s5Xp7Hf7lNVl
pwbtUm65Lu/3nXDxSb243zO435fMlMTfwB3f6vHk2oT0B7sko39uE76lJzriJZtvPDSNtjxTK92p
+E7V+7sZT7cjKv98cy+x5EW67uYvjwwAHj9QSwcH3vNRNZCYkXtm+vkcnsaoF3HxsgRY5dkrAtJH
22BLbuvVTIKvu/GYWMIxEHGjJPVvDQZCKB3tQ8Mct+8t9WT+hjcx6/yLQO0UczVhlE1I49QNMMWQ
2WAKTX50uczpDZzrnmNqavwzJEf8s3h17PiPm1uA7lL1mBmNsN4SuqLSZVtFqNXJDwr70XCNUSo3
nVnhUnDVPv7ZN00xhKsEn4+z/VpBsgj2yfcJQNl3NoCdhEQVEnXt6QZcloBkBBZXFqm2tqDlcSUA
6uGNI7aHhEApYMhRvtZuoU7JIlpzZhi+9OBnr+GCn1hf2d2UG4PreI8Sbf60hW5LSeGc5ycPJbW1
H042Z2s1s0kMAkKOLnd7LJdIo/BIS/skmrhxuCpSmOenWveY7tz93ymZwYvcmisfJ4J/WnEsHzzx
BZGKlQ+TSClrmT+aj22v+v4e/L5RTwlpWgW33qiNp+u9voE79rzSYiofLi4/UyW+SC6Sv55+Sbb/
hpzlk3XyXh/t7brMOj6MTfwKE6amlM5oyxBK0llbxkeW3G1SPH6tUs/VhB3wKoEGtgsZ9Zns7fbP
iDyCm6yNcLWYjmg37XSo0FovbLXCjn4slBBbrI4zm/5doiUjLf6Gh3dGssFGQZvKp88Mmphh9GpF
b52N6AMdxUnJhzDSGemkXH7p78PvLCJFx10UG5p/gEIkkDStAJ7Az4+yK1+IREqfIWSZZzA/3V2Z
AG2clhbXZ7YtE2UofIo0Pl2Dc7NkrkypDygdiBLh2AzzO2K5lOH77ZuzYHYFA262offdIoIIYb+W
V8+7abDv+pWT4rIVx+73bKRzS+d3aRBB7E0mn2vvP8aoU3HIXuC4tFjpj/hjPDRpUMn1KUjtoqDm
D21oR3uzu6CBnhaQWX0sR70bj5yON9na2XJhLdNlBLxOsnTn8ojn0bj5R/f8GlMi9NXdcFh8NPRU
XtHpP5yJI5xAImwedYVy4lmLqaic5g+oR5ynkec4IdKS9GiroAvgwGJFT6tYdzy+mIfbavJWwKEx
SuBEHv+8edyPgji9Nm+JnMEANYm0Ph5o4YfEP/+a1FbpnAAVR7lGW+gWc2duUIX46Pf7zU5X12b5
n9Ge/+VvsjZEzMo1Q+ymj2PTgzqg/MlVGyL7GUrnS2Le2vqyW1tQGqzPq45HMKMHewE8r6AXKpA0
j19RRn8BXj3fxOB3wxhbZvrAmAYkw5OJLXXdTn963MT9K4EnH+TSXFNKjrMr6Qzfd+vL552Mrik1
QaYEjddo/Drylr+8Mj6QruLpjfm0k4o71zOXzQ+lxzSauE4OXuqjjqTHrzDehY7f3mZ5wH9eWILG
bEV+drt86eWllkYGBX1SwJWWx2OhYZgMJUnVAEUvV48LOn+cWsuqwMpEJcX10BngyccUVGpQBexY
8MuhI4+0MW0ao30mT66o2NUn4BjO958D8Fcf+6t1X9loPIfm0/sFTYXFnyqIHct5yd/bppXKNlx7
ALM5oLu7gRKOowxqA32R539f8jWdCazewgIyxl1ln0HP13O6D+be/YRcAvGNWIdE0IpCyFZCZF2+
GPoBjR76Xyr/HjGMYUhflKaVqPQj7F1cL+jZ7ZgtY5JKM64qnhHOmMMLZ5CIeNC8gZko3w6nreiO
pQGS7dOlwGw27TkRPR703ec6n9noQxO6A0Pj5rhuiWANs60Vr9+23tUhHR6/Pb5pwlSzkXdCVHBi
h3Jk/Br63nB9Z9/jEW8k68XHZrUqHwwMza7a2Gap6KgZoFt0SYZm1GDhj8JXbYOP1Ec9CpqaP7d0
JdFRV5O8FYm5OkNoOtPGLx0gaTcy4sofj7FahB9xHrY8qzbfGcRzKri4sbRcGGqC3J0dnCM2lsYy
hV0+QoZbMRBzntczfk9GqWRa8R8IHSyh0tcZcc6gxRlaj1bHFgewlfFeREfRKPdJpvTaRD0HIFWA
PsDq7/XgNKyXNHOk7HIS2I51XN0KnZ72Rphbl1wAEUH7HOqIzYDwTRR83t1OXaGyAvAyfvGhauaN
D8wjRpdp/TqeSmlc2PERVX7gX9w10u+t+NLsrlKdsQ24rjROi/Qr9qdTUpMgLkAAdtDNLO4vYoc/
hzfOq944mDB6hqKfK8g1C+m6N3mCbj+2RLgM6OWdeN4uuj2NL1BADdFyYF16tSxZfpTgQDn+KK+q
l3GM4wkrQ67TiQlGozIVmjf7mZhKJMfEJbotF2C/Los3CCBdGeMh6p6QRcEmuc5gE0O1qwPyxzaf
n7dS+d1hiCYvOuS1wudRTzw8jYRuqt968uIWOkSmOPAc7roikMnUFW5CWB4mdIXtT3M5Bi/cobgB
wnoVunq6NMbA63DqBrRlvC/4jLuLugkbJnwGhwqtzsv5EbHMn+JXG1xcGtyJ8d0RhCO0zaCWoLpe
bu15/Ep9PDWSaAbUgYC+uGEywWAtfYobOOAtxlfHSP+ykY4Z65SqDOwHkLKoWjVvAdMd39RBz0t9
jfjQEJiw2qPtdOwlVg9udC8+iTzop4MP6aWzxMeCIjd6zt7hlho+SC2LhxHdQHxg7lPvsKdQ6lkI
N9UCmvCDDjilsJRP3302FL/ajSPa0xdWm9gvLSHNKx+5pX7lqcQMnqowbTxjp96SYrPtrUA3oMRu
fpgpESgWRA2fhPBc/KNw2szESPgR8tT+lZtV0HVY51g46S7QlFNfrF+fy+GCE6Qp++e3ZX62qEzU
uQFfWcslD+wn6CWK+R2NkOsdRodHhmh2ek32RMkpYH92XGB5UTzZ/xpLvlnQRpBhSGrPmi+VpFZh
qoe9iV+ow3T04kRuMsQfyVlCZFjnJRN7UNXXUBpuq+1Bq9U4HUJUTb8e0krFsDb6pPrhGr01AHhf
zr2G3ASzl8rYXu8/qyklY2U5jOoYK8lu8x3UcsV6cuxCuArZNGM3dim3OKZJegZDlCoQ39PEN3lW
5QbAomNjC2L/uw1F4AwyW9FIkYE0WwZOUE9TMuztoip+Q56dlwv/o862/QpEwHWkZ6fZTyR76lXt
C/rLNvOimNIt8ylv4fx596pC4ZmXGt0nwR5L/DUkJ64NmPa053y8RZPG8t/qMvuu91VaAJjndpzB
jYHBYUW8KtlZCDMbI4ngnHNoUcGziJAh39gw5Q36kzMxJeWNO9nHrsdy+O/xkQ9wy9xGFU+hy/aK
cih11yp9+6fzC/W6yooqfNOpzNVvimvrI7MA9FwlVfUUuZAPSvJ7GoOJTopt8lV47CBdN4u7QlXF
3AhnFQ6c2OZ9ouf4rmlI+jMplg2o0pN4d1exHysllnA7IBD3JbTqbmoCGeiycMbwhRnZqNslC5gY
XC6/j7jBRXzoOBnTo/FqQncN67IVuzQn7y/xnJn899auAcmdd8rpL1u/sWyyRUYPv2PlDc+Yno3D
KTE2syQ3HLXB+vNtAw18a3Dk7gV7nSXlxAHs9bo9Q9KfyaBXbscNcLse7TsEs/GmFWuuzUq0mP4C
dL4r0xzus6H07vC1/8sWfYAcJniH7nZEDRtbTUOxiB4OOX+73IlNH/nYBCBENf4dgxLIwZdbpKXS
7y5iDxarCrKGB/C9MqQQtpfbqOlAoJ8AoRuodfnx65aGN5fuaonSrhtpvzacHBGGLIogFcmXp02v
J44ciZami4AfVx7g15N5nTdfxo+E8KqxKtGj7F4mbjOpyq1vDr4WsuUJQ/c4xwdAL02qSn+SnccP
mo8NZP+xrGSWrBdhtqu3YtQzCLYb7hBiJV9g0zW0oeNR1/iTltbGWcNaKtBMl/C7fMEEZbVxCH3j
5U+1p8B3RSz2WpN3GnVW0PTmAy/d6YdF6lwerzoihoDXawGyTKkPTFlh5V2mnKp+vC18j6oS7sQ4
Id948IQr1kK6MXej+0xTlmLYhPTgH5D8gfUD+4+fRAm4jRn2QXeqXtvTEz0U8mYy7AjXqQ2wOBdT
uN0r7jG5qx5vn1eOTDxGhR5MbVY2/JEepZ4za2r3+cj6VI73XMyeLqitZg6KiYo3tjPKSyB+dzy1
Q6YsmLQ2egvjXzjiGHdgLxL6AnI47d4afqkHvmjNJguoRD8si1RSazDGd9zyvpH19t7Te3ejfdSN
2RDayp873hUUH6IVPERfOkdnrJoYRCFDL0DnXRonBgpoRW69MUtvegMODoMu9EqAH328TxtSoPNw
FC8SJbocPPvJTD3H+VLfuiXt5w0Mnhbf9O60h+5fpFMH2XPEnfPBo8Fy3TGaEE/OeSpdfDQbupnE
VTJzSyrzkLXzWWEkqh8KkbfzkXuoGyJ9pJS3CKWLiGiPRir220YbqhXvT9CqMneVcXsx/moDJkb4
lzpFtJJMrQtblhkawx4D1VwKEst+t/TT5WxZSOpCYnteo64hXWZoJsIAALUlpmddoRSJJuovEP+I
8C0Xdg2Oug0o2L9TXVPvAu28101YTUlsKnMTXOz6cBBUVeF2Y+/gEtXfe283JFhwmiLkq81ne+At
GA8gX7gz2A1ffZ0xPNq7/eB9Z77jzjqLDvc76wcSwYz5CUix06B5I5YC19AepJjY+fnJuaFpZVqc
mFopy5iW7QmH9Y0yqLjAN+he3Esd4oCsohZ9BDL+P1Adm1nuoFaM/QTbx3OVCTqHFSlbFbEI2jRN
2qP/oQ0LKK4zgOtp0cN8wTOAbx8aM37PULv4gSDqAC+E2sWoP3Q7mVX+4pjLzyJeSAkM0F3N2bPT
JGqQhSFsMj4l9IAzeXSFyxXdxy4sN/QagOLOWKoXN6nLUIjN5yGlAaLd/I79nVkmSBggTmkXcfUf
f5fSmBKaugKhgAE94bLWl/aYdAaRNWPn9mUXwydDjUhfpEdwOi/mUrDu6hFqsLAF4lJveCjmq7o1
CHcrTfx8/vsmS/4lwZzUSUxKj9+eaiiT6t9Rq3NRWI0NX7WBpuv6IRf3zM0WKf09pLTrlscP0RYs
pk0ocDlObWAI4odvRzWy+mCY8fW0BbMg13n13/jXmrrBdvbwJkBTIk5ucU66+2Ny+tKViEC4w9kS
vxrPl1wi3e453BS/xwzij+nixjhgW1cKzNqmahF50gMhEZdTzondt/80+gN4xGBdq5Audnj389t/
3y4SOYmWMf1Anobbwqyeght970NYtkU6ymLasMQxWWta9vNu9i7DFv2yf0tq/9p0KffpulqGZIEJ
q1zUTQR8DImGzLS3X6A4ADwpd4FuRGD0SVfoCdEUQHu2CDIAP291QGYGD3izWkO4oYgsEAYGZ98K
Onh6QrGmDzNnY6XHGxnoM4dpM8Uj7pwg1Blou0lIIbD+DjOVQwWpbLx18GX96H5C0AKcOSGGYab5
u2dEs8joWqt6Q+El9WOBsdkI8Qbh3YPO7S/hNQpCFKslijEg4Vg5XL4vLyiOt2uTQZe2JUZ/LBpY
FQNtbZY6EBOnWz92nupEnNH6iWhJ91E53mqxoq4YxnGpXte7R+ES2pIyFgBZg18XyvrM5wHNh1ls
DEdkYKPe6dtXHuoCgD7pL5ijUwNLHc6Ekx1ILqcWcQbMnC7CY2SgEzzJKcFRxaxnhn/LTHG6v/3p
DUigT29B+gQCfDWfERjoKJCGJJyAwhFglrk+zrkIhNESrQnu5NytzJq7+AVrs0NRojHsgUzvBi5G
7EyZcBVGdWYiHgrH6boBBgjt650la0IvTDqSu6BrbIOILrpOTkzmTwiSGwrTNckCwTsz8y9gOg47
o5LaGDCdGe1N8beqjJau1sBGN43yTnqedT9zH7SV+TzO1H2ikDeyVUjwAkBqHS8kR4UFW79LEghE
wdoEgv4deAmkC9zuOwQP3Dp0LzqESn5B/YQk/Ejay77nd+cdvrZxyNSHBEcr63EFpRN/pdv47REG
fJGfmhPJLO7c3J+j9AuzucdudUnAok6cxN3aDALHYT6k9DazEesLc4HWimlzYQSV0QMz/xVgkoG+
AXd5h6W2D4mcVberr2gQ3NCtL8ZPS7xKNhBLb1hpCAebkGCCVnI1APjyv9mh67Vj/XUkAsdALdLH
AP3AFrsyxoB/doVVV9vlfeUuteJW7650/zU856U7ioCRMNuKZYVeXT9jxX4DPgDdIfxKxNM+CqGN
CNfjo/DKadclNWfS7rLOAxsfd9X0wnP4Km4kLamo9mKOiUk8poTJ2xwNxc2+vUMeRCQHdpGedeLO
Bn8tqqwdLre+ixYWeb1Ykf+YO1Lxad4d/YCUZDmDd21LocIEVPaPz+DRqO6LfbdGURsrVUlmI8nK
WE6zqqQwhimGBgJgcL7wzgo/lUsvjAnfz+cdOY/8y12nMf/InwsRIkb6jhntTcWCW2rf17AHrpiU
96Nrpd4Q9iIRpbpesHGyofxX8f4/d7KzlRQ9vcLvKEPX0s1MEWMhLQ+ykgbucv5nIdeIabMPKRLe
LbT2G2nDHhyL7B9U6hssG9Ane+kiByIKwwY02RxZWsRnfHA+UnJEtDi9Bs/K9VYR8FRFb1BJhJ4l
3EFWwPtVMqdgBBduwzlyvuGQEMyuqGmpbjbgkKl9NV+YQ+6arvkipfL3n48NBk+JQWDQzUCJe73G
xn/RKgNXQCxKr1Kt9YBx5KZr+pqA3ZgBTvnq4s9+1IW4GYs2Tnlr5j8Cpp3LFCKK8Wody0PE86w7
1MnZJLjgmOD9q+rzVV8UkOajeLEN3SIV0+qga2EKD37ZL9+vDGJE74yuLQt4xslMvnEKUDfEnOMn
jCZdT8pl1cEYfD8bAqYOqL7kN5bRoFYCr1P9UnrROstYMwYPV/4zDcSywXC/HkkBXpjFSR8RjoQj
UEnkVJHpdb6LzsZlTGPOJ/NeC5/6TDBVDVPr7PK2PGZ8d3ciVU7dqjFmaL6BdA2DmMjRWRpXImoM
vqEVrQNwlJGK6vUldUPV1YgxYatpa+HcZLO7mL47wpF9EXAG6V02GyG+l3drxtts2DJ7736aNWGf
yn7xadS5eL58Qv9cLfyq2dg0086bNLKs3uDcBfEuf04L/PwK7SuosdPUcIK9m6U5y3G/Q9UR9VcE
OmSW6m4PigWM0Sq9wzRtwK41FtPE9JD30oHq78tB7a9FXXp/RLCEYnwKkoJeLDj/B0m5oyRFfMVP
TdFGPJgOjo1LmUfmB0ItNSLte3kRWvHQpSw348nOZ0SVpPNr8ARj9eDQfWVDkYrmqHSyP8uFPtKo
tQKXBhDEJ/GJpcc/4S2dyDRI15Dk/PIWV9+LBr8MkUEiAbGZW0YWrtf9CX+2rFJiHv9160p9CYL7
ok1TPZGjsTC2JNHFwwJwbnA9Fi+ErONwvlfqjgxVy7S0BaGyka9MBRPvDzYH4Tbhz3dSEesefUN6
I2Pbatc2gfFDQh4U+77g/ChhA2ofRR/oa9gFnA/y8K8YpxRmDJkqiIEQQMwJ9h+0xF7VNmg99pRw
7yro06DyWoL6+MeQpYdv8bobq47crkR9W7q4aPqdebcE/aijdglEs3QN/z36iFv18KFG9JtLDnHv
1rUbqEHrUopMHk+fdiRd9Urtg4q+nkv+5bB0u8VAU0N/bNB+xuHg2x5DgDtUGpkhM8QEU/IAj245
Wf9BU6VucFXFdI0fWV0y+qSaImvYSiclcS4r9PvBuz0exbeMikbqyV33/Er61ZnNiTc/w8LVnngm
Wbvez3HEAvo/YvHCl2PSHwDMm833KJne2ZQCP07KdUiNuNgFqCW0N/S/6mqbmZxvLPAjPrfV6WJv
Uu4UlayjRLvkRHKvrW15FedD816HrW2tVTzTqmr3M/kx8LOwjhvLhxVSaH8/X//qQ3rKw7wUWd1L
XdfMH4RbQcuaxztb9sEtd7hz5SgpAYNBkh2Ccve9GHPVhOOv/OjYtoTDoXBORuTnmdlZNJQtlCs9
xXICy1eRmddB1UsSKE6bpzcc47++NXECRLyrFS7kLiwg8L2+cF1WiN8XLLdb5/zJE7o/a9qZC5Xu
AcxuFxqCud0nj/gJ0YVXISkmKgM/Evf97ItzEyQ8b6rdRwzdx06+05mH+PXL3OWTG9/rZdk2W8H0
hwOZN8XE7dIXf1OylVyjvxUpOzRby9veB5w4IhuogCizWd9C6LCEQCNjgSzZ9xPyShtjYfrH8pTn
sWPFYy2MHP2PxQLkNFQmcwN5qJkDHMFf9SFXtX4quQrDLiopYTmiLprcwGZbmKcirykw3sBiIb6Z
kklM2Npe3tbnyeRUci5yDDo8vTz+xbLA0WNJgX8uQkhIR5SC1KY0X+5r+Sq3aAwEARIYdfeNtPTK
GmZ47EpyTrw3EFNHuGh1kgWSpGKowoBDGp1qWr7/emOektzp+UvOouLlemzk73EkQVZloSCQC26h
Mv4mnuNnPoUiC67eLB15LnSfnPyq/dP0cLiDEu61N0mbcPyvgTfEocADkuz1NsSfN543Xn2FcDjG
TOjPG4de4WPsujy2tDB6rxXcy5GHcRhOHiJPX3/GUdv4mJ9lixXI9XFCJvkpry1wnUnmjr7qYWIE
Z/jlhJPTSxD6GFh83GdesmEHAXH579z+Fi8Hxf3pUllBdR+H/I9C2VOPKSsvv395QZUmuCtemv1B
CqUFtXBBfrJj617hbI7uiHc4DjzgiyZ/m5GE+5yccO1ZQI/x1rV1JWBxQsuG5rLHsgAoQpm9c1Yz
4Nz4W0doOD/FeImGSPa5Z7m8Idd2OYdMYS4lzxQGGlaRPpgN8smQ6x2aesKhd+jFUP1Jv2M42jDt
KoiUo54unxNCF1JWjHYjf0wEOXUtATtxhegDsxLeZKJ85uEjmk3P/+tg5vjjBVC8eew+MehmEhp/
J9BANuzJFO7dtjftqeSfPfAjRLeJgV02I5Miq3rJ3sivD/nnjfUeo83nmoVPsl9EpXM8LcvPrf5E
ynGhtvIm65WT6zHPTOWgSgxVZXrsMqoaUF/XZ1mZ2LxXcOGdvVAhp700SwOLF50HcYeQPhgzPmaZ
kPg5En0lu13vJNmZ0A2S/vVpKp/Euvyq+NqvSfxlrvHy571bd6VzZDnWz0QUIICZjQoUVYyv9YF3
b89H47SXHoFgrQ3kp8r5J1203JG0fyZ8kengy8MfVChMHs+kCqtb0Xk3AeYVzMfn7y0UiNbbK6sx
fqN5bd8nzLO4SePiKIWGOPUSYUm7Ta7u208nmt7WT7/V9hAPsobrn53n+RbTtecYOznRT7MgUiyM
NxebLkb22fu5MdSZ0HzT8wPNjh2LWZqu5aPLFCLjlztYQsrOj/RE+9LfV4Vnmj+iFuo9PIx1CRkf
Hb0oDN3RNPdHF7StKhzL+vXpzw/kkTr2I4enekJkCvQaFPQvR3y/JvkgB7sVeGXTIV0t8OY2FP1p
ICzAHJc7e3v5/jzuWrnic/4A1V9XEo700JUzTmtCLu+CkVWFM07Ko9O5DGavQ65KvqhU4yU54tg0
A/7ykRujbADDtT6wW1KxY40s5U5vl2vbkfD+5Ki3tYmuoUf4FzVXBB5nHknZJByd1DvVRso15NiF
Kz96zb3rzkpYF6YxZvR4umsGGPT+7pYgQxJAB6UNz8z3Azsn6HLQUzUDJk0LZh1UIQst/hRhsEQn
8MLsScyALrDWnT/hchA6j7FFLll0wGwQlGVC1mJOp8rOGEm7DVg5LgtJhoO8gI/iVO01Kg9pg78p
bI44ZJfEuq+i9ACga/BgjBiSgJgT7sZyYpYMZaTa6NiMVK+8EK2bu6FWd86qZUZQsTAbf+sYo2/h
+Oh8ek7OKNf0N6uBJdnE3+yxxE5+SF0uoTYLKzIPI3EbshrrhjIayxyp3AL2qpxrwf+mUvj6mAib
LDR6XSGjXA71mZ9rBkyYzzs5DKRxpa+XJqwfPL47V2xzAh69513aH875KJLgzhBh9CFF7IAv2xcB
PY6Ycgpu/iOiFp3Srh0m25b1ZePyU02724SPLDM7HBNTHkwHiTttcBFUqp+UM6LPl5lhgNOElvGz
rAuAmJ4aj2BeXrZSZ/GzPRQ2TNQ85bqEWG4nGOrl9jhfvqY4IlXHQmGYA23e+ieBh26rfZ31F+Oi
8UzKRw78VtT/CNSLibz5EXkxxK9WLAevHBitTfpueIWVEtLDu5JOUMRuWGrs+FCUcqjauzk9ybOi
F0yyltlmt3jJ7DFJRwWNOWtC+eIhoPBJ+fmHj/gm5SflESaW8pPhBjMP16+cd45JIMyJsvRg8wP+
ZV6he0LvBs0fvW7ReKlI3wQiF/cwTNG/kzthsR89/RZbNVfjXZTIbMrk6PuTXoX1tQ1tXXhQjG/i
su2Eb/uAzFSNV5dWDYmOJZ5u+qYn+pSDJYGctiFkqRQarxLBH31f0VnoyktdbhSmzAlB23qZTrS4
rY5651INaCNsBUg9gbMT1c4i7lLA16jL1pls/0ZI9pxvVSCVKu3B+w1ZbxRRcFlBMD0rw1rsByk6
aSMhjio043qijFDr47RvBhV2XFwun2Q/1rCxodFI5DqCVOw9lDd4unUgFZde0nRdlszeJjczf8JY
gbqkWPI6g+tBe21tFYtYlFRxZK2HgSMqLnx24Mu9eFJcOMnGPlQhMnkvQoilni7saozCVWsRLM9o
snDLy1d9AKps/006M3pgCZ0W82134cPtRS6GQyzF4edd1BcFs0uJcn28UJCBKRD3UuoKJfZXt7tq
P+pQ5hEWQGk9P9gqMVwgHkuPtn6dk2dzgfVLEBFgB5fgTJCbI5BogO8pdnmYCEKxxp6Bm2oxuBvJ
bnFns5kKA94RuORBPB+NwTgh2GxyhnrOC6P9gu4aU8AqM2dXmVBkTkrEL9Zp9RnJ0rurCzXNS1zT
qhIvBZnawPmS4a++LSDEBRCONPjUSWArQG2B+kXnyw34zozgDUHQC8KKIF4oCs/cF/zRy5dknDbG
cKFRjeI1n5tsA9/5lQXUhNDixf1jmD+z3keO5TZ1xVLuulr/e/jjonFVEar/xnRBPsx4I+fwmcN8
iYynwTk3KxNQKufDRyIIGM5gD3z/beVS9tY9464PxZDihRS1PNLAI5Br/SrHXer3ssYrjxl8/Yrz
LtOLz/I4DxYDMoBC4A2LwagrXBwNb1aXLp3PGcO/IlgY2sZMWWB4Ll4GI3AtZpcJBqsyzCzZpJd8
KMK3CItKfrjQCGk4XwqPYZP+oInfJQni4Nxd106lm+SvjIGd321i1xVPqoRGGfHrG0z5YdtffYKH
Rm6+/ljhGXwyZAox66/bs2PuUkDYMW5KQvwB3WTuSCZqsHFaP+APYO7DQOjEA/pwA4gBxfKo4ub4
9grXoPRuw8ohF13wmLbnrvSc4t+R+o3lQ5Om010H6S8kpf7iFmO6Ic1fq6ifkSaCI/dUViI/SWNi
dqvLCKW4ZwBrtkynkcPEObbape2ZYe7b5YgHFL3qgHsbTbvfjJzA0oe2ocQTd0WuRz4cfbgLB1/b
ogOt3DUte7QJfVzJyQmGRNOfGkFcM4hMM0pHbzcuyZ7KrUdd+YiSOZNbRD3sxhygssS+ltlaSaE/
w1v0FjVnKK+JLywBQaSrjDs/VYF3unl550GZDPdWjoRB+EvgviRGEOjR0o+l66n/ShvUsJbLkYz4
sOeMtln/zt3/7MWWoJ8NsQMA0kuY/rDyoZilfvkhxc60wBQbmAETUG7AFQszswF9X7rgKEN3pSf2
5xX6hAzBVamjUubteGyR0XoDItha7bG0XbYg9esdV4N7J8nr1wGkUerPIMtzjCBD7Q3huhLEU2NA
lmbzQagcWT3FCmEzgQSVF6NGLUexJ2tQ7PJM5ifFE7vaemGFM9JFJo6j1zM9H1ynScry0zqpHX+o
fKtU5a/iHjCK/XkD+s1kSW3AiHSR17OAJhud8q5uTUdRBT1gavbKh00niLPRfsWjmIWmdhSWrCFP
+qzYn7w2eLcki4nQd05lqBk05f5qS/pA+kX4el+pVvbrXFTnaMDwncg3NROsjQqAoxkID34D1e/1
MzvT9CeTkfgjOYKO5aIrpEe4Y4pP4Guk4vULk3ZrA/zHM0Rfp27q2kCSeHB0PnUI4LP2kZPIIZ3G
L9Gy0jO9MiYoPUJSbDo2zPr27rbyxy64P80T80NmeRyRhuORK0/hWxLiX71KFURqOB9bvOn83UdS
O2X5L+xWefYc2b7T6oyFNFm7r8k045GZNCeAE5k5FLiY6pzjDeibJ5k2BWjSABN9pJuuvfD4o2rZ
Dw9Gv0lRHU+iuEmozm95oxycxmb8pYq7AtjUN5aJHjIyarkq3/70PD4MVEB955H4JnRN604WDCs9
u0yNtfJxOjYfwxn9wFJ1EclSrx/pAbBvAyZaJza4v4tpvuV1digLHOMech8D0PPIGlnMYC9YNYzq
R6I2OrnG6HjnGkJhYCfIS9flgAOH0EQBo2/fNsBTmn/oTdWM3z/rwQj/mn2zcd3YK+k6wyz2Zd17
Jg2KxaD5ulLLzkYXxD2/G5v8tT4Emt8F5xNgnQ2FEICAzmbjweX9VxYNREtIPyiuOlItCSXoHhFM
efTte9oe0w6rs6f4fIBlg4G+M7Bdl94NWr3WcEY6S0WD8kVOF5JePB5biwx65y3KJRfU3V8t+Opy
2+xQnSW85AcjlUbtbZB1mKkZpiazR84/3BxmBSk8CzBku11uPpy2YOhj3XzwYz4qzLx91oYvQ2yM
P165uWcFlqXIYMgTxFuCDns68WbyFjtaE/EUJDTKnvtF6ghLmrD49wfpjDyVZ4/E/lPbNAvVd/Lr
h2PIqTA9BkcOxxYoEAichKblyBj6HtTfQIzqizptAE0vWsou1qroG2/8uieYoNJ0tt5Kmv3eI/gm
pf/iDDKLvetQSiQkBJXq3zXFYO1y/ea89ZskyuVYxp6fWtRXjOussHjtOyp0uS1wgjPoaUX7x4tv
bnCvnjJ0ynJdcLDed6XrVi4F/TreuO9+74/2l42iObjB1E9TaAqPK5rVU5FBOSSfImrwOSU9FQNF
wYy3N7VFhMXamuv9LJzpfKtNKp9jdzPNXi9EPMCCHQ6TnscLn1ZXvAWsZQKcQlPkpOrWzs2R3Jdt
+W1DGafs9JK753/nIIdVFgjUTyMIlfHwF9+/fl5e3V29JfCkAiE8UAuDMcJWhDgqujaZ53QBGoHe
A9XftpCwjsc4v4RNzXbXH/Y5id+FctYWgSxj7F7ULw+YT8e8ac6DDzw16U1XQIx18J0znC/C9/Od
byYqjoUQfP+kTQYTOW3NNbbPIDbXfAmD+0FbLER3JCL0O5LnJ1W3GDWBbLc9c7KVA5PDSNgwyrOV
ywzSIpmhZ/TbKi7ZdvrcnqLsTcloTjURMCeeKVXP2yPMum7g07oqRF6xY7ihC3A49lJPLxYkfE2m
i/z9s+eLrffM/zFISpJe/YNT9Y0NPinZI67g5X8QuIho5hHDA3Tt1iXenXhNT+rBOHiLRbHyCydf
R8oAGZmjGWYluBiiNKClGMCmy5QpDASB1ugwYAjmPwGFwTP/U/QM3PdMmBCEHbQsMhO0RcfIxqRb
jWB07UFzOpbmy+2BuTfMzoJA/5ijNpuT92kcrYpDEP2SztZuHjJbspjSuAUHnjtTePlg7EJGE3gz
ZKxt6dMFBki13vD4E3arXqIsdKrjpulWONhE2DI8DjTfh45nPZhTOk9QjxfF+d+E1Fo+aQEoRSr1
hb8+2e8zAAHlAJo+BA18OQQRnuD6HSAk7/lMs1UcAtadlwWRDQMoTBegVmMX+SH0dznIAT8Uue+s
AmgZwQpc3hzA66P95mNvZBRhBDxbDJ868eO1J3Q7Bes+djAG4ah6MQXjM/Y8wtERmEo/oDr4ceDk
pId+5kzv4QME5gx8fh50Y4c/RogDasA52FCVJSZmxCmlX8g3K2UsK8tV4gHXz3/5QAOjK8W3w2Fc
su9log3D9+9onQ+KroEOCD92YiQWx0IUnyJLzvGIc1Z9Zub7ptMhcmWMW7SkIsbE7i6Rzuq5Brob
TlHnjQfmQjDaT4Qr6l239ECPOwDKNGyEa/qL+mbvETwguIkSvHvJvzdj96TKNocnO9fxWTFQi/ah
dfeW05/nAv2mu57BSgnnVeyTQe4O6Z1QadQGL7dC17ulEMJPNAXMPEhhGw4DAvRUD2gDgO63gF0Y
Cvz/fwCQ/+z+b+jRMyD+61AkI1M0/WduFKsUnKLCUI7o90+G/szMoiGmq8+X6JgBVMCIJf5v0514
WPzBCrRgy/yUuom6dbznf/Bd+ggpfXZTdK5gO1knkmSPTrakXrgW4uAVUzHjZ6sJ/bNqaJRZEl16
vAD9vGeiDlNnsCauv9n6LQzx2320jmfBHIa7bSghxr4wzLP00xiy7oRXhaRTJP+Ls7sfzv+4LIyx
IA83cbCD3E+SlXJBWXPrXOHn1WNvJlHtXZcIB1O3HvnzX8JHwJcxH7ZCy0IEid2ntFbRq4JEA+N4
pgbLgz2JgJGYmF34nZt3c09QqO8yHgd2fxqeZh58IXxGCZ0J3F4ab2LNymzyqWALKpQ+JKN6Y8iU
1YnVrLS2iXX67ubvi63oCe8ZJtKvic3vbkEBb/L5+nC33k4HbKcuBYlvqRz8wPyKQFns9mpK6B02
cD6CW3r1WqZwGaWwXc64Ym5wD6W+i89XZzb/3f1W02atnT5T7U84Z/LQ1LPDz26SjIwIl0PpTMk/
5mG/fsv+WdsN/fShGY2sc7aSXKufDcSsbd2yM40qNMtzQKz/WRNvUvNaf8WE2PFi+bLLh/izqzLq
G0WP2Dvd3Tvd4wX4Zp4qVn/Wt+Jn5un7LSddejNgmesgRYNhzGmBhtfGTBsw6sn8/f36gJ7NfEER
HLCD4yXzKsjkhez4rMuRX9qYgJ90Gv3rm3VUJGjGDIcLcg/Zb4aXipHeSonNMjmJk2w623VdrXYF
GfA3MPIREAx+YTi35X7ozrC39N4xEflD46LCSQkr99/APx631zLMASRr/uRvc/SMogjcp9ez6w15
99jrtWEbvcg64KSFr4z/B753KrKyBpN+JNMZxKgc7vt30md3+qub2qzDQTsBfrAZI1GENXtHOSTi
4OBa8MKUZvrAf7t8CRYnjKEPZ5AubeZFSe/VQ3VfloehfiudCcpwpzGNRD9VGgpdnV89ZEj9HgAV
tH4uqsbcW/nTKsElIbD10WpP0RtUa6ZhM17Xlq36j1uJAuNmBhzUWK7QQiFoL9laIdDnnwfCG4IJ
JpyKbFeYvBo0y9Ky+GDive6qFz9dX9F5dlEBPDv8M5COSZfO91lyRJUVP3oOsm6TZsMQvBPIH814
lcHIhdo0e1mYsTf+2l+Vrdd0TCP7/N03ZqzFdnqn39tLqj/UUYfTFcrhq7Tl9mE4wdJXZ7yFTHGs
/nB7StL7TitGNU4FH6IwnParQj1AQfcc7NOSVESS+sRfmYE0mWXwBsolMCkq1VwSwIHOV59FIvQa
vPWkWK9zI33vp/OwOigbzaUWzYsnwYMjxf0E/OkAyhMxrNf0GKY4LuDOn/b+lbjJLeaUQD199TPh
KWTc6mTwlfKwaud37VXuWj97fy2+mT4FiwqK96FLhJuMf+5ll08NPvvp0qIfZ5G8so9IaMODZY3p
L+48p3L7aq2XaoWm/CF89tzYjzOB039cbhkbLRpIqTor3oUdJ1otiLjZg7pXBWAt9nq35v1paA6I
JdxvXlOjBVWHGY+gZwUJms2sU1RBhEHkwBFa5unilpfVgfyvytTiB/c9a2arjCr7EgHrz9vArNue
SODJ/a40F8p1gIsEe1X05UCnC9bmYsg2we8uCR7nk1pkw+Q1Gedn+7ReLXHTBSmH1nrg35NqZRsD
YwDPEq7RxgutZSkCrNHBQbU08eflRg9p4Jo4GrCmkwMENj7VWXC2VoR3CNGPULBFWtwRbOetwXQB
aUkg4yPh7vhjbaJRcTjr2XzDbNKgJ6EL2HS/Ot67WMcV6AH4VWy2qvjPZ5F3ThHv1giOdLSrWpr/
+WWMdDVnTQqlHx9TDMuviuC+SPnpMBnodko6Axid3X7DvH1gnsYph+aV/Moee53uyzAoErb17Dxs
KVHQU4jSWqpKOqbGH4XhXSBPMy7rsgHF6J07b3IkOaEJb/YfX7OiGLmCYJ2jJoikZ27CSFATs3KX
kdbdyJINFS28uxmdxF3GlbMdI8uVI2c8inxXilc3Yk2ijFW5DyDjGoozODAWI31NjoI0tZ68gbHI
kcaatXQfHxwBA7q8UOzOjkC4dBtn+bHgyHZDvUPdLOt4h2fApN++q25qHcmQMhk41g5JXrpUFzv1
kGVoeq5wKg2VYGez4cBeqS+aX6k73WRNRWJ7QmB3wA3MlXPwbtznsHNfTuwhd1+5ATGTjt4CROmO
qTPAHK7Bts9jCF2Ixx1KB2cU038Y83x7xLU/1+LyYvRI41d83u5TQXAzh60TvFiLl+2AvQ8qk6zb
WwNguoKZp966kd93kQtDK/AsObH3z1cXPNs3yssDONxwCfTKlDLlyzPjvt01gkbBpHsf3ys7yEzo
N2enJITDZubXI3rGr1KualRGs//Kx9xv3Xkqna9LihlbmD+qI++VywmsALXsfxto+U7kaZQk30Nw
NIdrQulmFJvR+3UcfMuC2YKuLlB9bQxbEHG1Jx0ukWmx1jr5NeVNOqKw7NE67OJJ8FtPABc79yBh
pYJA9foW4R6Aq9NV9rnRhxTCsYPcFkUK68T35mMLAjymF0ZQmgAROI4bmZ0mVB376UOD7CW7MVtF
NF70ckb5cD+KZ/899KCdWZQr+KzekXY5JThoHjt+qAA64KZEQ3r3z7GaSQz8lvGrvYRuRUw/cT7i
1gZnxDffXhIyLQL9XQg3p1hP5CGjmtj4trI+GtK32cTrsKTBGn02XClEoLH+fBsJSZuIElYE8rvf
PDnRazZsFH8eTr1i8V8ctVF4Thg9D5zuUjNKtz51T2Om6I/t2VOxDbAcSuTYl45p4LCSA5qxPuAz
w4Jx0Dz7LvUiRxtaBNrGvlgfCn4vgnoHawz5WrKtNIDRkNRzBvm9iDwbins7dwMFf6CpR8ha/wDt
uPgM9/D4fvm5igtYZKHIxMrdUR1yTcA9whEfRckpXyYlG/krONbzMdYI31Z7gkljzUDHRVp2/l62
DXyQ5HVeyrCwkDAaLREtYlRNyaVb6X+lKUw905sMZzFG9mHPlXe8yoEuhpcucUUtuwICldJKFfwV
/d3GTjbyhfoIA95LkxzU2VSzPfsq0lvWstINlK3Wl5nPLUAAdkfKgftZOXNEJ3cV70N/Q3sHFqHo
fmU8UZ0+Oq90VPWwtS/ZHR2vrUXDIrUDCIH8N8W4VCcVvGjLelr7nETUOSPWmBAyDYyj6/mnlsa0
EfvlnAk5ntUKI7OgGsH6FWDZP9Vv8BHtzjjUfq97N9MXed04XDtYDtdUentwxFcXiLovuY/VKE64
9z6B6BgX/TrIXeQAJ6ylzq/YiGMlrD9dDIYshyMT3oQ8QjOJWUTzFVOwEpT3obv6nfTMy446cp/1
qgXfH3gF5Hy6pDkjOWB4odcKL8zSvCvSxQMXpdNlKQSoVPzLv3ir5/6CsGStWyzFzno4/K1nzpNa
Ldl3jbvtEjKZbOZp16RG6ZMdiWO7JkjukA2g764pO6FfHWHBBppLTERCT29skZsITKQ88g0f2xL1
E0chH5I+E8b8+2QnMbybIa5tz7gDtCFaR3wL541gCvh0cRq+2QnQs6DYAsfwGxnS4tnOo4EDbqr8
K8FBP8kUdmnj8iHpbs0cr0gZtjtrqZxJx7MtB/dq1McIb/yB5JxU+GuAS0KRGidQpehj6EqGxPn1
ECE+BTzaRzSJK2sCl19lFKkPPr/+kSsD8IRfW8KW8ywzNl1i0g7+1irr03hfgMpzRrxURYfD8YMu
9DaaQep67zxtj4zYio+SMhjAFsyKI0WYF656eDE+O2xsDEjXWDfybJbwLfH+Hj5oq5gSTvy63G9s
Fn0aque57kMbSoKl6ZYgWOy+E0a+MyTMDyIgHoJ8lvDKv/pACtiPUjdczNwFDc8HsDUeQm3fUaX7
4ICR9l6gWZPPze8FjBS9uY4yq84bd87OjdH2WsmPKt6lgyXi6Enqb2PawbdHabBZTnj83lMq86mg
6xQh6hTMeXg4c0PvFN4RD9CNOpRTnVklO5zeORRKdH5e2rrDDdJjAiMomVObe5A7ZdaP9KAqFdDr
5AaBCK6ahWynVTFT1MD03bNtbI/8FqqGirBRseA7hOsSfoKpwVtzLglYbBZjLFoazBbcmnzKL7aB
GITRvkdIqTh8JFwM8PfR64Tqh3tjpQTpmS2amx9F3zt+8jEFD1TYK2pVczj0ufQw+5ZUYFWVWFNb
t/Gix32Ekb+sinTitUJmpasqVt26inYVZRoG8S9MSRenxBLUeTUZctVxa0UhFEOXe8kBQSzXxLpm
xjmfI1vJ0QeANsGqRnLUrS4mv72Z0r3BTu+SyBJwuZVzGFKaWvq0RGJjBiIdhbStZMA8NS9leKfN
/ncGwnJPlwPLEZk3c68qkpO+uSrRCt2vm+xf+T6JlR+nrE+C3BbH1x3zge+Rbz7i+Uv7cYHNVAVG
l+EqdmuIjcl4JLzVDEC16/BCSmcCAl1ORCnpgPfQv/d1Ava5mlrmIaxguAsI+jhSd7z4usnnZO/5
22VHgkJJjPHweEjdv3XqmWAWPh1NPG/gDbbveKyxerCDzGy2y34M8Kp5E0TPDOVG0vMFrejwNXHJ
e2cGmGdzKTOwsTx68IiY3qRFTNcKB8o5j+KQG2yMVaq/FJgAdvFkZoJswvsv3vFJWhIKsgnz++3S
F5tk2zKaID1fKtm2yYN23HTh9ICIomSVLloQlXxFKDCl9fhFHSUV32QFmuwKCUgE/dONkcZh9Xkb
GL2iLui+dsFYdUllp00JanjQcGqyITLhXPIyYIalJgo/ur2NRnHHoYlTPh9D7IjNHs0MivFhWpv9
eLq5vRDFN7v5TkvsnfkBvb1N8Rr0vTL6u2SOBy0vzxKO497vxzXvxsKa86+bm9HZAXmsneiesy5Q
d62Em97U34alGAvpkNVEGYio3BaxeIrjWVavLM6p7poUzl3HVePA/vG6P4z7ldAhlKO2ouNZhLw/
UnwvvOzzdj/OlMCnM9DKZyjX+4mz19FZgIYVulQH6OI015OaIxAq2arbuESsdKn1Dx5/Atk+cJTi
ugeXCgm6EfAV1hHAylUMx2pBzFIXgK0Y2HSI3r/Jrq9sj5sf29XIH/SBNDa82OzxQnK7TkbLVRz0
ToI3TRxGFyw2wYJfg5CBUIX9FqeJ5rkLeWglwLvrQflu6/DmsXkgr9rY+9UrcB/EyPacu6NInA+r
KarzCM8+0d6vOp6UfK45k2i5AZlz0hzjN39Aj0Fe2P70kXKQVR5wKPjcEB3+Zmy6vV/dqu3GwA+d
lvqGXY1YgPHzASlmIVsALShG5joSLj0kLijJ0Xw7ogrHyNsGiL3QLWQ2pYirl7QTGhNwd61o8N5m
brPrjryz8l3+dBMWJabHYAADfB4PlplQjGYFfLZl7VY4pahR12WABWuYmMlYUMHyrf5LTanx8caT
l+5iEB+6G7mjObD162d193Ufv5GHNmUu87RBLH60CC1prG7rq/Yg59FzQfDASw7MZN5ccXgE+inq
I+yf6BMyJEJFo+6GQx1igClh1fJ/FJt3NBteG8ezN4kVm9hCEFsEjb2JvfcmdmLvoFa66W5V6aQ1
arRKraItLa0q1aKqOoLSgS55f++/z/O53+99znPuOfc55978n2sBP3C4R5ShdNHpZ0ElnBCFFdoT
+lfiak3W0mUMXTsgll6jejbIC+ExeTrG7qiawJfVw2YvHNuQYJwHHUsqUi86HlfPXsfsYHaC/ioI
45SvEqxdnmcj7l/sulvXpVbawPyOqICWxvU/uTZF13lt0xUaqpopawprveDi4nKnp0IzGglwfgsv
ncop/7gCvWc40eGEdN/bnfI6D6Dzjvp3Lfu3mo1ev8hnCD1bDBsosvKCnrg5154IOoU+Vz8RpUlH
r+wAjU9Q786i5mRZzgj0Me0G4L3bU9DAjLdRnPbjp+LF5pp9SIc/bJ3eXAPldWyodMonnLr+5nlr
0cXfRiibwON7daQ4ToHTGaqkIVxJC/UaRnawIbnd6BIDSR1bFZ7fnO+2e+1DzSAYsB5Z/vNBFjCV
d2A4Kqren2MTUp+steetSBlDswv0hN153nrnGGK9vnrc2ZgzKDpXBDo3dVLkxAkf9XDzgNg/gdOH
EeTKb/nvnjfVU+OBD9mxWT8zKk/hGwfXrmh3TPgP1H2qp8L9/dYWzlTPd5ciwjvy0V8GEk8uNHjj
6O4BXMzWFYvnCurrlL3z5vaq/02GQOzrGuNNnZMaZywNBqHeimghvkOZ45bGE95K9vOYrhlehpiB
mtlK3UB5PNwAU5IzmCVS3tBVBDt1u/vhjecL41I0sh4SS55Mb3GfR/Gvxr7+IFzobropli25GyRs
o28syJ+Gq4+42VHlsDoacDn0/XM2b7fawrb+rds74aZFFoqHEbjxbvjSduX+lMkFXch0g7CXWi1p
0DYEPWY6fWPVe/pDYIbYhds+9OraZYxokwb+zqC/RytGg/rpTVVAYsADxJVPjxBV6E1/w9sDZDAc
bC2WvhC7saCC3GIdk+mlOmgoSt0NuADSaLR3ZaTglKLhmNx3Y0HzTbXnDagdOUzNwct38V6cR4PM
BTOpOKfREL9Fvh9eW0/CM9ozxvzLxLgIcYMAkIj7v4jwhGG9Z5hFi247+LuumW2mtBwVnD79u2L7
wOAVf7QUl5bP/4a7Jc5oN/TzVcYWf3TCdFGxJC6a6M+YF4x9pM5i5Ebvdg7Th1LOBq8MnbDCSLTy
pnz/YJoT0BH+olw/Sd1ahfQ/rm7DrvzIq+K30hT8vSFjobqDvlMVWqGOYnP+ilffPWI+EOUHusuG
95ZKnDL/Qdk3wW6SBhNamvFwcUiq46zNZmGqa7XhO9G3GH4Ym/4XxGzf7xYEhuhEo2HR+jTgdVpH
wqymKEvG4TAL03pw4Sdb+UE1Fx7/1ipnoJ8/JOx8ih9XAVrqtlPrQDv79adRZ5IYd0P3OC8lT3UB
VdUG5hZgu3MKYV9gSgOXIUyB2NRDhmayuZbrrEhTyD5TqDwo6QTGwpDKQrO7s04tSNsEOOzYXGVq
bdStc4piUSxiUyvPJbJ5I6hJVnm7wnqXA6K3nhy3PvqUhuBu2fleAY90D0uYLZyfaLhTcOyo7LL/
B/riQfqcnffrT5Q9pilfOglSpfMyXZjx3XLN+Ni7jFW1eI46tep1d2yaZD5T22lzNwWXORh68zJ+
z1torCJ6pmzJEiDGFUjnRzDXHW4Ul2vrut3vIYZ+x4UMHi2s8Poul3FpQ1PQpxf4ne4E+H4X3evC
Xe9xO290aqzIwrliF3qRqdhS/M5aAqRmx7+MPuTP62Z/LPSe6/QuiYinB5eJi4T4MmZvequHhxSv
n2KYSRcDjatpbgIAwlHENeIn3bLdsLyNCRbfnYHysbIjwu3alYBYNdAjJMeXZ+uOyADm2XE4Qt5u
H589FeEaiIkBss14nWv+lPBBw4tXBB2960HafF2f22LxGVeBovNLAyeN1zvd8p5yoSZKlY4hbZE2
NT/bmXI+jO7rN36rFwj4uvvYnJffMYjP6gQ6WMiren5xWDjZv2PqXYp5xulj+vyCcSJ1H4ypVvZl
QcuYf0uddR/hkYxY9KgAImOyLIaQIEGpaikwrBDzKfN38Dz4NFOU/LDPiapRG6jMcS69f1ZtsACe
PfGjus+pRLrNq6jPLV/s+ZJCcHlprXBurgV7MEHjZ9cyXaqxll+cUHaD5vi1jPnrcKdNPj8QZi5u
i1MaSdCIfUk1NIWanjoUYvVDWLenqTveOeAg4hq2v0a+KEfB0vUjqt8nv+KMaY7ZcwLzJnRQcIQ0
4n8jgJUFK376VDRTkprr899911BIwI7F/wPG/zfA9A9od/4KO2swG1lFF66o7fLxNYJ/d2sovxJ4
4t1Z6zPHTlW2in1IulcmJN1XpElnrBCk2b+jMijBr0VKmp+xhApBNpU3VtJnEakgmWuwad01KBbA
z30WXTSPwlomZ9nPqn7pgoSeQBDTBJ4zJ0cLixjZKgygkKej6445zsn6Lywh0Q81w78Rhqz7W5Cr
hIOwCJQ+i0yx2ViggRXZ79OWQM2DIU0+FuWDPrQAJRpYdbBVhgA07DipZvPHIPsB1EtW3erosy5/
hLaJYFAGqppxSmOauMcY8OnDx0ZrvvTysIXHJdEv/9DlH4Dg6DsWAgPkAGYv7v6CmtFb0P2F0NEO
ieIr1nqq4jWgpr+HkQKyku0/RAomKsxYjtUgQw+MkNMHnFwbpzhRRz4EIp0dvGJnMnxfQdd83K7F
Oz/kMe/yKxvTvNa5e23GPB1O58H/YuPf1PIwo6VQacCc6hx7VSgzqMpThzxPlpAjfM/2xiUouWy9
p+k0P5zooJ+HYQeSmsRcBy9UAE+Kt8m01oYJs2SbpFE2VQP0HqJc08EcH+kLXE71cWUdlQKl/RqY
AtUcwzI5IaJONrdkGnQJa0NCdy5tIERLccKtr7HQXYTLK+104NvkgFJ4ygkbDRRlNK1Z7vNezKkI
mVoGCc2BiDKx76Fbg0aFYBq3+6E5XsLUDTHUlNI59+plBkKjlg8sZP3KwOVwJKioLzUfr4+/2EYa
BXINtHZoOHa/ezUi+7jIvaOGoexBoh+xtlRAZ6D3JadUqDrtFHCzExSi70aG+K8Ln9W/jTaYsorz
vaTOiPLxvhEwaskN8pVyFQr+Qm9PycqA9mdfMidfqPC2lK8gY14MyoW6dszmm8cIetFzRMFzgjH+
xOTbQyGS5ypAJsqDp2YScexqw5rPmH52x8OM0ylA5K2iSWLlODq4+vbXh3YjsffL46VnH9a8HhET
F4NXzcJvh9QPNphxvFV7pM7dgK58BSEQQxC703vevclvc5gqfUosKRqUif8nRa0Y9BbczYYOTne9
6b5dDeMCJYGvbBw1qqM0PmqKsRqwWem8yiM3lw11xBEF8pUFUqkjTx8ZwCCIhtOtcxJ6Br5mbtaf
Z70uZJM5qYMWW79FeUDxRtPuwvoAwJa5EyfczQkQyykrNp+UJrCKOyvFIQApllbndbgUlmM55WWo
lg90r7YqVnUFx3BWvAMuF7hrVn4q5Jc+8YXw3WEjgdnxz+2HOtfcVN64wAiB4MzuZyJN4TAZrirH
AuEb+iFxMmnAHSLBJWC73y50T3RJ4tGYsQUzLkoF8kpDWN4S/4P1eVDIccpeml9fkXZMK9SwjPyj
H6/08k1QEmTHSGx3Th7dY97bemUcu7yPdjEXvqJ5afvA/K4Hr/ZA0VGJCujx8Q9vrvTjxB6AKOOV
7FaswMnok5zMdWTl9G+E9cE5eOOHb1u4lRHdlHMFhpd0v62Pw2YQc8mwEW96T81HuUXlcC0HXAHP
6PmtlYMmxe90nU+0TSfXZlRWsuq5pqpNOhMglH3O3NTvBY4868ha+98KK/3Gm6RB7K3zqBWZdI7A
Vr/wGwc96aKs+JlR6+Gdf+iv4fVFxmrSCWN/x6ygMxY8p723/gHIA64dwmjYzlnVvqnRF7LF+eK3
C9F3Le53Jij8sUJMTWtbLaHPq3H1c3hKDMnTJbDk0voqYMKrXYPuPqcCgjmspkuptPKSUk5C59pe
9USBbjaP82+JqylTIGySyAz7TPDyQnqd4Ap3v/Z9S/E6XLMGvTlBq8wdksvCmVVXgRJgN7TaLIzW
H09HHbnN6Zmq2DR5ipquWepYCH0khyvKPi0xt2XGfOmdk3zoYP0nxV0BJ7KdvOUh5Dons1ygtNIT
OpJkQfWSrZZIwbsWXdddduwJkHY+Zb6kyRK1eb2ZqCmM/hx6lyvNhg+2TMl8vcS/hvLKf0C5LGUE
Y65a1G9bXHrYf3ncrhB3t+extKMbdmMI6WRddSDNZs6rAvlBoPSRxqnpnbPza1M7pcYolrOjumdj
yDX5BoEC9O6oTX1BJkkLROT0flB0IMT+sAlIsrA8O1i4xz3TPbbYWOqxaXIEuYU54bEvFyF6o/aa
4x4XxrkubwtMKU00vp4ar4zqkCkSE9C7MaTf1llPrILlbT36dCnkGr4i6sSf4rVOpMiVgEoyqDLV
YLPuDz/DOX1kOJElxFJYnSAIS3ACLo/R9WUkG0Qcz/MLFlZI5BuBWNqWOI6SC+P1cbpKPNSSxnHp
OWpyxMgyKUYPJdwtLwNUUflbaWE0lHb/c2eNcXSPHOLrKKhQ6FLcdWDoMmeEosDnL2SYp63IRdaC
rScsqTotsdA752uOb/T+XMQetiQtz7gm/vod/zvnndU91SMVTLVx0Y9CCrNujHLxPecKZK8Us9F7
JqZzhuR8x4OryfPP32yiXI19m4TG1ihL3x46Uf5V9IZU9aiNMpD+rtIWPtBaoTr3yWVpfKSmVIMu
xsI6orB1+EOVDlKCE9tZIl2LRCWE2pvmSs38Cto3kGfslCcJ8urpINrtBuqS99j3tLP3rjfhQNFN
MGBjtPTltCcv1JsoW8DAFYuf6TD+740L/eY+hfzf8hC8f5i/Hltmr/jcMfLONyxqJ29FHa6TFUWy
fTyP9yixK8RlqqCxtOesBPEkEuCWvi8sA2OhJ5QyD4NDLK4oksRvaP6wtXkF9Q+kO9qF2LyZzX8l
75707ZZr8w3F3z39xHMBChzRipFmo+UyMIvPQGaX4tZ5Hj1kFwwuIjTfAn8ddZ+S3ouA3MUAaRpF
d6MiggPU10V+t/rk7E7bj4lyBwHz9wlg6RLvrCP5WNVtNM/ih+VAHeb57XA5EoVhn9X5U1K63zCs
QrVv9+h154q6B6+6X14Ot3UrhNNioJp+B03HLal30iCTcNvNwI2FloCjXIkio0GkjQV3nR7VHZQu
hsqB2ipy3+TeCDSY9mbKgWSKI4Q1g7s6YqP8h+WkIshHFL9Y3bLma/2lo88t0s/7+RveSseLZspX
92JlfrcYDqGq4CZ+iwBb7jcJ5IigaTWTqwKdWmka9xv5cPtbPL++naRvsLwD3ceI8N8BDDyjesZn
Lv4TVGaGmuW/3ZwAJUFd5GNFvPHS2h4LBgdGAfGw5SneOST7pqrw0bU7dkZTFtdav8qc3gzVoH+W
hzyV6mW1Hfmp/pPWcp+uEnMLv0bavmvgBOqSt1xy8TtSivCqOaTywnnHVjAW6KkWTbk3bW5LqNP/
lD3CQHemrKh+gEAnlgiP2xFTn2YqbHA0BvphPN+6kNXLHBGNDjuB2z0VRzQOVLFH9p6Fnvnq95KA
Ybxw5qgaodKLCoQCrfdAGfsHCnCCh2uEpRQ6uSnjHGPZPeGiHtlpSMmuk0iD0A2RtM2hiJ8K5/w/
+GUdIHekyA5uiuUJMyn+Svw4v/CjTaik13PuO9bLwHaXBUrWp4eNAou10v5Tz1CJQGaty+2V3KoC
DKH/xfLgJrTDzvpsIKTdTaXzhqD0noNYjf7+LY6it9LzwS9QJKQueFpl4eYRuXG5AbkQgT9KHvGo
+Du8AnfnLEStIqSrtzHXjgVMVBRRQUdDSXdJw8Sqzl+Kxmv5WldeaVd4U8Px1ab0hR46zs2NUGLD
n4YOGgogcX8c4UpuokKRIPUUDRYUmlz0FQe02R37mqPEHkPxmloW7IVsbKsUxwFfQSZZLLDN9J+4
MPbHMdz9BwiO7T4l+qF5vPzpzLjGBxVLgMoKry9V+MsVg4a+gP0MvqLftDbVo49DHa6jHwYAMih7
+Umr3ID11zYUo6BM+Ftj5y8ZMuGwoHgvPi+ujrSba8djT64Htqe24s3tV5QQFLdRyfYVWqVwPcgR
87PhXVE+d0dDoHn+coaSiU8G/4IlXyBNUyG6nt5Y5M72itOUFVsAKqW35Hl62sqfvO/K1kOfuLei
JgV55eSNj6y26hIlu/1sre8WYfeGnkbeiDwD/COluxfwzO3KHL0Nh6P2tr5Rn68A/PrI7JMf3Y2H
cTrx3p8G/67H6uV6YwYo5r31UXPv/0GmfMbYyDJXpbOmnZST44Gqda8vo7cDY6ctcat8FqkZ+iDZ
sbo3b6pF8nSOftp/brAuXXmUEjrmRCfZa5FcIv/5pXf5oL3BP4TALIH4wi0E7JQIXvQH95/l1FpR
pjJXP/dc6+UcL8U1LJOy1TMNG7G6xtXmGllxjOMiWtlg04BiA0HnUJvHqlYPigtMjEyZU8+b4DB9
6++VQynvLa7sIryKfze/duwJaXS+94+HxfYNcexgLzbZa+DIcsJ0FcNAwdxJJa8xZ29Le97XF06S
2V8yDSVvevlmCPpiDl932JYW0jrEsUFLLh7hjnrgM74s7xVatxS12225LZpfVKM+U5c9Ts7y/cyp
h22jARHVA54MyHRuen+QeWba319oeYu/7pHqvw2K3vPaNFJDGizMz9lxPJ+Sdi1avLBq9rNMMl6z
c0l+n+3EDwHqwOiI0xXPHuq7jVqlxG8C72vvwEDarOFO1fiG77Qi3s5VCadPxNhqb32va9UFEPbH
rqzN1r8MgpK8NR71ZlYzmJoK5HcL7raix0QdDBGyTuien9VLpKlmI6jGrncZh/9phRTzLMQGvzY7
O/2ig7vi8x9c+5feBpPu95U/vTlOXH7GGiIp5TsgKL8/EAp6mcbVw9QsSDoCcCeny0pIzTlD/rPQ
QDftxZwEP8LctizEnUDN2XCcY93ckdAOsVdclYXUDAhevXdnhU/qqwSvCHmiWK9muDpx4fQ148o0
krdpSrfaXHlaKo2kGvATVfyOLx6oJqH5ErIrLxUR+kJYegQvGbgSn5Z7Qz1mod7hUobLFefGWbVl
SirGw/13bRgkU2nkmhQGvx27Ingby7GotiJgHsCaI7VcncqL30xtWlXSTQYvZAmUsZ9R8pYf04Ub
lurujQ5rzOvpvPJ8ACMy2bonjsyCgj0HSW18j72Qp+989DlRhAZ9kh1M3YDXXbsCDrj24iMhY6Go
u7nvWibEUtfkTTG/EQl6W5MEejY6/2a2PPxNqWVrsaT6+j1d5nAzu96+jGpe3GL9E4P4OYuTukTG
s2RYcr8ZX7GH3+S+qawpocpxphKFNIO4x8kK6o+sFQLnZlQx+EoHnBsCEFLgOKLb9238l5sJ46D2
AojGKwELkCd6pAhoZwzq/VjK0/xHpDh5xvk0wtjBcmesen4Qv+cjoCHHRhIaGuWSTbKYUY3WRyw/
59zYrApstk/aAUkCK0JWpdxtP8EXJ4cEYMEJWsUS69Td2ab3rU5g+RJSvEd1kSef18LTTEoJXYiT
0fP4ZfDC1ARaZJSqsyZCL84Bj8bLx4ML1t2K5uUMg/9aC6/YDHowxU51SlpZsonxziNdnkE3BGJM
K1EMDVinxZA8dg/ku5tcfac4mUYt9oVVYgwi2oYKaHzSkcCti1fE52VSIjNSWDNK/B83weWx7dRy
JltienIrHP1e3YxGEmlIhFvSIbV/PbKOV/Dte018CmT82AyWABfbae4Uh0b9wHX9LpP8Rpx16uFA
8Lshs0V/fs7+0LpTwB9J/fu+i33X8SArxEiLiVWbqJRYClXm6CEUWM0vEcpfSj5A+xe0qNMgihbz
gjQh1vrsAfKlvblNLrXBcz8Y2PYyf5wlMdZcVJ0QEa1es9eqLzVi/k3z6h/Yubez5fep1SUX1lGi
Jc8nAK3UdOSDKG2Pg6JtaZqI5wsyEQroMbfi+D8V+CMHOwWRMq5zp1n+E8BhPcN8C4/tr58gKAXr
O8QpoOuIWLNIiWY/TeXtsUS4Pl7YL4JC6dedQD1M1zMWb/uwTs6IvMKVmZcRtHg4j/lL3PMnFvwt
WfRkYCNyte77ppKIrKse3PpYsLWWUc7niI6Rnkee7MSJqqiJPUT5ndNztCn39BlJPeQdKxQECV/c
HLuKyFAVbrUyHiuCljibmyI/I+ij0DfVS4WP261ydcRsxcqLGekMrbVjUpF4uNfVo0eb6J1pa6Pe
NjF41KLOurM6/+uMq2X3SH8F2U8vO0SlBYT0pUA4aT5Jem87AqElxv4K1C7NYvCimbOOi1JwmiPi
BYgrotA7oFhn5jZewdXCMggEgbdXuBKDRdJ2L3akKrer/JF/phJfrzf81tJqZqb53d6mBdR7EaVe
ac0Rq12vDTKVyCnewp3hfN7xcfWh7C5D/wNBfwmLhLmBhFQO/S8mv8G38hOH332nj+VW2vDUWRAy
BtOv8hDMyNDLlNBOaxfE4OTs6utaS6ScVIuVmi++W5c6qvgMtEiu7nZFva5NpQK1Ky/9qwzPRRSB
rq42gpnTDb0Iq5K/6RuVudahA6dgaigJ/ZuZD6mf8V6vG6qPSHzcl7f6/DKcwoot3zSInbZh8Z4P
ygHHKA+XrvHJVy8ZUOwQLG2Apuippx8QVCYd9bJKcs0KDxE7hEjCLZmwpIGvmJQ0I5LcNC1Pf3VH
tXM2Ks3mHsT7t/mfqpBVaEmlyRushKE/FmwqmQ3pY3bIz8mSRrlPYkBzMkYgyAQWIQ/52cNNbttN
kkQeo5nT+BAO2T/wRzdOK0nYAZCe8DibM0AniBW74qZ/qa8nfYhaXOkRZnfA6ce7wcxupEyPyTn/
ipK6g+AzfMQSHX7BFf94b9pny6+fMCd0Tjn1flYrvv6mzg0S+GSprurMBERYPBU8uAlOLNMU6BXu
DhczxzzgjX8P0V0tWgGt6faruAEK6vc5JtoB3W6XBtkTO/rdhM5sGOWdxV7DqJ5WO8lEWyarwW29
fUtnXNRHptqnPkc+Ve6ZbRbYS8/JSAxwpbZ5K3pT49iWNldAURAu+0bifv4ZV5I+sL9x/trdHrNX
ALrAva3w4ArH6qxrFSNnR+5ZNSt/llBbVVWc66mzItGlbh/6GttUVmHUx+uAAVSKNSpYoZ/89roG
4A6kVxQ0K2h3kY200n2p+c6QRuK7bZk8xDRJXbYESBDrQPEDG201JK8yrrJJbsp//fchVlz/q8IN
OVwkQJymHzRs9AsufFPkCc71MEMHG0ju6EElkzwq/0RzLZD7k0UBpgCJ1xlIvyKBbondGqOAA38O
T855GZq+TxTCHHzRI0cR3ujAI4xHMzAvckfKwiHhT3keB47Yn1HIv4wVnk5R0Gx30Z/sPuxhORhi
OiwVcEOW3wuXaQ3EZuBus+tErmTIr1g4zh/riomWOT4JDpPVV/rD+P6o6wisccmI/ezM1zjfPap7
KDXsQXnjn6gHzqfWHvLzX5ul6cwI76WYqwOzb4uAJd0PHX70Ud6ExzDcrDVAHJFzf+9K40/L1t8l
MvSvzQ1amgNx4r9l4bdGEeWciiKUzMCagcb3hFJacJLWofx84eGemmLj96OTGXi9oINufzsbKm9T
Y7mw2KDOBsNTHqd3rv24pn8f+yMVx98UU/DOwJh6DDh/zuA8O2GkPVuMmG6wrZUwLFYspYIGlR6f
NNT+Gu/PAobTaUzTW+2kAChmGecLL8GWCLuJaV35ZdYgJf4m5OjhVsU8zyeKJx8QKJHfDduS4wMm
XwXZeZM9aN80Q/VEDHcGaYJg88j7tOdQYTuL+TDVL5hYrpWRCd5apn01znD1SeKZ2/oeu3LBQ9do
FnZzh2PciyaK6BoB/DUqoUa8uFfgsyfvOFw1mvSIo+AJKqpf6YpxF/IoKHwwaPUSTeT2FsOod8/C
efBFbDNT5DPxJi2iCBOPKEMmDHXT3W/9BWmpiR+ZpBw9HDtMczE4R/kgSo3lUznLLqMay/FiTCxh
sorY7jEf4/+ke4Myh2NQ/xVx5Krx6YMxMrQME5TxvKNtzOxDwCpk0fsRyuZsg1sl7E/CNMqUvi/8
FBYdVxeNQwljxBYsIuv4pTwJWopP2jVUhElQMbIU6Uk7SY9gpEq7pL7eTrYLZvgCd17PxgbmgmwQ
EX3WNqP9AThPYbEuBvwbk0AyJJNzXAx0DfSYRjImNJ0n0aO5dsV0+rW75yCSJnDxaObYihC8wp6F
bTbxtYi5aL2vUMaTYamGd/e8xH2uJiea0QkzIgOXCFbf2LsONrMkKV4EDScedWl+pntmqPJFWSWg
2qFy4LF15/1HAQ+KECDzrDm9CMv3NteyTNkZwv/Ku9HVpezG7ee85fqMO0dCk8wzVgkX+zKRH8GA
SABlgY9C8HUlS+PUjz1ShiGB7Sqo3EpLdzR4OPK40nqklqGw/VGHKww5wjfnp06D6hFCiCurrT6f
wA94hL4T0gMUXqEQe04+rBauDi9eLaNXE1FgSJ18rjTb7XhQ2EFXJEEb4m3wSAlCE5I42qJj3iej
c0qdeRH3A3gasO4ARCJO3LVXLiwqqW4/Sr8Roo7MLa158fBPnF88NQPEzZXgMj2KclfEbsX0cKMb
wyB71szNQBbO6zvX5XhHxNhVrm0VlwajLM23HddRf8pVhElbilWdO58PEcJhYNCAFPdHS4kInjk1
BWASatzBgk5ZAz1YkurxutAtb5NP8cApEKS119qo0lk95+Kd0qShi79BrUDbiwrL+ueEyc/pb1ZM
Ii0vWoWabz0V1MZl8iwIWW4TrJbQ7R3dWKYk1atUgSv6YASkN+5pd1fvZceNyfTzq2qUUcMvhzF8
C9vVRj746ilJ6D84e68wc0wAxJzGf9gX7AN7LdtzP903hz41X4KqGBU8oJt4Hi+kN+L+bnXpXizj
hjvkQ+96VtuSXouw3FrcFjIVuwvNW/an4cRRoC13KlatAfIjcHDDeTfkDhZBnGYqU7artqsAJqj8
+gz2Hgu2YwtkW/AEpjZ3yuYrfXDqGsTZMOIWSPTIkEqMkAjjyzzPMkSSBgxpCGqNwXRYTltmQfar
ezvLytylaQJSX8wp+zFCJeX1+2bueU+ieBs4YEoXgzhFGcuM9xH1yrmRbN2L69ixEHP2iUeJ21Q9
db/2h9qyXXBWRNnbj0mi7Pn8/x1OlffFJYSGSV7CnR0+MP+eZN8uMRX3MqNb4LFCJpWOEKN/vV6P
uBrDX3DErTPGx6LoXowy4kJVUrSyiaS56PQUzyr4EMgrEzDN4pv8UNGFT4Yjx6OPyuKFTw/GSgsD
kZ82Nob3oAFiYCYNAmGp37GobtmnnOrGWNUva23IDH5BsVdcln3baEN0d3pgOCQeQhtq+ZugafNh
sLTcybpBGJo5gmbcDC8+S9t4qT0AuykABAUWME5USSTn7owiOCWiJZO4tXqoLBfduzDL4ge1gK8N
/OYqyTPhmLkW6wBZrjix/yQU244EnjptWdwC1e7/peC9YzAscIP9B1IB48qIgIasd++6B/NToA1c
29YahxMxzBI1q+hVgMzR6NzI1GUkDTwBsDljImL0kCBvlXgAorQUbH32kWDl8kGxKnuEgaT/QMX4
zxrrw0cDBX9oeRosJGLm2WJ7r279eDKJ5EjwrEujWDLX4b72ZzZSsonhirhgD+bOrhC7TwSQjZKB
XREom130FPwQH6Ghdk6DgsSpm1/SV9vrgGMI84/eLhds1NzoUPcqziFN7LA2+N7OAfkRQVHHpM5u
lHxWf/c3h4gjzG2/hf3ZmX/89ovg2/kLGXSNQhLP+95BkQHbJd3PMh0x1HHk/LKx4iPnXsP7mcYs
kxzjRJkxCxxhp6jPzqv7QCgnv7m4PkWwc3PhCbgQwj4K4bAhNCAyY3BrMdRJt6pM3ZbKCnBOfhbd
FaQKJT2O4ByPFafmlu86jZ6GI/UopcmNfORd3DBowXxUGYe04qq5k490pOm6jO/YI/a9l8tmh76B
2oCc9nOi5g/WJPQV2Zu4VwVZuQoLmnN1T96+F9Z++/pSU+nm3Fy1CdGbKyf/O+7T8Y6MNGGCBBD2
dsEM/ZXzc266T+ibHqt+s61wFqeBWlBb/MUjzve5Q7N+obqiTdy/RBdNYKRunJ7Pler48uUPAuA9
PzDfAl/q/PEa7IjW9XOf/nMYaEdfIikufgb2UawHUdKGiLtfDuRN5/rjD3eq2bOHMnZlEiSkKwQo
jTp5y7wUqUMccnUoBZ1Xs7WL4BHWCtFFiaWEojFCVS3MahW1X/Q7pqJWUmmRujRnkZgwVwDjcXTn
6vsltTcrwX0b7HGG1ZITIV4kviOyswZ0f2qx+p9/zGAV2iSmbGw0pLqzilgxgBEme4tmIcU4zJ58
xcPaEnlc4J4tHcRJEUF2efbcmP7diCKKvVMInap6A1pYP6n6gVzBz9xS1Z8uboMkErUsly8sFc3t
ESu/lQKFP1IKEOBwdgCg/5N9i8NS025LBPcLtfOg1WBTG0hs8UEMf3UtJhNpUc3/r8n8t7od3GYU
JhfzWuA1D8T8mydE+V2IJb56KFEcAR6q7FOptGBCVH3DIm+KyZOwkCCP9vKO30A9X/7kVMUS+9aw
a1WcPlAs7aJ0hZVzcjUxxs/2fKX5uabCkiwvyrerzUqMXq2APE3iNy1Wq2wP/elO3OcCzU74gfyR
8B43tAmi7zI+veZ24j1BtAbwPLF2+/nLIr/uHRxL0ewP4+BHyR+STHVEGFM+s4PR8TNIXHgMnnI/
73roqi3EVghjkS2+K4FwgEVBiuQnHpLmWyV9vxsmkwPXq1uIL10V2qXWQ9NZyENMY6PcY4cNta/P
KuZLVwtZgBIEcMnHByXu82NLu7cfKucJQYkD4Crn0ovM0FLYd/FIsZt7fxgd9bhsPn6N3fPyHOfj
IYxWA9TxVNBd/L2/wqQS1PXyyjLh5z9cqg5VypXgHBQWdMmPx4UH04fmbP4E2qs3iyOAOGXhL3y1
qomopbdojYbGxmoskshRtd1ltjYspm0/xBdxOB0qNYOVNtanNsOCTdIicgnGJzxP1J9jkV4f2HF8
X48tUAarjQLIVV6ltA6wiDDA+VRp6BVftkiarcfI+JBkbfqWquWOJvANbccc83oM5km6cxf5TNWG
hWgdBdWhE4nSlWJsMuNSBJ9WG9C1BalGHLd5B6w/T4h4VfApEQGC8AwtvKSqt94HVWbxcaQfrS2V
13m2VatRD/GthltmPPJpdMZcOlw//e3W+0nX+uwiZSs4GmcdJDkFcb3MNIDe1+gAgvDmNmjmume9
zbCjZlOZKADet4isghWQ/yD2bMwduD/Nj8h+ecoyPz5lS0DYGWlc79LfkN3x4JFfPhTTmrFyao3+
WawpUQGmSymuu24PAHpa+I/49mQi0o9X+V61K7ZNa3LqcJGC1D14WP/6xivYgo1bhbVT2DpWFv8w
Zby6rvT5lLXHoBAQ0r+i9hj6ght7cO4mSEdWcmj5ZgFDfiIx4WP8fRhtZv+LHy6rAtY+7PJp7kqd
siDP5KtRterVFdKXSvjYcvxixySqJ0dq7mARuVpsHv2mIgxYfMwrj6t298gTlx01j1skmgqtgz7w
YwnftuB6wO6d+1c43y9zWX1h2j2vrLPL8g2b4RlEZeHk/Ti/4sFur+pixrPaepFp+wNBoWeS7c+6
HzsE8puWGTq8ba6pVazl+sjHXXu56diDcVW19CNP6FBJd9f74qvfLG7SibtuSpQbeiDRDLmzt4Hf
Y5Pa3UQ0MJ/opYbSF9bSLybUVltfsCtSUCOL/IQJ/rvmSL/jneY9UP7Pzdlo1d4YBNJAY+kLXe52
wEqO7QU3NfrAQd/mLFNRDfpKopOBj2jm0PPJL6v/hZR6rnvduebbXkkwEyrUHbUBiVqYkuWgRif7
Pt1YsxD0GbQzzlg66d89Wjmom7pwm37+ncDSSdWzW0x9e6hWfOtPxzKmmn0hrNGkfMRm+pIu1eJM
UVPQAnGQZ5ea45m78DavAC4MImKid0f5FEKRDuauFiYIB68AiPBsRSxAy/asmtCUtwDqdKzPtntQ
9zf1kl+Szv0FAP1Myq9KDGo0Dwn9K9hkdSsDvBu0NRbqxVTLtnY8p61M70k6/3VGHfLDvbdRPhji
hWYEXMro5SpMafhWYLTKfR+vnRQQsH3hCgCoMKpLrPIAACBoZ5/Hg0H+/4MTiUJj+NBYyQAyVZuq
FJkIQMLLCqlojDYsMjkkLimOiZ781zaflo2ApEcxQ6Jj0PjdkPQ/yYCI5CRMSlo0mmBqSlCGF8oG
FKQnM6JCkqIIicmRUeKWri4eO0kLIS5ga2ENMwcp+/2uqHUXe08buURqsqiGNoDXDV+wy0RqxSaL
aiVGZ2llJqclpC896NxICYvgywEWJcekazFyksEpO21HeRrammRdrfS0iIh7mhEA+8taW7P7UWlJ
YQwCVDMNQJi1d7Le6d4ps1JKi0ph2CrGEJTSCZlqNplNb2Lh1TZm0ZEqiZLrGwTyflQ6UyYAHEQJ
ap1ddHXUF7bG7yf3bljk6utJRCTqyz11tmov/vctxNPaIz0hLiXFzEWOm7QB+LWemOKQbqex25V4
M+NanF10HOPOOMl8qD2ECKa+25zEQevTWeHQsMhIn8SFsgyC2eJtZQLCpE0V3M6bzSNW5EQlR6uG
EwEx+wJr0beiAChF+0hjwlL/n19GBEI4S5vgkpxBwBJ0yNq7IvvJHG0dYz19Y109gmKEPEAqOblt
SFtbWoACsKxNDEtKP1S2EZluLOuaFJeAvJ/cMmNBiH7u2O3k79ptkb5AyChNF09ykHRK7rRIKI/4
QZwIVudA12/HCtk6ewoIoc4NneFSBB0BTlFZvf+0bKIYGv5xGRqu4eGxLFoagS6hzggQTYCh9FH3
+97ObWQTY2KHVS2JTygUA5StWMLjpyhEhpLVQuK72WVl57AoFtwm7W1OQtPOJsE3bJjgkByLsRJO
oma1aNygNA+nh0ktybyPiTL21ojM+h1r7+0U8mdv/ytfakZEtE2SCz3BK4U/2SYKufS+a8uD8Dca
kxVUKaw5XLG+qBnk8F1Dm0gtEk/SCUt/PKlBUZEfS0KeYjL3NSK7cEM61/hUsybpsflRWWYUwYfj
ma8w5YwD9BSiLSNcJYHzrzAqKnJIvzZ53NZeJbI/m6Pk0T2XfTuRdjdNtlbql4yoI5MZ5U6G6Xib
4cuOdpSlhi/Exuwx3nvHGsYoEWJhKQt9J1qRTsftmZlhJgzHW8wJalI2Kia1tDu12oeQ+ifqonLG
sJOCkNqR8OSuUbuPB0QiMl1nCPSEK2sNc+jwF9KtXVYbKSCzZBvuGY30RTWZIBXRDfXcryoOg7IR
rLS2P2eX/ZEj/1St7N3VXtGeaCiglVRE0vmOattGZ0bY3sgSMzjlFu2ZlO5fbPBLLunAlIje3WEI
zVVdJq7tkAD/mdNJAkrkUqei8y5EOj5a7Q5YjKb9SEN35I6GsZGgZGNTtCYhM0qCgDUrrxn200pf
8Xp0JsGcf9nzjXcmNdE7mkSJiNCOPjhVMd1FS8GWLTcP2tz8K2ucA81rtT+cFI7Q+Vhw4Dyk379z
kHiVYC90OLWBbZaWRCIUl+h1Qb5kaXZv/FyEOPmt/02u2RjuDM/RyooGR5VjWkQuUAnZR1+pq9jE
tlAaPFRtFRN7kcMfD1tqGR2Oi84WeZvegsMLzvouJgYmFB5Mf4GxdfEC172zSdeb+0R3fpv17lb+
xbhxhxThdOKmaWNMrv1BGw9+m+SnI/hBX+ljUSkR1aaZdmk6EeYTHpiQXCPSU1XNvjRfUsLZP4bE
Fj+GsPHG4Gd1nKHn+Yazc2GnE/pRHw29nfV2X8da/KDnfGXsSkhdZYQBukgWpA2+xXCCz0E7Vy8F
cxc/9Bp5YcPc3R3u6Xcz80VHoPq5zjSHuFsf/f4MGiy/7GRm33e2dre005nub7ljbhFv5/fABh5w
6W3SrocH3NXdU8/d811T1469pZcTDOV1wtXDelvRZNFCJqruROjlLSOp6Ib3p7EsvpiA++k7fnPp
ZxlrsRtt1bF7shoUPi8L8FjYWH5PICOsBsCwjlklHY9eELTZTT9tpL34weIq3c9WU7vJwM53lkkS
4yoLH0qpfhmhGa5PudYl/sXJ0ZmeA9XQygtYWGKZ6+qSlSw+MzWcu/RjtVTIOtqA0yKG/Ctalbte
HuZcFU1fTQfNsLECR3EJk0RJWrJwgFLwC003M/G8KmOFsIBqrUcATaltV02ppJR93zf2WrEHtx28
YZ7GiJGBhr7S/LJS9funuu6FvOboDQyxBy5aYxlQ1JiNTSI9I4vBSj4bFxb+XaVd5Wz6OydpqNPL
GQJf69FAhbAoXeV8zS4dY0N1g7Cdy+fxMkcAaWFNE+z0w69SWSkfAepXHK2Wc5Ns1bNRgsJgQIXT
aYIYnWLLBzW4mJQpZNxNbxA8BI8wyY80ukPaMfLT1/UvtndeNQmi3I2dNt6MyP2rhjJljJNmo0Oc
VEu84Q4iv6kh1menky5vRDw0M41GJxZemqG/0DPUM9I1gDrpczybn0gaMrwYbVfwzfx8JvGR/Isy
vy8zzuno618E8CQAfB8f+76CgCUgfFDIb0qNAThtMk3bCNByWi2qTuN5+2PT4zfxW97krd4EoZ5J
BivzAkqJTMFudEU1K0F0NbHawkqAxxZzZwVkJfAfn22bS1nh4J5xiQNv5tgu0WIaAPhH5ZlgmNhv
WQTzakKyQTocgiLDKWSjwfM6WBygtl5YF3eeDzzpsjMaB8BKhyRldZZ21AKyCJDAm3Rj0xRJR0lm
B+wJ/nKljvbcXTVoRlmlUXrdWoDLf1vs0JGXL1dSfyjJLyWXlHz5J9SyJqVSWjsr1zBDUl03C2PD
0CJnKfPloGHfnZOtbZL5JlmSJi+rlVx0PHA5t/48dvXQjRPVFVhAP7/5ITGpFbuT1WaTEPNvqnds
hN8JnoHZVGF+IRG2qcxWc7mSfODmd9WM5Gg1IplIvHu12HSIGKMJuWMafAoRlwmQu0iPnCbhh8Nv
KJt02MLr/mLmrUbBl6L8hQDhF0s6I68quGUQSSV3Bqx7rin7ow1X53FP+Camw/QZCr0Mg9ZifKYd
g3+rbqpCQhiiDX9UaiT8Jhw80UASRWqmhBOpGpnMPCPntKPGDxszvom1mEt6r7VVlFSHv3WJORHX
aRcT8WDn8o7ETKwuQ0sOJVwTpmKUGKf6ZCVR/MmiGE5U3JB6olVTb18Dib48365ctcCeU4ytKcRc
khA9cxacFhHfnqOD7FqSUgrnE01KywAwVQHR4gnliXBwTjj9stuZmPSfC2285MkNcDxAXe2AqVYg
AYnig7PrlNjIYRQnGSi5I5WvKGZj64Y0DYWdefHxKCEsGatfen9mycVC1MvF0cXVx8bW2vMhXsOh
6+lJ99ezmJbC4eGfU7qPdd8M1TweDhNw0nv7ia0hAmiLSvHV+LrVhVguT0RvwCdP/eWPgjPjeoQr
vCYQlJesD8s2etNJRrFp53xbjS8pfASM+lrcWVtOZti4oyDKNHAgwBtAogH9P8JPMITnFdMPkl8a
2pPEI1gwc0AWAHQuPHEIoKNed+mk0j3bKwaAk7oGhscPaewbUWLtDtC9McappmJhFfDW9Yj+WYFu
j4iQ9+HG31TL8JjRJeKHXkyH/ZLddroG+V7uSrt7aYhYbRxdu9Tb4WyKG9hXTtkuWsEq89KM/1aF
6317X/yt1oOeSZIHiOrO+AGL2U41dfqO+zyJ5KPAbjdkQHyq1ZsVfGtCsFELtmq5IGtXGxX4cpYj
H/kjE5PWg6L2HIqJFTDUGKyhGwxZmjMEAFQ/0thNgtcrMozNV5xmGB2WiD00EqMUE5Xuk/nIJdmv
8OVFXYZgstg1yT4KUqn1MoAqdgZ4pd0Y8EbqTjttbVThOFUYH12GSREERZtQt87IKvXrR0YVNYsd
OWAwq74oYPqM4aQ8MPEy5lrqnlxKPKkhIoD4BRKewgxzSRpSbqVZ2ARINbbPe+qsYuID2PyRmqDb
Oq/2pqfz8swAkrdzFySIaspS2oZkAFH1WIpVlyqGTEeDNvRtM9iIMiGN6z5VYyXPVc2pRSo5Ueed
y6eOiEPUAG2Hs/ymbhYxsn+WTZcGth5/4ujtlAVwDllN/BK87Y9Dzou5163fNQsmzANU+MXP8wUg
YYAkCzlKgIpGcvjkDMSYVrhdj2NDdxP1r5jEN+1YPRtfPnjxT6wfbztW8wMAkqEaptKmcrQVl5Gi
qoI6DdDlZjJTOKUPz4s8ttL5PVzwn4wK0eLIijQ5XlrLGvZQvstjcvMeLDTZx3AtG118+kmii2GT
VXEU8Ssx1G77dX06FbWj/cVBb2TmZJygJ0POUm8ko7PHskpRMz/dUT3hrmgMqF2/gigZ2qFEqglv
y7hV4j14EUltYUTe3VU9VO4bO1aReT+22LhbOPf5+ahECzSK8jsYF7DoGH606nz4E/s8lURkdG/U
o4MqYikqHjvRwEsL7up21cIqkrZeKh/aOQ8vL5lFq5S/JDIGBZdDvyWGIRN35JNv3cQ+8M741DlZ
LaOR+yQYd9pbU90vDFoH8sJuVxSRs1qaY/mdkitSpzstDq2jVFIQthGEagdVtFSkb3l4/6WOY3IJ
5kvpHingm5pxcP5LnoPnfU0KtVMau4oHfZw0XPClWT9MVYilCmV9sbYMIUlZQqS5GtzqfLjb+HBi
sc6XxhLNIYBZkdvhUxqxWjKAk5IHVkCqAkPkmmGrnrKdLzS6g4zHS0m2E3+rnO5LevupAIvi6Dnv
HPJJhaw0MNFSyL4EoEyUi5S7riK4kKt+fRibe6CHU4/BWErfiutaGFkEp6zBZ04MgJ1fkM6ui320
Sjk3cuaKnwHgEzmy/s0yvpUB+Kx3Kzs3/3BKk9pdR2bUnWGxtmeAYUBhqRxeATX2A2kxX8P/OQQe
gpPpcltpxaTwQHGWf87dHE9RQvokb8GanQacPqckjiKG9axDevHOL1gL+qtnKWBTeLh5tJ7zCz1r
e2c3L70n0fO0lXi/1aCwSUPnF1FxiSQYDJHK8i7KSAEHnLsC0Ftap0M/qGMpZLbe2yJLWRQO2xpN
qwPhvESxn2GN4tpS/FqqtC5B4DzWak7ySLSLhg37gB54FoM2l02y1w2dLMK4YVHg5xg7f77w1OPR
dnWA2GS5uSOmVozMpQKs/n9kRnkI4LAuYbIIC6wEBSoehhQSkCfmADqEkMiL2DNAP/5YYHdcI2Y8
jeYEaGbY6dq9atMC+d/RAz/nsz98nzkop/X+yMeXWt7AKN1QnwmtrNYZWE30ESByxjVaTBxod8DZ
dnQ8xPW1FvB91DNC8io2BZgcAl1X42J33xkUaYe6ehz+pPVelWxx8dvt5c5jQPAICd/W/xfwbgYV
0813NNEYd2ALCZZdP6MVisaKoMG72Et8gR3zQPR2SHmuYfNyRDxANVHDTNQXiMJBHPi3AIZ2LG2i
J3mqEygeYpuQsGMxKRHmIU1QJ6AnRA6F7hWLlygx6k7J3nQ+2v7ExEzrtHbC4MFOXxZcVVu/TWGe
v8JaWiyLDNCDN69rO+FV8S7Flxz2MMec8Otjt9IDtIM0i6VMNavzb02hAIrDZssDFU/ELRMAU1Ez
ZD11S6eGwSNzGeVD7JUpP8ModZ7OeMYliZuHJcG+Svcjt0AqrzC1didEbor7IkoHhRpuWzebhv1W
TFy5IIkfOCagkq98IlSJpHD3vc5/AyBxBa0VSr75NskAZ6AH0SVevIyb+KLTOHu5rFTJPC9s3Qtw
yTa8q/DdHvArIReYnkgKiA9C6mutOkTwldheo4/gmIk69HrY6zlArA7i3Re0hGdYiOS37rYwMwMm
BKBE1ovUmFd89kiHYAz4mdiSf0NEKVwKDfa7/qSLjVSyk3cm6CZ/ULNXcv1UdPd5OulGsbI4gBfS
JZFubmkZYu3iHeLu5eKS2GfFEAsZc7N25sjpsICYvsY01asW/7wk4SkkwaQgYpbDEQFRorqlScNU
ooE6O/eOHSIrUdm8woJgfPGmgFzY/HedTIhwbALrdfrdaMTal+Bs6a/ISDKGYag+cx1k5AApWvoq
Gt0wKo0KMAWh0WFKhcI7AKE7Mpevq1w62EpLcUl05K4XC6SFPfwEkSyPPALEAQh/SvLMqUAwTjR0
KVeOitUiiMJ+lfAVgUVBBK359mEhoKgQwWC+vQMhAcLSgVd5PJ5lHRmeSQrj37znhjwtPpIIP/wj
PkW/KeLz5f+SvNrICzoVC/yNPW+yZT6/XK6Yf35S3akL2ZfXVgDX6Pl49O9/jPSDI5G4bulvhNZ4
u9S73xfN+0vzZ15LcFffJrp1zvB4Wm76L9jepxhCA9cSInN+hf9bfxRV5crjBfwLBArPM2Cdptjf
P/kQnHFCEjUYF+e0av5/59CXaakxDdOrYBPbGYjjvzK+5wmdqKbf1zr1GafaLWYf/Z/pXaD9ut/8
njGnwwpcsn7GN/vGz3T6ZjN5XtkYsOhW5PZ/5tU4PmzqQeePO3IKIx3ep2vvPcM29fVc6AGpIiqg
Ie3zPJ4jDzOq+BnddjBPQsAp1vemdNigqUZ01xHzNxmw/yukLgDvE56fUsO9/FpZ1pYq29kFa1p9
cFF3pudTwyN4lh6Pp1H/a+Iask7sFP+7Xr74Qg3TRMTJfxye5PxRYGDE2g+3hITwJKk5clSZy39y
yZ3fNmphjNwmoOTgD0SJzwj25+h8axT9c7fgl553/wH91XkVd/8uSb8Nt9d4LfI970VNlNPMH9+w
duenpKX9wgjjgvuXV8i9M+ZNSw6vaufCRqYb/6D/8Xg/Zl/Ef46z/DufpGdTEdMy1RP6Ol0D0i79
/xrAuiFOqBuP8vjemqRC6TfMQSo4HLq/nIB5pNQZ69NR9h8CIiNnou9eXA17yv9rMfjUo3mi/Kmp
1jvXXv1//ZioaYdfMd8dTWpgX3cL494Agd2z0dXx4AxmpQt85nnb/xkmxfbll9uZz8+LKC9A2Pg3
6YLoiPel1MgTZvGJfNf843khn3g56Gf8q+d5yOMqtzbbLsDzrY5jt55fRhyE7vN4sd+AMvtDsLgg
n/DdJzFRRakGDPGPsATx/2tLzuwvq7+o9M95jZtweBuTvHT316+v/Ycs+7tFXx5qDf30f+by7GzB
8elLI0NzRu7flly3Y3qnY3zuX3kL6zzgaNBu9/r/jDrfnJMk6vaOA9ggNAdiPvw1dMlrKeba10NJ
2tn9cTY1/2c6wp5IsKPv1z9PVjJYiXWvzcRMgG3RvXfxIHktZahX//8Z6r2Hqui+9mP+d9Qh8R2B
V3mvhrSfTXVWX5gnSd9aDGj9/R8zFHlW+VvE+rnL8SzU8UTQdRjy3AEDfm5ZDDxNyOe/fHdwicOD
WuGcnpVb6m25RpJdmLOHnh/j63/5ofvr/z3e5MktLaBnnr8ot1iZ2d7reBubxH4tO4nvLw5Qvrvz
LpO3tNfSWmzbLbbmn58Sj+U9psKiA9FAAd6DbR7vQtThuFvhey94CWEjDHzWCqp+8znfK082RHKQ
x3MZ6gBTwpRjru7iQ6ecbePsF9Id/u8YP33rjf/sfDR6ibZFnTP0uHX/+sMLvYuFz9ptP/M6yVL/
Z7719VUN3XtMPt7hceXyHUWowdS947BXk0SfRc8HMfMKmv9ncqI7ehzCRkCSsUEn1JPVDPvRdy4d
wgxLLkH9OV9BGjpdoeswOT0AxLVZd6rCFnUMcCCa29FmaUI+kAKOhMNt6qvNf6c5htqHKoXjNQMA
KiPSAiAgAfwLMP6IAAKEAgcgQ97fCHMIOXRx4M5LBPDhAUjaPHgWARy9UWNAoj7Ar9XvH34x4/CN
9/19dY0HiFfyh/eS9+PQz8amKzmu7w4ApKems4qKeSu8lor9ad748EjZ1pU/H7cP8kqyka8X9//X
pZXHQ9l2/2v2wWBCxhJuhZJdVJZkXx7bZA/FYJgmjGZBq2WyjGm0L4+WR7unaCEmS1lSUYkiyVIj
6hmhtDwo5fqN531/v8/7/u5/rnOdc77nOuc7931d574/AyWODhQ9BYAHGAwejcYSCTgcAQXFH18f
xGPRaBU0WaIKNBZXflGxVlHRfvapRl3LFQ2UjAhaRhXka5LnzzV0Dsk1pFhQ2hYni7cTN5opmCGZ
51OmG3DoYHnjlSbf4R/e50IiQWiIG9B0dESZo3DmKHzFe+nJvab+NuZReJPyZ6J+c1uCiM6NWflB
l6Ont0RPF4XVAY4oDdrkmqKI8uJqXMpanUO751LwS5cuW2YADA2NjNilXfMtQEHJICxsI0ppTkx0
IymRVdS0tMESKN7jqosYrDB9uNbR7begaHoyb3vWOIS7c/IEooNHfz999mLFakdXT19qeDl7556c
Y3+E6f/oK6sCt+oaG5fCtzGlPaAV2ILVYA1YO3tRssbTJdjNx0cxC5iB4DBKrL6c/1AI787VaYum
cI0QBg94srdQWJtnXsdoJ9C26+L1OYSZzWvGDVPlTNhES7wYfb12FZl4gOpJvxaytSsOHEyuIrxI
PclRZWCSlXlJK/rJOit+7n8E3Oe0vlV7WvA2rWITF2fma/l32O8+Kt1d4Ehft93p09lMox8uXLqu
V9PIk44zvmQq0YW4n14dT1WgIS/VCm7uCx9ba2xInxm44aNTsjlq+8ZNuNSAaHO7BIACZKL8zUnG
+VizEstMdITty8EaVuoNY9yfeFNqfPJ50I4CxL5tq964UrDr0UQi+oNC1j3bNsTE0mzZ8n2paZkR
MA/dE8BAeyR50VzcguNrU+fqOc4EZfIiFdXFTJwNwd5p9OlnF0yisjKByEj2C/7w0KyOu82TSNSO
azKVEEPtCltBsSUMRtwUMSs9zrTHFpLXklCh7+KcCW6MFi95S1O5OM1DNpiAKRfe8Gi0ouvO+MsU
9ZOhvp0Wbvvsb53rC0ygp2+JD9tVmPuCoutS3h+gsm3ymUcmF6lK79N/6Opqdbi/3DbVjRNvyAsL
ERETV/IPHUUPet8Poj9CirehpH0ltMDte2np3tVylr1dwO2yvNu2witllCVaB/XWAMv2zSa3Y7uf
p11UjU3PdW24s/UB7TBP372DzYu70oPKLO65pN1cYyb3oJLlZeYy6JoJelSVzCayJUdSfDhKzBPY
E16DIStTlgRky6GdNFGrh3HqhTqTv5PqXWlpK+JmxohKDskRyXHZIaktlX/NGLfuZ6/OVd11Bjks
TNwrmegyMrT18VQnaXchW3dZmlnQO7u0t3kJp6PPDpB8c7ceTmsJLUoofv+42PblnTJUVc1suZGJ
5jEkLl6XnXY8Fh08kZDpyFfwPEcLLrkT3LDXg9Tsa6yx6QGi4KmGTssM7Gqn5746tulBfoYmnbv9
QTWGKy+6LHn9BJuSznJ9FWeFeQF6jrQsGZpmyBvlbB/zCdXer/6M+d10W/qL8ia1zOOXY1uM/n7l
95mB67hjvYd+8XlkU2xBpsYqpvuSY4tJSjq9RpZijl9+ANWpI/Ts0cK0w35/mnmwDVEfaeGfFYNv
2O7ycS42jHLFPKUj+Wit9hMpe42dTx+630NWib2Baqed9FiGhKfvLC6KJ7ESrOJ9KtJi1TilrWPP
y2NvfdyH3dqQR9G+fWUqJUSxqzL85nU2u0puGe9M2h+BaDMgPnCBc6HMf5Z8Glv7KYdFtI8TPiJz
d4CToZ8k31q0nvupKXV+jmOqWDcqKhFeGCW/hU+rw7PyOcYcl4R0f6UXuf2xNcF35ovfn8tvrVXs
fOvcoctTbZEkyxmqnF/pv43hbv5X2WX7krGP2VFBnsF78zzjawpuYhezQVgZrz9X7UDlgEmqEuMB
9lVhMa1TEtdfkLtntvZ6XpPn5oC4GOzrdmC39e9Fay2XrUVeBKwr1PA7EpoJ9k+XLibrzRVIFid6
qOviMxJ8SmkUA3Oibr0PXuHQLpp5a6gfo93cXe6C6rYSNc7142SG6mr/Lakf0PpzErovLTkyMlmO
lLxxKtl5iW4DTn3P+cxrncmUjODPyYNEfm7GLK3ay9JEb9zMqCH3c1Xwro9V/dHNld6PLgb3Wpe9
P7LNfH/0i5F1qWYcyqyJEV//ieauHP/l8RtVf4y6Z7BXn9yWpvSZ4KfbcowbQVmtvibZ2737d/dk
2qJKTUdK+jn/Gu/Er5k6R5sSKS53lxwwjSAYpa1bIXBTCEuCJnWxdlcMNYkuZZrHW+kXAx52NZF0
vhGVoyqT2A3fJtVtWDaSNBe92Pz4MJsc/R0+DccswzX6ti9zM60pr70hJFs14s7dZ9CPyv/SiDjM
kLM4lVh05nzSJI1w0nBndY//CO1pxLTrieXflz09t9lFfu12rtLFB49s9zsRb73vt97GPKF3rER4
aPGNLuXz7GdNncn+pU7RfaOJROB9YOAYtSQ2jL2c+Dgt0IahQak7+Iyv5c7+EPlbX4amIW1HgFbd
Usmbr1EJlsHUyaOkgyGcIv6BMfVfJhEefQYBt7mj8Q5bo07eOaxR8SXtaAy9AJ3LD7XWvkAw6fex
CDzrLlnhIfLU3bSo1yBu3YeuAZtb/MgyaWInCL/x55qJWBXfqFsll8YiwQpM2/PHl7aUDL6f+oE9
qbTwZUBjqelqp26Z5B0UnZgmU/jvyN13pbJuwdb6pOdY6YJt8P3H6XkZjk48pbLEyLJEJtq7+oXF
vh4bvLRF+gV/+jwG6bZ5fEmdeYq9I3chSNGhU5eu1Tj4d3OT7j542usgQ1xzxwRExCUvgNOzi46c
KdsiEysbZItUyqTX0qlZ9Bn/btsOkjqisNLGabzkmoPezwxrb+UgjAxySZbdjr0L4OIT58prmhbA
j3oG3o5VySJO/f0D9QdxYV1ldR0DE+uF5O2c3H7bkEcgN2VlwxyvecPsfsgXPsmHBbAQCrL05SkK
UDjiXwz3wwMwfbgl72AMPKwNjw7A41d7I89+hr9D+yf9ffm3p8TikFJ4th6ehxfgxSx42WIwNdoB
/qkNrz6GFUJ4Ha5MzRq+AW/6wip4C1YPQPHttHjj17WrYH0UvNOS17jpPGwch82wBd5Lgvevpw2n
j8CHV2H7HHzMgR0bIgob62En7ILPbnUrrJGHCv298CXsg69g/+IYOPQgZlZhP3wDJXBYD47AUfiO
JYKuw/CvP4zhBzgOJ+Dkzjztzy3w0yr4GSLwm52sGO0COCOG39fDuZOM3Mpt8NdhuAGt2Fe4XBOn
QKgrY2Bb5YDubEzZG7YiUALKTxcBFaAK1MBimNeHhFK2agItIOtiQF6Z4awO0AV6mfpDxyQQLgOy
zgcYgeVgBTAGK4EJMD1ijomgAqd5e5X7Fq3AoXodcALr690vuk7RQJxd8wZjZso34DtW+FuzZ8FW
Bflm1663zWlHmz27zg2N88QMrML6jMNLd4DciaipKBANEBADYtfDmMKsnWAX2A32ZGeDHJAL+GAv
yAPQLBfmgwJQCASgKH8fEIFisB/IXmbgAXAQHAKHwRFwFBwDx8EJ8DsoASfBKXAanAF/mFwRnSOc
1Zc5Zp0HF8BFcAlcBmXgT3AFXAXloAJcA9fBDXATBLlkjVeCKiGoBjVAPPdNAUzVgrrIBnAH3G2F
rMWbm0AzaAH3ZN3ZffAAwka1lDbQDh6Bx+AJ6ABPQSfoAs/A84W+HHSDHvAC9IKXoA+8Av1gAAyC
IfAavAESMAze9iz4jIBR8A68B38BKRgDH8A4mACT4COQvZM0bhj7DL6Ar+Ab+BtMgxkwC76DH2Cu
PO3QPPgJfoF5oFKKLlTY03bYsQ3nfIqENNzV7Mj6GohtbNcutjlTnWZSVnH/9JYXMfNr4KVO6KOX
2dIt2TeW3e5Vd7jMpfj199NZ3xt/zo3vubLB7eFBpUvXSjVHzntSSLycfB58B7+kt3v7v9u07v4m
B+GZQ5EnjyNF6KDY1tFKynND3iWMswIBn+CMV9Zb7ywbs6BMHnVx81rQBcZkufPRodjmeFXFMkOv
nS4Bn14qzuX/vWJrakAuxokIgKKDS3agkx8W/HNlQ3cMANYoANCyGe1j1gFwYymemJ2DJZZiVLfr
i4cgoOaqxuam5T9OyWCTCw4QbBCHG4VNnwZecjFdUwoo7b2jOCIGP53PkweY9apzjXrTyqrJO1hp
wX6RgVYRZnZAqe/lHgQBzpgo2VIoUJ+61VdLG43p4YeGTJ9JqXSjrJetjnOYrn1HRvT3YghSBYfR
JrQlwqAMiQ9cy8QwVhlguPqr/L8KccTf7/5K390ckFBARinHFIZMm00kYU4Wxj8yr8xA64GJeJrx
RF69NkVxpNotm+KL4TxeX5lDwCQk4YllSOmk/YxtAl/OgRBCaUJ2Hnv/XLnIL6nwpyC8hF61zm8p
hRqbJKHS5ptPYzDKrkU4IrFmvKSiWy8ie1W0wq10SvbG9KEh7VELZ8GJf5GHoCAo7wDA5ikA0iQZ
geNKkREbj8cWhRggYx9lDrFCroGBcJeBKb8DjisfWLiK8J6FBC/UAI8g1OV//LlooIjAHr8t4x/8
h25RE2qA84kn9+8pZ9SqnS/FsBWbJUQZnj8FuemC+//p3/5fEx7i8v+M/xWM34zO+QW+Qsh/hhKp
WfB//SzCKqcrFaFdZZn58GbGNsseDyGa34IVEPgPsfwxbMZnVOeY94I2xMAAE2Kwwq6Xi/I2kRUm
U1gK7AXuBkto9wTz41ZFMgTKMZst77g/A5+dNa/MxZtKHVEZS9dSeLpCQmKzlNjcRM5vNZ5FTf1v
9H8wb/v4Y6isJtrxDdSge/LAwCc1kWWPhDC2cJDELcl0RDam0eK30hOQjC1cBsJl0JFQagRCz6TH
87i0OJnHP2Y2wuBy0+wtLHhpmeacRPNUOhcxWAiXYP+P/ypzy7WIGyttO3tLEoOLrHAzRqzs7Fab
WVtaWf1f0BA6LcUccUlORoIWvDhIEJ1DZ6fTE8wXYjF1IqU6st+LGhQYEuMR4eG26x8pPMgnxANJ
pMmSTTCX38hERzGxER3ZVVlMlGzYLPKYEa41nhW0YJy9cVAZyHbzKQiZKKpVOxMyravJTMIGqmCE
GSlzdsAo5x2TsX3o8M8KqmCYuiGogloh45gUBl/wW03vbZJWyKzeeAAScbJHx0ceDIghDA82bq/L
lN2FgveCv4wHpGI5INqIVXyWkQHhB80m42aRQWvec97XnAkieelf0hO6D/FI1swQC6wTrKoDqqIA
rFRZzSKNzYq34KTMQUhPTrSQ0QuimKGynPKaMrEPsCjFzc6QBR8D+WBhsLfoiBbUk895u1BJkHFT
Ih/CIbiTjOGhDlFF/H1eQg2hN5bfTBKFGJBivaQsrBjOsFT4rQYbo3JanSOb35JQ3dTA1Y7Xtb+s
n7vO+0mlooxf3YiCyyuuZsyxkrJ4PwreoN7IheP5swgHo1zVoiQ7cKIUeFP8T0RUu/PXq95Xrwrm
Y+4Fi6INiMxIGQmfrkfdA+FhgqfBgnt2RGEdR56Z1YbaCuGa8fm8di6xiOBViHZ9O4viYI1XdduJ
Jy/uDBW8s3jQzZ8Qe6zaTdo5UwAFHoN8e6C8t9YrdLSWRBBMSS+Oz2TNpEBkVOgvPaia8112j+jz
lPlOxllNNf77x7ExvBS75owvDpgBFmsg4yOBE+MhEaEhb4IKMzY07s6Yg+lYGBrXqubVKQyVOMTz
HhZ5TE43DsO9TdyleZCrm/MDPMnA2qnyVJRrmqiYgYGaOY+P1CA7+NKGnCtFeTbg6x2HWsblBdhj
SqEYlFJN/Vy6cVeQADPmoiKif3vdfo+cNwQhV7EMxZOzI3MJ/Cbi+IxVkwpGcDZKnJL0I3ja+8zo
d4FAWHOnemiT2k8hj1gh2NQVRlokzqLNSTDpS7T6RjLJaGVxsCkkC1iNknneE+UjTVbw1GqBf0+z
pPAz/ymEvVPQN1PP0y5FRbiBaPeFgxb0Yjao2b38tT+LQ6xtRnx5IznvrY1nBF9Ef4gUL593ch0h
qDOXwd49c21LmZBKCA6Tms2rfdAQtIiwWsaigZraL/uz205M8feY1u1+iL/n0ZrCHUKF7x4EXKSm
XtxY8YS/Fs/TKGmrw2hjvIkqfeL3q9WM8GOlhg/17FGwd/6Tct4Rhd806p3jNnBKlrbxhwkYT2IK
ZwA2D6OFQeTuccGrRKnKlomeX3t2DjcJAxQeoJrlGgTNOcPlxk/7zFkwVeQoz11fIfRoYlrfQg8U
D77+2wtDrdr+y+5B0hCfFpVZmz2C7eKhU6mEwR8NUx/TR0VR0v3LIiLHl4U+Ebol1E+aV7RRKUwV
bFP4erua3fD26bTZmT5D3jQDfXBg8vH0ba7KrttYJ67c+9sPKM4K8FXf2x+/dg/xoqZdNVDc7meN
U4qsx2rCjeCJiKSFWeRAJfOUhjyN+wRt/h6HFlltHWkRs37a8HdgqX4YEs8Zta6SxdiRRbvHxT9E
EVYoOls950tu1kx/nJfWHxd8FXy3mdpNhjW4fdx3i28fpoiW4B02NVTwClo4uuqqncaKOKfp1XMN
wdKEnz2R2uOPrqmKAjG2M891Aoq+45hRnl31dzas0H4R7WLKr0bs5COKP93ctwxloZITQ+ewLtbL
TTHNzCk9vX4srqRNo0eAd/78XeNemgghf3u1I/heSt/uJVd4NkRRuM07SBuQot2Q16GjIg9EFKoR
NjdyWeHo97k+Uh0uTBxoiowPRnltmGgxkDbeXbRSKTk5VVyH1ZFeI7uOMlZ03Hl2ynei30Z6VCOo
pC5fPPVNKvjxYmIz9dTvqVXrJtTr5J42h0qkyYtsTTX6E4UJi2MSt+x5PVn3IpC/gseMsmofETzC
RLzUXlTzGMfMyVxcpzhVp1DxNIeJUV2nKF9Rm5PDY4+em3GK+ouVybxg4D+VJp6bZAcSpt86MPHW
JNfuhGE6rXnLaMuuXFuJbK+3BN4AQDXZbqsE/tUqOWdnHwD/1wZEy1Sj/7a8nxyAwJzD4HDZsoNJ
buEfwTguPbOq925tIgnHZiXQtAi4zkEWpDNiEtm0FLpcPJfF5qjjmfG43nQxG5vEso3jKIGONf4Z
CstR+CPZWOKRJdkkawS1qM1gTceNG6ZBKHLEDMaaMvgcRxxcRglMfYNgdNCk7uUguA05Hj3hhRga
Hn+3CoPgS0vR7O+YdJPAUllXV3oOC9Yg/Gis7YXl2STE/wLRMZuk31Fe7patqD90K+OW9/K7B2w6
bt/eh/GPNieMfSLGLnM/r8G8S6866guWBwGAJSepyEqEC2Uv7IL/ImBBUiShiW9/Bv35Leco/X/Z
KL2IArLDKTVbJv8PGitODw==
'''
_XZ_EXE_ = \
'''DISABLED'''

# Main part of the module
class StreamError(Exception):
  def __cinit__(self, errno, errinfo=''):
    '''StreamError(errno, errinfo)
    
    Error while operating with stream.
    
    1: need stream or string, <object> found
    2: cannot convert char <char> to ustream
    3: fstream path doesn't exist: <path>
    4: fstream path is a directory: <path> 
    5: unknown fstream mode: <mode>
    6: unknown fstream filter: <filter>
    7: cannot filter through <filter> filter
    8: cannot perform I/O method <method>
    9: pstream returned non-zero exit status <code>
    10: need pstream object, <object> found'''
    self.errno = (errno -1)
    self.errinfo = errinfo
    self._errlist_ = \
    [
      "need stream or string, %s found" % repr(self.errinfo),
      "cannot convert char %s to ustream" % repr(self.errinfo),
      "fstream path doesn't exist: %s" % repr(self.errinfo),
      "fstream path is a directory: %s" % repr(self.errinfo),
      "unknown fstream mode: %s" % repr(self.errinfo),
      "unknown fstream filter: %s" % repr(self.errinfo),
      "cannot filter through %s filter" % repr(self.errinfo),
      "cannot perform I/O method %s" % repr(self.errinfo),
      "pstream returned non-zero exit status %s" % repr(self.errinfo),
      "need pstream object, %s found" % repr(self.errinfo),
    ]
    
  def __str__(self):
    return(self._errlist_[self.errno])

cdef class bstream:
  '''A special stream to operate with binary data. This is similar
  to default bytes type, so it was called bstream.
  All methods which can be used with default bytes type can be
  used with ustream class. bstream also has several advantages
  in comparison with the bytes type, like a possibility to use
  regular expressions in some of the methods. It also adds methods
  to encode or decode it using zlib or base64 encoders.
  And the greatest advantage of the ustream is a possibility to
  perform all actions with four types of strings: str, unicode,
  QByteArray and QString.'''
  cdef readonly object _stream_
  cdef readonly int _cursor_
  cdef object __weakref__
  
  def __cinit__(self, string=''):
    try:
      from PyQt4 import QtCore
      types = \
      [
        str, unicode, bstream, ustream,
        QtCore.QByteArray, QtCore.QString
      ]
    except ImportError, error:
      types = [str, unicode, bstream, ustream]
    if type(string) not in types:
      raise StreamError(1, type(string).__name__)
    if isinstance(string, bytes):
      self._stream_ = string
    elif isinstance(string, bstream):
      self._stream_ = string._stream_
    elif isinstance(string, QtCore.QByteArray):
      self._stream_ = bytes(string)
    elif isinstance(string, unicode):
      self._stream_ = string.encode(__encoding__)
    elif isinstance(string, ustream):
      self._stream_ = string._stream_.encode(__encoding__)
    elif isinstance(string, QtCore.QString):
      self._stream_ = unicode(string).encode(__encoding__)
    self._cursor_ = 0
    
  def __add__(self, y):
    '''x.__add__(y) <==> x+y'''
    x = bstream(self)._stream_
    y = bstream(y)._stream_
    result = x.__add__(y)
    return(bstream(result))
    
  def __bool__(self):
    '''x.__bool__() <==> bool(x)'''
    if self._stream_ != '':
      result = True
    else: # if self._stream_ == '':
      result = False
    return(bool(result))
    
  def __contains__(self, y):
    '''x.__contains__(y) <==> y in x'''
    result = self._stream_.__contains__(bstream(y)._stream_)
    return(bool(result))
    
  def __getitem__(self, i):
    '''x.__getitem__(i) <==> x[i]'''
    result = self._stream_.__getitem__(i)
    return(bstream(result))
    
  def __getslice__(self, i, j):
    '''x.__getslice__(i, j) <==> x[i:j]'''
    result = self._stream_.__getslice__(i, j)
    return(bstream(result))
    
  def __hash__(self):
    '''x.__hash__() <==> hash(x)'''
    result = self._stream_.__hash__()
    return(int(result))
    
  def __int__(self):
    '''x.__int() <==> int(x)'''
    if self._stream_.isdecimal():
      result = int(self._stream_)
      return(int(result))
    else:
      message = 'invalid literal for int() with base 10: '
      raise TypeError(message +repr(ustream))
    
  def __iter__(self):
    cdef int line, last
    cursor = self._cursor_
    if '\r\n' in self:
      lines = self[cursor:].split('\r\n')
    elif '\n' in self:
      lines = self[cursor:].split('\n')
    elif '\r' in self:
      lines = self[cursor:].split('\r')
    else:
      lines = [self[cursor:]]
    _temp_ = len(lines) -1
    last = _temp_
    for line from 0 <= line <= last:
      yield lines[line]
    self._cursor_ = len(self)
    
  def __len__(self):
    '''x.__len__() <==> len(x)'''
    result = self._stream_.__len__()
    return(int(result))
    
  def __mod__(self, int y):
    '''x.__mod__(y) <==> x%y'''
    result = self._stream_.__mod__(y)
    return(bstream(result))
    
  def __mul__(self, n):
    '''x.__mul__(n) <==> x*n'''
    result = self._stream_.__mul__(n)
    return(bstream(result))
    
  def __richcmp__(bstream self, string, operand):
    if operand == 0: # __lt__ <==> <
      result = self._stream_.__lt__(bstream(string)._stream_)
    elif operand == 1: # __le__ <==> <=
      result = self._stream_.__le__(bstream(string)._stream_)
    elif operand == 2: # __eq__ <==> ==
      result = self._stream_.__eq__(bstream(string)._stream_)
    elif operand == 3: # __ne__ <==> !=
      result = self._stream_.__ne__(bstream(string)._stream_)
    elif operand == 4: # __gt__ <==> >
      result = self._stream_.__gt__(bstream(string)._stream_)
    elif operand == 5: # __ge__ <==> >=
      result = self._stream_.__ge__(bstream(string)._stream_)
    return(bool(result))
    
  def __repr__(self):
    '''x.__repr__() <==> repr(x)'''
    length = len(self._stream_)
    cursor = self._cursor_
    return('<bstream object, length=%d, cursor=%d>' % (length, cursor))
    
  def __str__(self):
    '''x.__str__() <==> str(x)'''
    return(self._stream_)
    
  def __sizeof__(self):
    '''bstream.__sizeof__() -> size of bstream in memory, in bytes'''
    result = self._stream_.__sizeof__()
    return(int(result))
    
  def __unicode__(self):
    '''x.__unicode__() <==> unicode(x)'''
    return(self._stream_.decode(__encoding__))
    
  cpdef bstream base64_encode(self):
    '''bstream.base64_encode() -> bstream
    
    Encode bstream using base64 encoder. Returns bstream object.'''
    from base64 import encodestring as encode
    result = encode(self._stream_)
    return(bstream(result))
    
  cpdef bstream base64_decode(self):
    '''bstream.base64_decode() -> bstream
    
    Decode bstream using base64 decoder. Returns bstream object.'''
    from base64 import decodestring as decode
    result = decode(self._stream_)
    return(bstream(result))
    
  cpdef bstream capitalize(self):
    '''bstream.capitalize() -> bstream
    
    Return a capitalized version of bstream, i.e. make the first character
    have upper case and the others lower case.'''
    result = self._stream_.capitalize()
    return(bstream(result))
    
  cpdef bstream center(self, int width, fillchar=' '):
    '''bstream.center(width[, fillchar]) -> bstream
    
    Return bstream centered in a bstream of length width. Padding is
    done using the specified fill character (default is a space)'''
    result = self._stream_.center(width, bstream(fillchar)._stream_)
    return(bstream(result))
    
  cpdef int count(self, sub, int start=0, int end=-1):
    '''bstream.count(sub[, start[, end]]) -> int
    
    Return the number of non-overlapping occurrences of substring sub
    in bstream[start:end]. Optional arguments start and end are
    interpreted as in slice notation.'''
    result = self._stream_.count(bstream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef bint endswith(self, suffix, start=0, end=-1):
    '''bstream.endswith(suffix[, start[, end]]) -> bool
    
    Return True if bstream ends with the specified suffix, False otherwise.
    With optional start, test bstream beginning at that position.
    With optional end, stop comparing bstream at that position.'''
    result = self._stream_.count(bstream(suffix)._stream_, start, end)
    return(bool(result))
    
  cpdef bstream expandtabs(self, int tabsize=8):
    '''bstream.expandtabs([tabsize]) -> bstream
    
    Return a copy of bstream where all tab characters are expanded using spaces.
    If tabsize is not given, a tab size of 8 characters is assumed.'''
    result = self._stream_.expandtabs(tabsize)
    return(bstream(result))
    
  cpdef int find(self, sub, int start=0, int end=-1):
    '''bstream.find(sub [,start [,end]]) -> int
    
    Return the lowest index in bstream where substring sub is found,
    such that sub is contained within bstream[start:end]. Optional
    arguments start and end are interpreted as in slice notation.
    
    Return -1 on failure.'''
    result = self._stream_.find(bstream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef int index(self, sub, int start=0, int end=-1):
    '''bstream.index(sub [,start [,end]]) -> int
    
    Like bstream.find() but raise ValueError when the substring is not found.'''
    result = self._stream_.index(bstream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef insert(self, string):
    '''bstream.insert([string])
    
    Insert string or stream to current position. This function do exactly
    the same as bstream.write(string), but doesn't overwrites characters.'''
    cursor = self._cursor_
    string = bstream(string)._stream_
    length = len(string)
    head = self._stream_[:cursor]
    tail = self._stream_[cursor:]
    result = bstream('%s%s%s' % (head, string, tail))
    self._stream_ = result._stream_
    self._cursor_ += length
    
  cpdef bint isalnum(self):
    '''bstream.isalnum() -> bool
    
    Return True if all characters in bstream are alphanumeric and
    there is at least one character in bstream, False otherwise.'''
    result = self._stream_.isalnum()
    return(bool(result))
    
  cpdef bint isalpha(self):
    '''bstream.isalpha() -> bool
    
    Return True if all characters in bstream are alphabetic and
    there is at least one character in bstream, False otherwise.'''
    result = self._stream_.isalpha()
    return(bool(result))
    
  cpdef bint isdigit(self):
    '''bstream.isdigit() -> bool
    
    Return True if all characters in bstream are digits and
    there is at least one character in bstream, False otherwise.'''
    result = self._stream_.isdigit()
    return(bool(result))
    
  cpdef bint islower(self):
    '''bstream.islower() -> bool
    
    Return True if all cased characters in bstream are lowercase and there is
    at least one cased character in bstream, False otherwise.'''
    result = self._stream_.islower()
    return(bool(result))
    
  cpdef bint isspace(self):
    '''bstream.isspace() -> bool
    
    Return True if all characters in bstream are whitespace and there is
    at least one character in bstream, False otherwise.'''
    result = self._stream_.isspace()
    return(bool(result))
    
  cpdef bint istitle(self):
    '''bstream.istitle() -> bool
    
    Return True if bstream is a titlecased string and there is at least one
    character in bstream, i.e. upper- and titlecase characters may only
    follow uncased characters and lowercase characters only cased ones.
    Return False otherwise.'''
    result = self._stream_.istitle()
    return(bool(result))
    
  cpdef bint isupper(self):
    '''bstream.isupper() -> bool
    
    Return True if all cased characters in bstream are uppercase and there is
    at least one cased character in bstream, False otherwise.'''
    result = self._stream_.isupper()
    return(bool(result))
    
  def iter_chars(self):
    '''bstream.iter_chars() -> iterator
    
    Return iterator which operates with characters.'''
    char = int()
    last = len(self) -1
    while char <= last:
      yield self[char]._stream_
      char += 1
    
  def iter_ints(self):
    '''bstream.iter_ints() -> iterator
    
    Return iterator which operates with integers.'''
    char = int()
    last = len(self) -1
    while char <= last:
      yield ord(self[char]._stream_)
      char += 1
    
  cpdef bstream join(self, iterable):
    '''bstream.join(iterable) -> bstream
    
    Return a bstream which is the concatenation of the strings in the
    iterable. The separator between elements is bstream.'''
    result = self._stream_.join(bstream(iterable)._stream_)
    return(bstream(result))
    
  cpdef bstream ljust(self, int width, fillchar=' '):
    '''bstream.ljust(width[, fillchar]) -> int
    
    Return bstream left-justified in a bstream of length width. Padding
    is done using the specified fill character (default is a space).'''
    result = self._stream_.ljust(width, bstream(fillchar)._stream_)
    return(bstream(result))
    
  cpdef bstream lower(self):
    '''bstream.lower() -> bstream
    
    Return a copy of the bstream converted to lowercase.'''
    result = self._stream_.lower()
    return(bstream(result))
    
  cpdef bstream lstrip(self, chars=' '):
    '''bstream.lstrip([chars]) -> bstream
    
    Return a copy of the bstream with leading whitespace removed.
    If chars is given and not None, remove characters in chars instead.'''
    result = self._stream_.lstrip(bstream(chars)._stream_)
    return(bstream(result))
    
  cpdef list partition(self, sep):
    '''bstream.partition(sep) -> (head, sep, tail)
    
    Search for the separator sep in bstream, and return the part before it,
    the separator itself, and the part after it. If the separator is not
    found, return sep and two empty bstreams.'''
    result = self._stream_.partition(bstream(sep)._stream_)
    result = [bstream(item) for item in result]
    return(list(result))
    
  cpdef bstream read(self, int size=-1):
    '''bstream.read([n]) -> bstream
    
    Read n characters, returned as a bstream. If the argument is negative
    or omitted, read until EOF is reached.'''
    cursor = self._cursor_
    source = self._stream_[cursor:]
    length = len(source)
    if size < 0 or size > length:
      size = length
    self._cursor_ = cursor + size
    result = source[:size]
    return(result)
    
  cpdef bstream remove(self, string):
    '''bstream.remove(string) -> bstream
    
    Delete all the occurrences of the string in the bstream.'''
    result = self._stream_.replace(bstream(string)._stream_, '')
    return(bstream(result))
    
  cpdef bstream replace(self, old, new, int count=-1):
    '''bstream.replace(old, new[, count]) -> bstream
    
    Return a copy of bstream with all occurrences of substring
    old replaced by new. If the optional argument count is
    given, only the first count occurrences are replaced.'''
    old = ustream(old)._stream_
    new = ustream(new)._stream_
    result = self._stream_.replace(old, new, count)
    return(bstream(result))
    
  cpdef int rfind(self, sub, int start=0, int end=-1):
    '''bstream.rfind(sub [,start [,end]]) -> int
    
    Return the highest index in bstream where substring sub is found,
    such that sub is contained within s[start:end]. Optional arguments
    start and end are interpreted as in slice notation.
    
    Return -1 on failure.'''
    result = self._stream_.rfind(bstream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef int rindex(self, sub, int start=0, int end=-1):
    '''bstream.rindex(sub [,start [,end]]) -> int
    
    Like bstream.rfind() but raise ValueError when the substring is not found.'''
    result = self._stream_.rindex(bstream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef bstream rjust(self, int width, fillchar=' '):
    '''bstream.rjust(width[, fillchar]) -> bstream
    
    Return bstream right-justified in a bstream of length width. Padding
    is done using the specified fill character (default is a space).'''
    result = self._stream_.rjust(width, bstream(fillchar)._stream_)
    return(bstream(result))
    
  cpdef list rpartition(self, sep):
    '''bstream.rpartition(sep) -> (head, sep, tail)
    
    Search for the separator sep in bstream, starting at the end of bstream,
    and return  the part before it, the separator itself, and the part after
    it. If the separator is not found, return two empty strings and bstream.'''
    result = self._stream_.rpartition(bstream(sep)._stream_)
    result = [bstream(item) for item in result]
    return(list(result))
    
  cpdef list rsplit(self, sep=' ', int maxsplit=-1):
    '''bstream.rsplit([sep [,maxsplit]]) -> list of bstreams
    
    Return a list of the words in bstream, using sep as the delimiter
    string, starting at the end of the string and working to the front.
    If maxsplit is given, at most maxsplit splits are done. If sep
    is not specified, any whitespace string is a separator.'''
    result = self._stream_.rsplit(bstream(sep)._stream_, maxsplit)
    result = [bstream(item) for item in result]
    return(list(result))
    
  cpdef seek(self, int offset, int whence=STREAM_BEG):
    '''bstream.seek(offset[, whence])
    
    Move to new bstream position. Argument offset is a byte count. Optional
    argument whence defaults to STREAM_BEG (offset from start of bstream,
    offset should be >= 0, if not, position remains unchanged); other values
    are STREAM_CUR (move relative to current position, positive or negative),
    and STREAM_END (move relative to end of bstream, should be negative, if
    not, position remains unchanged).'''
    cursor = self._cursor_
    length = len(self._stream_)
    if whence == STREAM_BEG:
      self._cursor_ = whence +offset
    elif whence == STREAM_CUR:
      self._cursor_ = cursor +offset
    elif whence == STREAM_END:
      self._cursor_ = length +offset
    if self._cursor_ < 0:
      self._cursor_ = 0
    elif self._cursor_ > length:
      self._cursor_ = length
    
  cpdef list split(self, sep=' ', int maxsplit=-1):
    '''bstream.split([sep [,maxsplit]]) -> list of bstreams
    
    Return a list of the words in bstream, using sep as the
    delimiter string. If maxsplit is given, at most maxsplit
    splits are done. If sep is not specified or is None, any
    whitespace string is a separator and empty strings are
    removed from the result.'''
    result = self._stream_.split(bstream(sep)._stream_, maxsplit)
    result = [bstream(item) for item in result]
    return(list(result))
    
  cpdef list splitlines(self, bint keepends):
    '''bstream.splitlines([keepends]) -> list of bstreams
    
    Return a list of the lines in bstream, breaking at line boundaries.
    Line breaks are not included in the resulting list unless keepends
    is given and true.'''
    result = self._stream_.splitlines(keepends)
    result = [bstream(item) for item in result]
    return(list(result))
    
  cpdef bint startswith(self, prefix, int start=0, int end=-1):
    '''bstream.startswith(prefix[, start[, end]]) -> bool
    
    Return True if bstream starts with the specified prefix,
    False otherwise. With optional start, test bstream beginning
    at that position. With optional end, stop comparing bstream
    at that position.'''
    result = self._stream_.startswith(bstream(prefix)._stream_, start, end)
    return(bool(result))
    
  cpdef bstream strip(self, chars=' '):
    '''bstream.strip([chars]) -> bstream
    
    Return a copy of the bstream with leading and trailing whitespace
    removed. If chars is given and not None, remove characters in chars
    instead.'''
    result = self._stream_.strip(bstream(chars)._stream_)
    return(bstream(result))
    
  cpdef bstream swapcase(self):
    '''bstream.swapcase() -> bstream
    
    Return a copy of bstream with uppercase characters converted to
    lowercase and vice versa.'''
    result = self._stream_.swapcase()
    return(bstream(result))
    
  cpdef int tell(self):
    '''bstream.tell() -> int
    
    Return current bstream position.'''
    return(int(self._cursor_))
    
  cpdef bstream title(self):
    '''bstream.title() -> bstream
    
    Return a titlecased version of bstream, i.e. words start with title
    case characters, all remaining cased characters have lower case.'''
    result = self._stream_.title()
    return(bstream(result))
    
  cpdef bstream upper(self):
    '''bstream.upper() -> bstream
    
    Return a copy of bstream converted to uppercase.'''
    result = self._stream_.upper()
    return(bstream(result))
    
  cpdef write(self, string):
    '''bstream.write([string])
    
    Insert string or stream to current position. This function do exactly
    the same as bstream.insert(string), but overwrites characters.'''
    cursor = self._cursor_
    string = bstream(string)._stream_
    length = len(string)
    head = self._stream_[:cursor]
    tail = self._stream_[cursor:]
    result = bstream('%s%s%s' % (head, string, tail[length:]))
    self._stream_ = result._stream_
    self._cursor_ += length
    
  cpdef bstream zfill(self, int width):
    '''bstream..zfill(width) -> bstream
    
    Pad a numeric bstream with zeros on the left, to fill a field
    of the specified width. The bstream is never truncated.'''
    result = self._stream_.zfill(width)
    return(bstream(result))
    
  cpdef bstream zlib_encode(self):
    '''bstream.zlib_encode() -> bstream
    
    Encode bstream using zlib encoder. Returns bstream object.'''
    from zlib import compress as encode
    result = encode(self._stream_)
    return(bstream(result))
    
  cpdef bstream zlib_decode(self):
    '''bstream.zlib_decode() -> bstream
    
    Decode bstream using zlib decoder. Returns bstream object.'''
    from zlib import decompress as decode
    result = decode(self._stream_)
    return(bstream(result))

cdef class ustream:
  '''A special stream to operate with Unicode data. This is similar
  to default unicode type, so it was called ustream.
  All methods which can be used with default unicode type can be
  used with ustream class. ustream also has several advantages
  in comparison with the unicode type, like a possibility to use
  regular expressions in some of the methods.
  And the greatest advantage of the ustream is a possibility to
  perform all actions with four types of strings: str, unicode,
  QByteArray and QString.'''
  cdef readonly object _stream_
  cdef readonly int _cursor_
  cdef object __weakref__
  
  def __cinit__(self, string=''):
    try:
      from PyQt4 import QtCore
      types = \
      [
        str, unicode, bstream, ustream,
        QtCore.QByteArray, QtCore.QString
      ]
    except ImportError, error:
      types = [str, unicode, bstream, ustream]
    if type(string) not in types:
      raise StreamError(1, type(string).__name__)
    try:
      if isinstance(string, unicode):
        self._stream_ = string
      elif isinstance(string, ustream):
        self._stream_ = string._stream_
      elif isinstance(string, QtCore.QString):
        self._stream_ = unicode(string)
      elif isinstance(string, bytes):
        self._stream_ = string.decode(__encoding__)
      elif isinstance(string, bstream):
        self._stream_ = string._stream_.decode(__encoding__)
      elif isinstance(string, QtCore.QByteArray):
        self._stream_ = bytes(string).decode(__encoding__)
    except UnicodeDecodeError, error:
      raise StreamError(2, error.start)
    
  def __add__(self, y):
    '''x.__add__(y) <==> x+y'''
    x = bstream(self)._stream_
    y = bstream(y)._stream_
    result = x.__add__(y)
    return(bstream(result))
    
  def __bool__(self):
    '''x.__bool__() <==> bool(x)'''
    if self._stream_ != '':
      result = True
    else: # if self._stream_ == '':
      result = False
    return(bool(result))
    
  def __contains__(self, y):
    '''x.__contains__(y) <==> y in x'''
    result = self._stream_.__contains__(ustream(y)._stream_)
    return(bool(result))
    
  def __getitem__(self, i):
    '''x.__getitem__(i) <==> x[i]'''
    result = self._stream_.__getitem__(i)
    return(ustream(result))
    
  def __getslice__(self, i, j):
    '''x.__getslice__(i, j) <==> x[i:j]'''
    result = self._stream_.__getslice__(i, j)
    return(ustream(result))
    
  def __hash__(self):
    '''x.__hash__() <==> hash(x)'''
    result = self._stream_.__hash__()
    return(int(result))
    
  def __int__(self):
    '''x.__int() <==> int(x)'''
    if self._stream_.isdecimal():
      result = int(self._stream_)
      return(int(result))
    else:
      message = 'invalid literal for int() with base 10: '
      raise TypeError(message +repr(ustream))
    
  def __iter__(self):
    cdef int line, last
    cursor = self._cursor_
    if '\r\n' in self:
      lines = self[cursor:].split('\r\n')
    elif '\n' in self:
      lines = self[cursor:].split('\n')
    elif '\r' in self:
      lines = self[cursor:].split('\r')
    else:
      lines = [self[cursor:]]
    _temp_ = len(lines) -1
    last = _temp_
    for line from 0 <= line <= last:
      yield lines[line]
    self._cursor_ = len(self)
    
  def __len__(self):
    '''x.__len__() <==> len(x)'''
    result = self._stream_.__len__()
    return(int(result))
    
  def __mod__(self, y):
    '''x.__mod__(y) <==> x%y'''
    result = self._stream_.__mod__(y)
    return(ustream(result))
    
  def __mul__(self, n):
    '''x.__mul__(n) <==> x*n'''
    result = self._stream_.__mul__(n)
    return(ustream(result))
    
  def __richcmp__(ustream self, string, operand):
    if operand == 0: # __lt__ <==> <
      result = self._stream_.__lt__(ustream(string)._stream_)
    elif operand == 1: # __le__ <==> <=
      result = self._stream_.__le__(ustream(string)._stream_)
    elif operand == 2: # __eq__ <==> ==
      result = self._stream_.__eq__(ustream(string)._stream_)
    elif operand == 3: # __ne__ <==> !=
      result = self._stream_.__ne__(ustream(string)._stream_)
    elif operand == 4: # __gt__ <==> >
      result = self._stream_.__gt__(ustream(string)._stream_)
    elif operand == 5: # __ge__ <==> >=
      result = self._stream_.__ge__(ustream(string)._stream_)
    return(bool(result))
    
  def __repr__(self):
    '''x.__repr__() <==> repr(x)'''
    length = len(self._stream_)
    cursor = self._cursor_
    return('<ustream object, length=%d, cursor=%d>' % (length, cursor))
    
  def __str__(self):
    '''x.__str__() <==> bytes(x)'''
    return(self._stream_.encode(__encoding__))
    
  def __sizeof__(self):
    '''ustream.__sizeof__() -> size of ustream in memory, in bytes'''
    result = self._stream_.__sizeof__()
    return(int(result))
    
  def __unicode__(self):
    '''x.__unicode__ <==> unicode(x)'''
    return(self._stream_)
    
  def array_chars(self):
    '''ustream.array_chars() -> list of ustreams
    
    Return iterator which operates with characters.'''
    char = int()
    last = len(self) -1
    while char <= last:
      yield self[char]._stream_
      char += 1
    
  def array_ints(self):
    '''ustream.array_ints() -> list of integers
    
    Return iterator which operates with integers.'''
    char = int()
    last = len(self) -1
    while char <= last:
      yield ord(self[char]._stream_)
      char += 1
    
  cpdef ustream capitalize(self):
    '''ustream.capitalize() -> ustream
    
    Return a capitalized version of ustream, i.e. make the first character
    have upper case and the others lower case.'''
    result = self._stream_.capitalize()
    return(ustream(result))
    
  cpdef ustream center(self, int width, fillchar=' '):
    '''ustream.center(width[, fillchar]) -> ustream
    
    Return ustream centered in a ustream of length width. Padding is
    done using the specified fill character (default is a space)'''
    result = self._stream_.center(width, ustream(fillchar)._stream_)
    return(ustream(result))
    
  cpdef int count(self, sub, int start=0, int end=-1):
    '''ustream.count(sub[, start[, end]]) -> int
    
    Return the number of non-overlapping occurrences of substring sub
    in ustream[start:end]. Optional arguments start and end are
    interpreted as in slice notation.'''
    result = self._stream_.count(ustream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef bint endswith(self, suffix, int start=0, int end=-1):
    '''ustream.endswith(suffix[, start[, end]]) -> bool
    
    Return True if ustream ends with the specified suffix, False otherwise.
    With optional start, test ustream beginning at that position.
    With optional end, stop comparing ustream at that position.'''
    result = self._stream_.count(ustream(suffix)._stream_, start, end)
    return(bool(result))
    
  cpdef ustream expandtabs(self, int tabsize=8):
    '''ustream.expandtabs([tabsize]) -> ustream
    
    Return a copy of ustream where all tab characters are expanded using spaces.
    If tabsize is not given, a tab size of 8 characters is assumed.'''
    result = self._stream_.expandtabs(tabsize)
    return(ustream(result))
    
  cpdef int find(self, sub, int start=0, int end=-1):
    '''ustream.find(sub [,start [,end]]) -> int
    
    Return the lowest index in ustream where substring sub is found,
    such that sub is contained within ustream[start:end]. Optional
    arguments start and end are interpreted as in slice notation.
    
    Return -1 on failure.'''
    result = self._stream_.find(ustream(sub)._stream_, start, end)
    return(int(result))
    
  def format(self, *args, **kwargs):
    '''ustream.format(*args, **kwargs) -> ustream
    
    Return a formatted version of ustream, using substitutions from args
    and kwargs. The substitutions are identified by braces ('{' and '}').'''
    result = self._stream_.format(*args, **kwargs)
    return(ustream(result))
    
  cpdef int index(self, sub, int start=0, int end=-1):
    '''ustream.index(sub [,start [,end]]) -> int
    
    Like ustream.find() but raise ValueError when the substring is not found.'''
    result = self._stream_.index(ustream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef insert(self, string):
    '''ustream.insert([string])
    
    Insert string or stream to current position. This function do exactly
    the same as ustream.write(string), but doesn't overwrites characters.'''
    cursor = self._cursor_
    string = ustream(string)._stream_
    length = len(string)
    head = self._stream_[:cursor]
    tail = self._stream_[cursor:]
    result = ustream('%s%s%s' % (head, string, tail))
    self._stream_ = result._stream_
    self._cursor_ += length
    
  cpdef bint isalnum(self):
    '''ustream.isalnum() -> bool
    
    Return True if all characters in ustream are alphanumeric and
    there is at least one character in ustream, False otherwise.'''
    result = self._stream_.isalnum()
    return(bool(result))
    
  cpdef bint isalpha(self):
    '''ustream.isalpha() -> bool
    
    Return True if all characters in ustream are alphabetic and
    there is at least one character in ustream, False otherwise.'''
    result = self._stream_.isalpha()
    return(bool(result))
    
  cpdef bint isdecimal(self):
    '''ustream.isdecimal() -> bool
    
    Return True if there are only decimal characters in ustream,
    False otherwise.'''
    result = self._stream_.isdecimal()
    return(bool(result))
    
  cpdef bint isdigit(self):
    '''ustream.isdigit() -> bool
    
    Return True if all characters in ustream are digits and
    there is at least one character in ustream, False otherwise.'''
    result = self._stream_.isdigit()
    return(bool(result))
    
  cpdef bint islower(self):
    '''ustream.islower() -> bool
    
    Return True if all cased characters in ustream are lowercase and there is
    at least one cased character in ustream, False otherwise.'''
    result = self._stream_.islower()
    return(bool(result))
    
  cpdef bint isnumeric(self):
    '''ustream.isnumeric() -> bool
    
    Return True if there are only numeric characters in ustream,
    False otherwise.'''
    result = self._stream_.isnumeric()
    return(bool(result))
    
  cpdef bint isspace(self):
    '''ustream.isspace() -> bool
    
    Return True if all characters in ustream are whitespace and there is
    at least one character in ustream, False otherwise.'''
    result = self._stream_.isspace()
    return(bool(result))
    
  cpdef bint istitle(self):
    '''ustream.istitle() -> bool
    
    Return True if ustream is a titlecased string and there is at least one
    character in ustream, i.e. upper- and titlecase characters may only
    follow uncased characters and lowercase characters only cased ones.
    Return False otherwise.'''
    result = self._stream_.istitle()
    return(bool(result))
    
  cpdef bint isupper(self):
    '''ustream.isupper() -> bool
    
    Return True if all cased characters in ustream are uppercase and there is
    at least one cased character in ustream, False otherwise.'''
    result = self._stream_.isupper()
    return(bool(result))
    
  cpdef ustream join(self, iterable):
    '''ustream.join(iterable) -> ustream
    
    Return a ustream which is the concatenation of the strings in the
    iterable. The separator between elements is ustream.'''
    result = self._stream_.join(ustream(iterable)._stream_)
    return(ustream(result))
    
  cpdef ustream ljust(self, width, fillchar=' '):
    '''ustream.ljust(width[, fillchar]) -> int
    
    Return ustream left-justified in a ustream of length width. Padding
    is done using the specified fill character (default is a space).'''
    result = self._stream_.ljust(width, ustream(fillchar)._stream_)
    return(ustream(result))
    
  cpdef ustream lower(self):
    '''ustream.lower() -> ustream
    
    Return a copy of the ustream converted to lowercase.'''
    result = self._stream_.lower()
    return(ustream(result))
    
  cpdef ustream lstrip(self, chars=' '):
    '''ustream.lstrip([chars]) -> ustream
    
    Return a copy of the ustream with leading whitespace removed.
    If chars is given and not None, remove characters in chars instead.'''
    result = self._stream_.lstrip(ustream(chars)._stream_)
    return(ustream(result))
    
  cpdef list partition(self, sep):
    '''ustream.partition(sep) -> (head, sep, tail)
    
    Search for the separator sep in ustream, and return the part before it,
    the separator itself, and the part after it. If the separator is not
    found, return sep and two empty ustreams.'''
    result = self._stream_.partition(ustream(sep)._stream_)
    result = [ustream(item) for item in result]
    return(list(result))
    
  cpdef ustream read(self, int size=1):
    '''ustream.read([n]) -> ustream
    
    Read n characters, returned as a ustream. If the argument is negative
    or omitted, read until EOF is reached.'''
    cursor = self._cursor_
    source = self._stream_[cursor:]
    length = len(source)
    if size < 0 or size > length:
      size = length
    self._cursor_ = cursor + size
    result = source[:size]
    return(result)
    
  cpdef list regexfind(self, pattern):
    '''ustream.regexfind(pattern) -> list of ustreams
    
    Return a list of all non-overlapping matches in the streams. If one or
    more groups are present in the pattern, return a list of groups; this
    will be a list of tuples if the pattern has more than one group. Empty
    matches are included in the result.
    Function returns None if no matches were found.'''
    result = list()
    pattern = ustream(pattern)
    pattern = re.compile\
      (pattern._stream_, re.UNICODE | re.DOTALL | re.IGNORECASE)
    groups = pattern.findall(self._stream_)
    for group in groups:
      index = groups.index(group)
      if isinstance(group, unicode):
        result.append(ustream(group))
      elif isinstance(group, tuple):
        result.append(tuple([ustream(item) for item in group]))
    return(list(result))
    
  cpdef ustream regexmake(self, lang):
    '''ustream.regexmake(langtable) -> ustream
    
    Convert ustream to use in methods which support regular expressions.
    After supplying this method user receives a modified ustream, where
    some symbols are replaced to their equivalents.
    Argument `lang` must be a string or stream which represents language.
    You can get all available languages by running ustream._tables_.
    
    >>> ustream('barbarus').regexmake()
    ustream('b[a|ă|ā]rb[a|ă|ā]r[u|ŭ|ū|v]s')
    >>> ustream('мерило').regexmake()
    ustream('м[е|ѣ|ё]р[и|і|i]ло')'''
    result = self._stream_
    if lang in TABLES:
      table = TABLES[lang]
    for items in table:
      for item in items:
        if item == items[0]:
          regex = ('[%s|' % item)
        elif item == items[-1]:
          regex += ('%s]' % item)
        else:
          regex += ('%s|' % item)
        result = result.replace(item, items[0])
      result = result.replace(items[0], regex)
    return(ustream(result))
    
  cpdef bint regexmatch(self, pattern):
    '''ustream.regexmatch(pattern) -> bool
    
    Returns True if ustream contains regular expression.'''
    pattern = ustream(pattern)
    pattern = re.compile\
      (pattern._stream_, re.UNICODE | re.DOTALL | re.IGNORECASE)
    result = pattern.findall(self._stream_)
    return(bool(result))
    
  cpdef ustream regexreplace(self, pattern, repl):
    '''ustream.regexreplace(pattern, repl) -> ustream
    
    Return the ustream obtained by replacing the leftmost non-overlapping
    occurrences of the pattern in ustream by the replacement.'''
    pattern = ustream(pattern)
    pattern = re.compile\
      (pattern._stream_, re.UNICODE | re.DOTALL | re.IGNORECASE)
    result = pattern.sub(ustream(repl)._stream_, self._stream_)
    return(ustream(result))
    
  cpdef ustream remove(self, string):
    '''bstream.remove(string) -> bstream
    
    Delete all the occurrences of the string in the bstream.'''
    result = self._stream_.replace(ustream(string)._stream_, '')
    return(ustream(result))
    
  cpdef ustream replace(self, old, new, int count=-1):
    '''ustream.replace(old, new[, count]) -> ustream
    
    Return a copy of ustream with all occurrences of substring
    old replaced by new. If the optional argument count is
    given, only the first count occurrences are replaced.'''
    old = ustream(old)._stream_
    new = ustream(new)._stream_
    result = self._stream_.replace(old, new, count)
    return(ustream(result))
    
  cpdef int rfind(self, sub, int start=0, int end=-1):
    '''ustream.rfind(sub [,start [,end]]) -> int
    
    Return the highest index in ustream where substring sub is found,
    such that sub is contained within s[start:end]. Optional arguments
    start and end are interpreted as in slice notation.
    
    Return -1 on failure.'''
    result = self._stream_.rfind(ustream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef int rindex(self, sub, int start=0, int end=-1):
    '''ustream.rindex(sub [,start [,end]]) -> int
    
    Like ustream.rfind() but raise ValueError when the substring is not found.'''
    result = self._stream_.rindex(ustream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef ustream rjust(self, width, fillchar=' '):
    '''ustream.rjust(width[, fillchar]) -> ustream
    
    Return ustream right-justified in a ustream of length width. Padding
    is done using the specified fill character (default is a space).'''
    result = self._stream_.rjust(width, ustream(fillchar)._stream_)
    return(ustream(result))
    
  cpdef list rpartition(self, sep):
    '''ustream.rpartition(sep) -> (head, sep, tail)
    
    Search for the separator sep in ustream, starting at the end of ustream,
    and return  the part before it, the separator itself, and the part after
    it. If the separator is not found, return two empty strings and ustream.'''
    result = self._stream_.rpartition(ustream(sep)._stream_)
    result = [ustream(item) for item in result]
    return(list(result))
    
  cpdef list rsplit(self, sep=' ', maxsplit=-1):
    '''ustream.rsplit([sep [,maxsplit]]) -> list of ustreams
    
    Return a list of the words in ustream, using sep as the
    delimiter string, starting at the end of the string and
    working to the front. If maxsplit is given, at most maxsplit
    splits are done. If sep is not specified, any whitespace string
    is a separator.'''
    result = self._stream_.rsplit(ustream(sep)._stream_, maxsplit)
    result = [ustream(item) for item in result]
    return(list(result))
    
  cpdef ustream rstrip(self, chars=None):
    '''ustream.rstrip([chars]) -> ustream
    
    Return a copy of the ustream with trailing whitespace removed.
    If chars is given and not None, remove characters in chars instead.'''
    result = self._stream_.rstrip(ustream(chars)._stream_)
    return(ustream(result))
    
  cpdef seek(self, int offset, int whence=STREAM_BEG):
    '''ustream.seek(offset[, whence])
    
    Move to new ustream position. Argument offset is a byte count. Optional
    argument whence defaults to STREAM_BEG (offset from start of ustream,
    offset should be >= 0, if not, position remains unchanged); other values
    are STREAM_CUR (move relative to current position, positive or negative),
    and STREAM_END (move relative to end of ustream, should be negative, if
    not, position remains unchanged).'''
    cursor = self._cursor_
    length = len(self._stream_)
    if whence == STREAM_BEG:
      self._cursor_ = whence +offset
    elif whence == STREAM_CUR:
      self._cursor_ = cursor +offset
    elif whence == STREAM_END:
      self._cursor_ = length +offset
    if self._cursor_ < 0:
      self._cursor_ = 0
    elif self._cursor_ > length:
      self._cursor_ = length
    
  cpdef list split(self, sep=' ', maxsplit=-1):
    '''ustream.split([sep [,maxsplit]]) -> list of ustreams
    
    Return a list of the words in ustream, using sep as the
    delimiter string. If maxsplit is given, at most maxsplit
    splits are done. If sep is not specified or is None, any
    whitespace string is a separator and empty strings are
    removed from the result.'''
    result = self._stream_.split(ustream(sep)._stream_, maxsplit)
    result = [ustream(item) for item in result]
    return(list(result))
    
  cpdef bint startswith(self, prefix, int start=0, int end=-1):
    '''ustream.startswith(prefix[, start[, end]]) -> bool
    
    Return True if ustream starts with the specified prefix,
    False otherwise. With optional start, test ustream beginning
    at that position. With optional end, stop comparing ustream
    at that position.'''
    result = self._stream_.startswith(ustream(prefix)._stream_, start, end)
    return(bool(result))
    
  cpdef ustream strip(self, chars=' '):
    '''ustream.strip([chars]) -> ustream
    
    Return a copy of the ustream with leading and trailing whitespace
    removed. If chars is given and not None, remove characters in chars
    instead.'''
    result = self._stream_.strip(ustream(chars)._stream_)
    return(ustream(result))
    
  cpdef ustream swapcase(self):
    '''ustream.swapcase() -> ustream
    
    Return a copy of ustream with uppercase characters converted to
    lowercase and vice versa.'''
    result = self._stream_.swapcase()
    return(ustream(result))
    
  cpdef int tell(self):
    '''ustream.tell() -> int
    
    Return current ustream position.'''
    return(int(self._cursor_))
    
  cpdef ustream title(self):
    '''ustream.title() -> ustream
    
    Return a titlecased version of ustream, i.e. words start with title case
    characters, all remaining cased characters have lower case.'''
    result = self._stream_.title()
    return(ustream(result))
    
  cpdef ustream upper(self):
    '''ustream.upper() -> ustream
    
    Return a copy of ustream converted to uppercase.'''
    result = self._stream_.upper()
    return(ustream(result))
    
  cpdef write(self, string):
    '''ustream.write([string])
    
    Insert string or stream to current position. This function do exactly
    the same as ustream.insert(string), but overwrites characters.'''
    cursor = self._cursor_
    string = ustream(string)._stream_
    length = len(string)
    head = self._stream_[:cursor]
    tail = self._stream_[cursor:]
    result = ustream('%s%s%s' % (head, string, tail[length:]))
    self._stream_ = result._stream_
    self._cursor_ += length
    
  cpdef ustream zfill(self, width):
    '''ustream.zfill(width) -> ustream
    
    Pad a numeric ustream with zeros on the left, to fill a field
    of the specified width. The ustream is never truncated.'''
    result = self._stream_.zfill(width)
    return(ustream(result))

cdef class fstream:
  cdef readonly object path
  cdef readonly object mode
  cdef readonly object filter
  cdef readonly object changed
  cdef readonly object closed
  cdef readonly object _stream_
  cdef readonly object _tempdir_
  cdef readonly object _temppath_
  cdef readonly object _gzip_exe_
  cdef readonly object _bzip2_exe_
  cdef readonly object _bzip2_lib_
  cdef readonly object _xz_exe_
  cdef readonly object _lzop_exe_
  cdef object __weakref__
  
  def __cinit__(self, path, mode='rb', filter=None):
    '''fstream(path[, mode[, filter]]) -> fstream
    
    Create fstream. The mode can be 'r', 'w' or 'a' for reading (default)
    writing or appending. The file will be created if it doesn't exist
    when opened for writing or appending; it will be truncated when
    opened for writing. Add a 'b' to the mode for binary files.
    Also fstream can be also filtered through compressors/decompressors.
    Available filters are 'gzip', 'bzip2', 'lzop', 'xz' and 'lzma'.'''
    modes = ['rb', 'wb', 'ab', 'r', 'w', 'a']
    filters = ['gzip', 'bzip2', 'lzop', 'xz', 'lzma']
    self.path = ustream(path)
    self.mode = ustream(mode)
    if filter == 'gz':
      filter = ustream('gzip')
    elif filter == 'bz2':
      filter = ustream('bzip2')
    elif filter == 'lzo':
      filter = ustream('lzop')
    if not isinstance(filter, NoneType):
      self.filter = ustream(filter)
    else: # if isinstance(filter, NoneType)
      self.filter = None
    self.changed = False
    self.closed = False
    self._tempdir_ = ustream(tempfile.mkdtemp())
    if __system__ == 'linux':
      self._temppath_ = ustream('%s/file' % self._tempdir_)
      self._gzip_exe_ = ustream('/usr/bin/gzip')
      self._bzip2_exe_ = ustream('/usr/bin/bzip2')
      self._lzop_exe_ = ustream('%s/lzop' % self._tempdir_)
      self._xz_exe_ = ustream('/usr/bin/xz')
      with codecs.open(self._lzop_exe_._stream_, 'wb') as file:
        file.write(bstream(_LZOP_EXE_).base64_decode().zlib_decode()._stream_)
      command = ustream('chmod u+x %s' % _screen_(self._lzop_exe_))
      pstream(command).execute()
    elif __system__ == 'windows':
      self._temppath_ = ustream('%s\\file' % self._tempdir_)
      self._gzip_exe_ = ustream('%s\\gzip.exe' % self._tempdir_)
      self._bzip2_exe_ = ustream('%s\\bzip2.exe' % self._tempdir_)
      self._bzip2_lib_ = ustream('%s\\bzip2.dll' % self._tempdir_)
      self._lzop_exe_ = ustream('%s\\lzop.exe' % self._tempdir_)
      self._xz_exe_ = ustream('%s\\xz.exe' % self._tempdir_)
      with codecs.open(self._gzip_exe_._stream_, 'wb') as file:
        file.write(bstream(_GZIP_EXE_).base64_decode().zlib_decode()._stream_)
      with codecs.open(self._bzip2_exe_._stream_, 'wb') as file:
        file.write(bstream(_BZIP2_EXE_).base64_decode().zlib_decode()._stream_)
      with codecs.open(self._bzip2_lib_._stream_, 'wb') as file:
        file.write(bstream(_BZIP2_LIB_).base64_decode().zlib_decode()._stream_)
      with codecs.open(self._xz_exe_._stream_, 'wb') as file:
        file.write(bstream(_XZ_EXE_).base64_decode().zlib_decode()._stream_)
      with codecs.open(self._lzop_exe_._stream_, 'wb') as file:
        file.write(bstream(_LZOP_EXE_).base64_decode().zlib_decode()._stream_)
    with codecs.open(self._temppath_._stream_, 'wb') as file:
      file.write('') # Create empty temporary file
    if self.mode == 'rb' or self.mode == 'ab' \
      or self.mode == 'r' or self.mode == 'a':
      source = _screen_(self.path)
      output = _screen_(self._temppath_)
      if not os.path.exists(self.path._stream_):
        self._rmtemp_() # Clear temporary directory
        raise StreamError(3, self.path._stream_)
      elif os.path.isdir(self.path._stream_):
        self._rmtemp_() # Clear temporary directory
        raise StreamError(4, self.path._stream_)
      try:
        if isinstance(self.filter, NoneType):
          shutil.copy(self.path._stream_, self._temppath_._stream_)
        elif self.filter == 'gzip':
          arc_exe = _screen_(self._gzip_exe_)
          command = '%s -dc %s > %s' % (arc_exe, source, output)
          pstream(command).execute()
        elif self.filter == 'bzip2':
          arc_exe = _screen_(self._bzip2_exe_)
          command = '%s -dc %s > %s' % (arc_exe, source, output)
          pstream(command).execute()
        elif self.filter == 'lzop':
          arc_exe = _screen_(self._lzop_exe_)
          command = '%s -dc %s > %s' % (arc_exe, source, output)
          pstream(command).execute()
        elif self.filter == 'xz':
          arc_exe = _screen_(self._xz_exe_)
          command = '%s --format=xz -dc %s > %s' % (arc_exe, source, output)
          pstream(command).execute()
        elif self.filter == 'lzma':
          arc_exe = _screen_(self._xz_exe_)
          command = '%s --format=lzma -dc %s > %s' % (arc_exe, source, output)
          pstream(command).execute()
      except StreamError, error:
        self._rmtemp_() # Clear temporary directory
        raise StreamError(7, bytes(self.filter))
    else: # self.mode == 'wb' or self.mode == 'w'
      with codecs.open(self.path._stream_, 'w') as file:
        file.write('')
    if 'b' in self.mode: # open fstream in binary mode
      self._stream_ = \
        codecs.open(self._temppath_._stream_, self.mode._stream_)
    else: # open fstream in string mode
      self._stream_ = \
        codecs.open(self._temppath_._stream_, self.mode._stream_, __encoding__)
    if self.mode not in modes:
      self._rmtemp_() # Clear temporary directory
      raise StreamError(5, self.mode._stream_)
    if not isinstance(filter, NoneType) \
      and self.filter not in filters:
        self._rmtemp_() # Clear temporary directory
        raise StreamError(6, self.filter._stream_)
      
  def __enter__(self):
    return(self) # Return initialized object
    
  def __exit__(self, exc_type, exc_val, exc_tb):
    self.close() # Close using internal function
  
  def __iter__(self):
    return(self)
  
  def __next__(self):
    '''x.next() -> the next value, or raise StopIteration
    
    Object which user receives after supplying this method is based on mode
    which is used to open the fstream. Function returns bstream object if
    fstream was opened in binary mode, otherwise it returns ustream.'''
    if not self.closed and 'b' in self.mode: # binary mode
      result = self._stream_.next()
      return(bstream(result))
    elif not self.closed and 'b' not in self.mode: # binary mode
      result = self._stream_.next()
      return(ustream(result))
    else:
      raise StreamError(8, '__next__')
    
  def __repr__(self):
    '''x.__repr__() <==> repr(x)'''
    path = repr(self.path._stream_)
    mode = repr(self.mode._stream_)
    cursor = self.tell()
    if not isinstance(self.filter, NoneType):
      filter = repr(self.filter._stream_)
    else:
      filter = 'None'
    if not self.closed:
      return('<opened fstream object, path=%s, mode=%s, filter=%s, cursor=%d>'
        % (path, mode, filter, cursor))
    else:
      return('<closed fstream object, path=%s, mode=%s, filter=%s, cursor=%d>'
        % (path, mode, filter, cursor))
    
  cpdef _rmtemp_(self):
    os.remove(self._temppath_._stream_)
    os.remove(self._lzop_exe_._stream_)
    if __system__ == 'windows':
      os.remove(self._gzip_exe_._stream_)
      os.remove(self._bzip2_exe_._stream_)
      os.remove(self._bzip2_lib_._stream_)
      os.remove(self._xz_exe_._stream_)
    os.rmdir(self._tempdir_._stream_)
    
  cpdef close(self):
    '''fstream.close(stream)
    
    Close opened fstream. fstream.close() method may be called more than once
    without error; the first time it is called it removes all the temporary
    files and temporary directory.'''
    self.closed = True
    self._stream_.close()
    if self.changed:
      source = _screen_(self._temppath_)
      output = _screen_(self.path)
      if isinstance(self.filter, NoneType):
        shutil.copy(self._temppath_._stream_, self.path._stream_)
      elif self.filter == 'gzip':
        arc_exe = _screen_(self._gzip_exe_)
        command = '%s -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'bzip2':
        arc_exe = _screen_(self._bzip2_exe_)
        command = '%s -zc %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'lzop':
        arc_exe = _screen_(self._lzop_exe_)
        command = '%s -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'xz':
        arc_exe = _screen_(self._xz_exe_)
        command = '%s --format=xz -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'lzma':
        arc_exe = _screen_(self._xz_exe_)
        command = '%s --format=lzma -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      self._rmtemp_() # Clear temporary directory
    
  cpdef flush(self):
    '''fstream.flush()
    
    Flush the internal I/O buffer.'''
    if not self.closed and self.changed:
      self._stream_.flush()
      source = _screen_(self._temppath_)
      output = _screen_(self.path)
      if isinstance(self.filter, NoneType):
        shutil.copy(self._temppath_._stream_, self.path._stream_)
      elif self.filter == 'gzip':
        arc_exe = _screen_(self._gzip_exe_)
        command = '%s -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'bzip2':
        arc_exe = _screen_(self._bzip2_exe_)
        command = '%s -zc %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'lzop':
        arc_exe = _screen_(self._lzop_exe_)
        command = '%s -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'xz':
        arc_exe = _screen_(self._xz_exe_)
        command = '%s --format=xz -zc %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'lzma':
        arc_exe = _screen_(self._xz_exe_)
        command = '%s --format=lzma -zc %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
    
  cpdef read(self, int size=-1):
    '''fstream.read([size]) -> bstream or ustream
    
    If the size argument is negative or omitted, read until EOF is reached.
    Notice that when in non-blocking mode, less data than what was requested
    may be returned, even if no size parameter was given.
    Object which user receives after supplying this method is based on mode
    which is used to open the fstream. Function returns bstream object if
    fstream was opened in binary mode, otherwise it returns ustream.'''
    if not self.closed:
      result = self._stream_.read(size)
      if 'b' in self.mode: # binary mode
        return(bstream(result))
      else: # string mode
        return(ustream(result))
    else:
      raise StreamError(8, 'read')
    
  def readline(self, size=-1):
    '''fstream.readline([size]) -> next line from the fstream
    
    Retain newline. A non-negative size argument limits the maximum
    number of bytes to return (an incomplete line may be returned then).
    Return an empty stream at EOF.
    Object which user receives after supplying this method is based on mode
    which is used to open the fstream. Function returns bstream object if
    fstream was opened in binary mode, otherwise it returns ustream.'''
    if not self.closed:
      result = self._stream_.readline(size)
      if 'b' in self.mode: # binary mode
        return(bstream(result))
      else: # string mode
        return(ustream(result))
    else:
      raise StreamError(8, 'readline')
    
  def readlines(self, size=-1):
    '''fstream.readlines([size]) -> list of streams
    
    Call fstream.readline() repeatedly and return a list of the lines to
    read. The optional size argument, if given, is an approximate bound
    on the total number of bytes in the lines returned.
    Object which user receives after supplying this method is based on mode
    which is used to open the fstream. Function returns bstream object if
    fstream was opened in binary mode, otherwise it returns ustream.'''
    if not self.closed:
      result = self._stream_.readlines(size)
      if 'b' in self.mode: # binary mode
        result = [bstream(item) for item in result]
        return(list(result))
      else: # string mode
        result = [ustream(item) for item in result]
        return(list(result))
    else:
      raise StreamError(8, 'readlines')
    
  def seek(self, offset, whence=0):
    '''fstream.seek(offset[, whence])
    
    Move to new fstream position. Argument offset is a byte count. Optional
    argument whence defaults to 0 (offset from start of fstream, offset should
    be >= 0); other values are 1 (move relative to current position, positive
    or negative), and 2 (move relative to end of fstream, usually negative,
    although many platforms allow seeking beyond the end of a fstream).
    If the fstream is opened in text mode, only offsets returned by tell() are
    legal. Use of other offsets causes undefined behavior.'''
    if not self.closed:
      self._stream_.seek(offset, whence)
    else:
      raise StreamError(8, 'seek')
    
  def tell(self):
    '''tell() -> current fstream position as integer'''
    if not self.closed:
      result = self._stream_.tell()
      return(int(result))
    else:
      raise StreamError(8, 'tell')
    
  def truncate(self, size=-1):
    '''fstream.truncate([size])
    
    Truncate the fstream to at most size bytes. Size defaults to the current
    fstream position, as returned by fstream.tell().'''
    if not self.closed:
      if size == -1:
        size = self._stream_.tell()
      self._stream_.truncate(size)
    else:
      raise StreamError(8, 'truncate')
    
  def write(self, stream):
    '''fstream.write(stream)
    
    Write stream to fstream. Note that due to buffering, flush() or close()
    may be needed before the fstream on disk reflects the data written.
    Due to some encoding problems for Windows, this method was designed
    to receive only bstream and ustream objects, which have no problems
    with encoding on all platforms.'''
    if not self.closed:
      if 'b' in self.mode:
        self._stream_.write(bstream(stream)._stream_)
      elif 'b' not in self.mode:
        self._stream_.write(ustream(stream)._stream_)
      self.changed = True
    else:
      raise StreamError(8, 'write')
    
  def writelines(self, sequence):
    '''fstream.writelines(sequence)
    
    Write the list of streams to the fstream. Note that newlines are not added.
    The sequence must be a list of strings and/or streams.'''
    if not self.closed:
      if 'b' in self.mode: # binary mode
        result = [bstream(item)._stream_ for item in sequence]
      else: # string mode
        result = [ustream(item)._stream_ for item in sequence]
      self._stream_.writelines(result)
      self.changed = True
    else:
      raise StreamError(8, 'writelines')

cdef class pstream:
  '''A pipeline stream to run an external applications. Subprocess can be
  used to execute an applications or to get stdout as bstream.
  This class was created as an alternative to default Python subprocess
  module which has some problems with encoding on Windows. It doesn't have
  all the possibilities of the original module, but yet can be useful.'''
  cdef readonly object _command_
  
  def __cinit__(self, args):
    '''pstream.__cinit__(args) -> pstream
    
    Create pstream. This function receives one argument, which must be a full
    command or a list of streams which are later used as options.'''
    try:
      from PyQt4 import QtCore
      types = \
      [
        str, unicode, bstream, ustream,
        QtCore.QByteArray, QtCore.QString
      ]
    except ImportError, error:
      types = [str, unicode, bstream, ustream]
    if isinstance(args, list):
      args = [ustream(arg) for arg in args]
      for arg in args:
        if arg == args[0]:
          self._command_ = \
            '%s' % arg._stream_.encode(__console__)
        else:
          self._command_ += \
            ' %s' % arg._stream_.encode(__console__)
    if type(args) in types:
      self._command_ = \
        ustream(args)._stream_.encode(__console__)
    
  def __repr__(self):
    '''x.__repr__() <==> repr(x)'''
    return('<pstream object, command=%s>' % repr(self._command_))
    
  def pipeline(self, process):
    '''pstream.pipeline(process) -> pstream
    
    Connect pstream with another pstream through pipeline. Programs can be run
    together such that one program reads the output from another with no need
    for an explicit intermediate file.
    
    >>> pstream('echo one\\ntwo\\nfour\\nfive').pipeline(pstream('grep f'))
    <pstream 'echo one\\ntwo\\nfour\\nfive | grep f'>'''
    if not isinstance(process, pstream):
      raise StreamError(10, type(process).__name__)
    result = '%s | %s' % (self._command_, process.command)
    return(pstream(result))
    
  def execute(self, exception=True, stdin=''):
    '''pstream.execute([exception[, stdin]) -> bstream or tuple
    
    Try to execute the specified command. If command was run successfully,
    function returns stdout as a bstream object. If an error occurred and
    the exception flag is True (default), StreamError is raised, otherwise
    function returns tuple from return code and stderr as a bstream object.'''
    import subprocess
    pipe = subprocess.PIPE
    process = \
      subprocess.Popen(self._command_, shell=True,
      stdin=pipe, stdout=pipe, stderr=pipe)
    if stdin != '':
      stdout, stderr = process.communicate(bstream(stdin)._stream_)
    else:
      stdout, stderr = process.communicate()
    retcode = int(process.returncode)
    stdout = bstream(stdout)
    stderr = bstream(stderr)
    if __system__ == 'windows':
      stdout = stdout.replace('\r\n', '\n')
      stderr = stderr.replace('\r\n', '\n')
    if retcode == 0:
      return(bstream(stdout))
    elif retcode != 0 and exception:
      raise StreamError(9, retcode)
    elif retcode != 0 and not exception:
      return(tuple([int(retcode), bstream(stderr)]))
