# coding: UTF-8

'''Module to operate with different kinds of streams.'''

# Define global imports
import os
import re
import sys
import codecs
import shutil
import inspect
import tempfile
import platform
import __main__
from io import BytesIO
from io import StringIO
from libc.stdlib cimport *

# Repair Windows Unicode
codecs.register(lambda name:
  name == 'cp65001' and codecs.lookup('UTF-8') or None)

# Define module information
__author__ = 'Dmitriy Selyutin'
__license__ = 'GNU GPL v3'
__console__ = sys.getfilesystemencoding()
if not hasattr(__main__, '__file__'):
  __encoding__ = sys.stdout.encoding
else: # if hasattr(__main__, '__file__'):
  __encoding__ = 'UTF-8'
__system__ = platform.system().lower()
__version__ = 0.2
__test__ = {}

# Define global functions
def _screen_(path):
  '''_screen_(path) -> ustream
  
  Return screened version of the path to use with pstream.'''
  path = ustream(path)
  if __system__ == 'linux':
    path = ustream("'%s'" % path)
  elif __system__ == 'windows':
    path = path.replace('^', '^^')
    path = path.replace(' ', '^ ')
    path = path.replace('&', '^&')
    path = path.replace('|', '^|')
    path = path.replace('(', '^(')
    path = path.replace(')', '^)')
    path = path.replace('<', '^<')
    path = path.replace('>', '^>')
  return(ustream(path))

# Define global constants
NoneType = type(None)
STREAM_BEG = 0
STREAM_CUR = 1
STREAM_END = 2
TABLE_GR = \
[ # Greek table
  ['α', 'ἀ', 'ἁ', 'ά', 'ὰ', 'ᾶ', 'ἄ', 'ἅ', 'ἂ', 'ἅ', 'ἆ', 'ἇ', 'ά',
    'ᾳ', 'ᾀ', 'ᾁ', 'ᾴ', 'ᾲ', 'ᾷ', 'ᾄ', 'ᾅ', 'ᾂ', 'ᾅ', 'ᾆ', 'ᾇ'],
  ['η', 'ἠ', 'ἡ', 'ή', 'ὴ', 'ῆ', 'ἤ', 'ἥ', 'ἢ', 'ἣ', 'ἦ', 'ἧ', 'ή',
    'ῃ', 'ᾐ', 'ᾑ', 'ῄ', 'ῂ', 'ῇ', 'ᾔ', 'ᾕ', 'ᾒ', 'ᾓ', 'ᾖ', 'ᾗ'],
  ['ω', 'ὠ', 'ὡ', 'ώ', 'ὼ', 'ῶ', 'ὤ', 'ὥ', 'ὢ', 'ὣ', 'ὦ', 'ὧ', 'ώ',
    'ῳ', 'ᾠ', 'ᾡ', 'ῴ', 'ῲ', 'ῷ', 'ᾤ', 'ᾥ', 'ᾢ', 'ᾣ', 'ᾦ', 'ᾧ'],
  ['ι', 'ἰ', 'ἱ', 'ί', 'ὶ', 'ῖ', 'ἴ', 'ἵ', 'ἲ', 'ἳ', 'ἶ', 'ἷ', 'ί',
    'ῐ', 'ῑ', 'ϊ', 'ῒ', 'ΐ', 'ῗ', 'ΐ'],
  ['υ', 'ὐ', 'ὑ', 'ύ', 'ὺ', 'ῦ', 'ὔ', 'ὕ', 'ὒ', 'ὓ', 'ὖ', 'ὗ', 'ύ',
    'ῠ', 'ῡ', 'ϋ', 'ῢ', 'ΰ', 'ῧ', 'ΰ'],
  ['ε', 'ἐ', 'ἑ', 'έ', 'ὲ', 'ἔ', 'ἕ', 'ἒ', 'ἓ', 'έ'],
  ['ο', 'ὀ', 'ὁ', 'ό', 'ὸ', 'ὄ', 'ὅ', 'ὂ', 'ὃ', 'ό'],
  ['β', 'ϐ'],
  ['θ', 'ϑ'],
  ['κ', 'ϰ'],
  ['π', 'ϖ'],
  ['ρ', 'ϱ', 'ϼ'],
  ['σ', 'ς', 'ϲ'],
  ['ϕ', 'φ'],
] # Greek table
TABLE_LA = \
[ # Latin table
  ['a', 'ă', 'ā'],
  ['e', 'ĕ', 'ē'],
  ['o', 'ŏ', 'ō'],
  ['i', 'ĭ', 'ī', 'j'],
  ['u', 'ŭ', 'ū', 'v'],
] # Latin table
TABLE_RU = \
[ # Russian table
  ['ф', 'ѳ'],
  ['е', 'ѣ', 'ё'],
  ['и', 'і', 'ѵ'],
] # Russian table
TABLE_DE = \
[ # German table
  ['a', 'ä', 'ae'],
  ['o', 'ö', 'oe'],
  ['u', 'ü', 'ue'],
  ['ss', 'ß'],
] # German table
TABLE_GR = [[ustream(item) for item in row] for row in TABLE_GR]
TABLE_LA = [[ustream(item) for item in row] for row in TABLE_LA]
TABLE_RU = [[ustream(item) for item in row] for row in TABLE_RU]
TABLE_DE = [[ustream(item) for item in row] for row in TABLE_DE]
TABLES = \
{
  'GR': TABLE_GR,
  'LA': TABLE_LA,
  'RU': TABLE_RU,
  'DE': TABLE_DE,
}

# Delete temporary variables
del __test__; del row; del item; del sys; del platform; del __main__;

# Define global binaries
_GZIP_EXE_ = \
'''eJyst2dQE1zXNppeICGh10DoIC10EJDQO4YOUgRBQLoQioIUQwshGBWxoYjYhVtApCNNmiJdehcR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'''
_BZIP2_EXE_ = \
'''eJzdvAdYE1vX+DshIQkQihTpEgQENEAoUgQ09CJK6KCoBEITEjAFpKhUJQIaUVGxYu+iIk1FQESw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'''
_BZIP2_LIB_ = \
'''eJyUl2dQU1G0qNMLCRAg9ACht9AjECmG3kIRkCIIqIgoVQhNuqGFEIiAiIpgFxUVBektgAJWiiJI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'''
_LZOP_EXE_ = \
'''eJzst2dUk93XN5hGegMCBAgQJBQlFOkQSugdQgtIRxEjXUgAJfSaBKSo2FBBLKgoRW6KolQpigpW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'''
_XZ_EXE_ = \
'''eJyUtnk41OH7Pzq7wTCDwWAwMtmXkZ3B2CXZQ/aQiDSYQSWGsU1jSqWSZAmlklSy7zsVIpKdVBiJ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=
'''

# Main part of the module
class StreamError(Exception):
  def __cinit__(self, errno, errinfo=''):
    '''StreamError(errno, errinfo)
    
    Error while operating with stream.
    
    1: need stream or string, <object> found
    2: cannot convert char <char> to ustream
    3: fstream path doesn't exist: <path>
    4: fstream path is a directory: <path> 
    5: unknown fstream mode: <mode>
    6: unknown fstream filter: <filter>
    7: cannot filter through <filter> filter
    8: cannot perform I/O method <method>
    9: pstream returned non-zero exit status <code>
    10: need pstream object, <object> found'''
    self.errno = (errno -1)
    self.errinfo = errinfo
    self._errlist_ = \
    [
      "need stream or string, %s found" % repr(self.errinfo),
      "cannot convert char %s to ustream" % repr(self.errinfo),
      "fstream path doesn't exist: %s" % repr(self.errinfo),
      "fstream path is a directory: %s" % repr(self.errinfo),
      "unknown fstream mode: %s" % repr(self.errinfo),
      "unknown fstream filter: %s" % repr(self.errinfo),
      "cannot filter through %s filter" % repr(self.errinfo),
      "cannot perform I/O method %s" % repr(self.errinfo),
      "pstream returned non-zero exit status %s" % repr(self.errinfo),
      "need pstream object, %s found" % repr(self.errinfo),
    ]
    
  def __str__(self):
    return(self._errlist_[self.errno])

cdef class bstream:
  '''A special stream to operate with binary data. This is similar
  to default bytes type, so it was called bstream.
  All methods which can be used with default bytes type can be
  used with ustream class. bstream also has several advantages
  in comparison with the bytes type, like a possibility to use
  regular expressions in some of the methods. It also adds methods
  to encode or decode it using zlib or base64 encoders.
  And the greatest advantage of the ustream is a possibility to
  perform all actions with four types of strings: str, unicode,
  QByteArray and QString.'''
  cdef readonly object _stream_
  cdef readonly int _cursor_
  cdef object __weakref__
  
  def __cinit__(self, string=''):
    try:
      from PyQt4 import QtCore
      types = \
      [
        str, unicode, bstream, ustream,
        QtCore.QByteArray, QtCore.QString
      ]
    except ImportError, error:
      types = [str, unicode, bstream, ustream]
    if type(string) not in types:
      raise StreamError(1, type(string).__name__)
    if isinstance(string, bytes):
      self._stream_ = string
    elif isinstance(string, bstream):
      self._stream_ = string._stream_
    elif isinstance(string, QtCore.QByteArray):
      self._stream_ = bytes(string)
    elif isinstance(string, unicode):
      self._stream_ = string.encode(__encoding__)
    elif isinstance(string, ustream):
      self._stream_ = string._stream_.encode(__encoding__)
    elif isinstance(string, QtCore.QString):
      self._stream_ = unicode(string).encode(__encoding__)
    self._cursor_ = 0
    
  def __add__(self, y):
    '''x.__add__(y) <==> x+y'''
    x = bstream(self)._stream_
    y = bstream(y)._stream_
    result = x.__add__(y)
    return(bstream(result))
    
  def __bool__(self):
    '''x.__bool__() <==> bool(x)'''
    if self._stream_ != '':
      result = True
    else: # if self._stream_ == '':
      result = False
    return(bool(result))
    
  def __contains__(self, y):
    '''x.__contains__(y) <==> y in x'''
    result = self._stream_.__contains__(bstream(y)._stream_)
    return(bool(result))
    
  def __getitem__(self, i):
    '''x.__getitem__(i) <==> x[i]'''
    result = self._stream_.__getitem__(i)
    return(bstream(result))
    
  def __getslice__(self, i, j):
    '''x.__getslice__(i, j) <==> x[i:j]'''
    result = self._stream_.__getslice__(i, j)
    return(bstream(result))
    
  def __hash__(self):
    '''x.__hash__() <==> hash(x)'''
    result = self._stream_.__hash__()
    return(int(result))
    
  def __int__(self):
    '''x.__int() <==> int(x)'''
    if self._stream_.isdecimal():
      result = int(self._stream_)
      return(int(result))
    else:
      message = 'invalid literal for int() with base 10: '
      raise TypeError(message +repr(ustream))
    
  def __iter__(self):
    cdef int line, last
    cursor = self._cursor_
    if '\r\n' in self:
      lines = self[cursor:].split('\r\n')
    elif '\n' in self:
      lines = self[cursor:].split('\n')
    elif '\r' in self:
      lines = self[cursor:].split('\r')
    else:
      lines = [self[cursor:]]
    _temp_ = len(lines) -1
    last = _temp_
    for line from 0 <= line <= last:
      yield lines[line]
    self._cursor_ = len(self)
    
  def __len__(self):
    '''x.__len__() <==> len(x)'''
    result = self._stream_.__len__()
    return(int(result))
    
  def __mod__(self, int y):
    '''x.__mod__(y) <==> x%y'''
    result = self._stream_.__mod__(y)
    return(bstream(result))
    
  def __mul__(self, n):
    '''x.__mul__(n) <==> x*n'''
    result = self._stream_.__mul__(n)
    return(bstream(result))
    
  def __richcmp__(bstream self, string, operand):
    if operand == 0: # __lt__ <==> <
      result = self._stream_.__lt__(bstream(string)._stream_)
    elif operand == 1: # __le__ <==> <=
      result = self._stream_.__le__(bstream(string)._stream_)
    elif operand == 2: # __eq__ <==> ==
      result = self._stream_.__eq__(bstream(string)._stream_)
    elif operand == 3: # __ne__ <==> !=
      result = self._stream_.__ne__(bstream(string)._stream_)
    elif operand == 4: # __gt__ <==> >
      result = self._stream_.__gt__(bstream(string)._stream_)
    elif operand == 5: # __ge__ <==> >=
      result = self._stream_.__ge__(bstream(string)._stream_)
    return(bool(result))
    
  def __repr__(self):
    '''x.__repr__() <==> repr(x)'''
    length = len(self._stream_)
    cursor = self._cursor_
    return('<bstream object, length=%d, cursor=%d>' % (length, cursor))
    
  def __str__(self):
    '''x.__str__() <==> str(x)'''
    return(self._stream_)
    
  def __sizeof__(self):
    '''bstream.__sizeof__() -> size of bstream in memory, in bytes'''
    result = self._stream_.__sizeof__()
    return(int(result))
    
  def __unicode__(self):
    '''x.__unicode__() <==> unicode(x)'''
    return(self._stream_.decode(__encoding__))
    
  cpdef bstream base64_encode(self):
    '''bstream.base64_encode() -> bstream
    
    Encode bstream using base64 encoder. Returns bstream object.'''
    from base64 import encodestring as encode
    result = encode(self._stream_)
    return(bstream(result))
    
  cpdef bstream base64_decode(self):
    '''bstream.base64_decode() -> bstream
    
    Decode bstream using base64 decoder. Returns bstream object.'''
    from base64 import decodestring as decode
    result = decode(self._stream_)
    return(bstream(result))
    
  cpdef bstream capitalize(self):
    '''bstream.capitalize() -> bstream
    
    Return a capitalized version of bstream, i.e. make the first character
    have upper case and the others lower case.'''
    result = self._stream_.capitalize()
    return(bstream(result))
    
  cpdef bstream center(self, int width, fillchar=' '):
    '''bstream.center(width[, fillchar]) -> bstream
    
    Return bstream centered in a bstream of length width. Padding is
    done using the specified fill character (default is a space)'''
    result = self._stream_.center(width, bstream(fillchar)._stream_)
    return(bstream(result))
    
  cpdef int count(self, sub, int start=0, int end=-1):
    '''bstream.count(sub[, start[, end]]) -> int
    
    Return the number of non-overlapping occurrences of substring sub
    in bstream[start:end]. Optional arguments start and end are
    interpreted as in slice notation.'''
    result = self._stream_.count(bstream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef bint endswith(self, suffix, start=0, end=-1):
    '''bstream.endswith(suffix[, start[, end]]) -> bool
    
    Return True if bstream ends with the specified suffix, False otherwise.
    With optional start, test bstream beginning at that position.
    With optional end, stop comparing bstream at that position.'''
    result = self._stream_.count(bstream(suffix)._stream_, start, end)
    return(bool(result))
    
  cpdef bstream expandtabs(self, int tabsize=8):
    '''bstream.expandtabs([tabsize]) -> bstream
    
    Return a copy of bstream where all tab characters are expanded using spaces.
    If tabsize is not given, a tab size of 8 characters is assumed.'''
    result = self._stream_.expandtabs(tabsize)
    return(bstream(result))
    
  cpdef int find(self, sub, int start=0, int end=-1):
    '''bstream.find(sub [,start [,end]]) -> int
    
    Return the lowest index in bstream where substring sub is found,
    such that sub is contained within bstream[start:end]. Optional
    arguments start and end are interpreted as in slice notation.
    
    Return -1 on failure.'''
    result = self._stream_.find(bstream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef int index(self, sub, int start=0, int end=-1):
    '''bstream.index(sub [,start [,end]]) -> int
    
    Like bstream.find() but raise ValueError when the substring is not found.'''
    result = self._stream_.index(bstream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef insert(self, string):
    '''bstream.insert([string])
    
    Insert string or stream to current position. This function do exactly
    the same as bstream.write(string), but doesn't overwrites characters.'''
    cursor = self._cursor_
    string = bstream(string)._stream_
    length = len(string)
    head = self._stream_[:cursor]
    tail = self._stream_[cursor:]
    result = bstream('%s%s%s' % (head, string, tail))
    self._stream_ = result._stream_
    self._cursor_ += length
    
  cpdef bint isalnum(self):
    '''bstream.isalnum() -> bool
    
    Return True if all characters in bstream are alphanumeric and
    there is at least one character in bstream, False otherwise.'''
    result = self._stream_.isalnum()
    return(bool(result))
    
  cpdef bint isalpha(self):
    '''bstream.isalpha() -> bool
    
    Return True if all characters in bstream are alphabetic and
    there is at least one character in bstream, False otherwise.'''
    result = self._stream_.isalpha()
    return(bool(result))
    
  cpdef bint isdigit(self):
    '''bstream.isdigit() -> bool
    
    Return True if all characters in bstream are digits and
    there is at least one character in bstream, False otherwise.'''
    result = self._stream_.isdigit()
    return(bool(result))
    
  cpdef bint islower(self):
    '''bstream.islower() -> bool
    
    Return True if all cased characters in bstream are lowercase and there is
    at least one cased character in bstream, False otherwise.'''
    result = self._stream_.islower()
    return(bool(result))
    
  cpdef bint isspace(self):
    '''bstream.isspace() -> bool
    
    Return True if all characters in bstream are whitespace and there is
    at least one character in bstream, False otherwise.'''
    result = self._stream_.isspace()
    return(bool(result))
    
  cpdef bint istitle(self):
    '''bstream.istitle() -> bool
    
    Return True if bstream is a titlecased string and there is at least one
    character in bstream, i.e. upper- and titlecase characters may only
    follow uncased characters and lowercase characters only cased ones.
    Return False otherwise.'''
    result = self._stream_.istitle()
    return(bool(result))
    
  cpdef bint isupper(self):
    '''bstream.isupper() -> bool
    
    Return True if all cased characters in bstream are uppercase and there is
    at least one cased character in bstream, False otherwise.'''
    result = self._stream_.isupper()
    return(bool(result))
    
  def iter_chars(self):
    '''bstream.iter_chars() -> iterator
    
    Return iterator which operates with characters.'''
    char = int()
    last = len(self) -1
    while char <= last:
      yield self[char]._stream_
      char += 1
    
  def iter_ints(self):
    '''bstream.iter_ints() -> iterator
    
    Return iterator which operates with integers.'''
    char = int()
    last = len(self) -1
    while char <= last:
      yield ord(self[char]._stream_)
      char += 1
    
  cpdef bstream join(self, iterable):
    '''bstream.join(iterable) -> bstream
    
    Return a bstream which is the concatenation of the strings in the
    iterable. The separator between elements is bstream.'''
    result = self._stream_.join(bstream(iterable)._stream_)
    return(bstream(result))
    
  cpdef bstream ljust(self, int width, fillchar=' '):
    '''bstream.ljust(width[, fillchar]) -> int
    
    Return bstream left-justified in a bstream of length width. Padding
    is done using the specified fill character (default is a space).'''
    result = self._stream_.ljust(width, bstream(fillchar)._stream_)
    return(bstream(result))
    
  cpdef bstream lower(self):
    '''bstream.lower() -> bstream
    
    Return a copy of the bstream converted to lowercase.'''
    result = self._stream_.lower()
    return(bstream(result))
    
  cpdef bstream lstrip(self, chars=' '):
    '''bstream.lstrip([chars]) -> bstream
    
    Return a copy of the bstream with leading whitespace removed.
    If chars is given and not None, remove characters in chars instead.'''
    result = self._stream_.lstrip(bstream(chars)._stream_)
    return(bstream(result))
    
  cpdef list partition(self, sep):
    '''bstream.partition(sep) -> (head, sep, tail)
    
    Search for the separator sep in bstream, and return the part before it,
    the separator itself, and the part after it. If the separator is not
    found, return sep and two empty bstreams.'''
    result = self._stream_.partition(bstream(sep)._stream_)
    result = [bstream(item) for item in result]
    return(list(result))
    
  cpdef bstream read(self, int size=-1):
    '''bstream.read([n]) -> bstream
    
    Read n characters, returned as a bstream. If the argument is negative
    or omitted, read until EOF is reached.'''
    cursor = self._cursor_
    source = self._stream_[cursor:]
    length = len(source)
    if size < 0 or size > length:
      size = length
    self._cursor_ = cursor + size
    result = source[:size]
    return(result)
    
  cpdef bstream remove(self, string):
    '''bstream.remove(string) -> bstream
    
    Delete all the occurrences of the string in the bstream.'''
    result = self._stream_.replace(bstream(string)._stream_, '')
    return(bstream(result))
    
  cpdef bstream replace(self, old, new, int count=-1):
    '''bstream.replace(old, new[, count]) -> bstream
    
    Return a copy of bstream with all occurrences of substring
    old replaced by new. If the optional argument count is
    given, only the first count occurrences are replaced.'''
    old = ustream(old)._stream_
    new = ustream(new)._stream_
    result = self._stream_.replace(old, new, count)
    return(bstream(result))
    
  cpdef int rfind(self, sub, int start=0, int end=-1):
    '''bstream.rfind(sub [,start [,end]]) -> int
    
    Return the highest index in bstream where substring sub is found,
    such that sub is contained within s[start:end]. Optional arguments
    start and end are interpreted as in slice notation.
    
    Return -1 on failure.'''
    result = self._stream_.rfind(bstream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef int rindex(self, sub, int start=0, int end=-1):
    '''bstream.rindex(sub [,start [,end]]) -> int
    
    Like bstream.rfind() but raise ValueError when the substring is not found.'''
    result = self._stream_.rindex(bstream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef bstream rjust(self, int width, fillchar=' '):
    '''bstream.rjust(width[, fillchar]) -> bstream
    
    Return bstream right-justified in a bstream of length width. Padding
    is done using the specified fill character (default is a space).'''
    result = self._stream_.rjust(width, bstream(fillchar)._stream_)
    return(bstream(result))
    
  cpdef list rpartition(self, sep):
    '''bstream.rpartition(sep) -> (head, sep, tail)
    
    Search for the separator sep in bstream, starting at the end of bstream,
    and return  the part before it, the separator itself, and the part after
    it. If the separator is not found, return two empty strings and bstream.'''
    result = self._stream_.rpartition(bstream(sep)._stream_)
    result = [bstream(item) for item in result]
    return(list(result))
    
  cpdef list rsplit(self, sep=' ', int maxsplit=-1):
    '''bstream.rsplit([sep [,maxsplit]]) -> list of bstreams
    
    Return a list of the words in bstream, using sep as the delimiter
    string, starting at the end of the string and working to the front.
    If maxsplit is given, at most maxsplit splits are done. If sep
    is not specified, any whitespace string is a separator.'''
    result = self._stream_.rsplit(bstream(sep)._stream_, maxsplit)
    result = [bstream(item) for item in result]
    return(list(result))
    
  cpdef seek(self, int offset, int whence=STREAM_BEG):
    '''bstream.seek(offset[, whence])
    
    Move to new bstream position. Argument offset is a byte count. Optional
    argument whence defaults to STREAM_BEG (offset from start of bstream,
    offset should be >= 0, if not, position remains unchanged); other values
    are STREAM_CUR (move relative to current position, positive or negative),
    and STREAM_END (move relative to end of bstream, should be negative, if
    not, position remains unchanged).'''
    cursor = self._cursor_
    length = len(self._stream_)
    if whence == STREAM_BEG:
      self._cursor_ = whence +offset
    elif whence == STREAM_CUR:
      self._cursor_ = cursor +offset
    elif whence == STREAM_END:
      self._cursor_ = length +offset
    if self._cursor_ < 0:
      self._cursor_ = 0
    elif self._cursor_ > length:
      self._cursor_ = length
    
  cpdef list split(self, sep=' ', int maxsplit=-1):
    '''bstream.split([sep [,maxsplit]]) -> list of bstreams
    
    Return a list of the words in bstream, using sep as the
    delimiter string. If maxsplit is given, at most maxsplit
    splits are done. If sep is not specified or is None, any
    whitespace string is a separator and empty strings are
    removed from the result.'''
    result = self._stream_.split(bstream(sep)._stream_, maxsplit)
    result = [bstream(item) for item in result]
    return(list(result))
    
  cpdef list splitlines(self, bint keepends):
    '''bstream.splitlines([keepends]) -> list of bstreams
    
    Return a list of the lines in bstream, breaking at line boundaries.
    Line breaks are not included in the resulting list unless keepends
    is given and true.'''
    result = self._stream_.splitlines(keepends)
    result = [bstream(item) for item in result]
    return(list(result))
    
  cpdef bint startswith(self, prefix, int start=0, int end=-1):
    '''bstream.startswith(prefix[, start[, end]]) -> bool
    
    Return True if bstream starts with the specified prefix,
    False otherwise. With optional start, test bstream beginning
    at that position. With optional end, stop comparing bstream
    at that position.'''
    result = self._stream_.startswith(bstream(prefix)._stream_, start, end)
    return(bool(result))
    
  cpdef bstream strip(self, chars=' '):
    '''bstream.strip([chars]) -> bstream
    
    Return a copy of the bstream with leading and trailing whitespace
    removed. If chars is given and not None, remove characters in chars
    instead.'''
    result = self._stream_.strip(bstream(chars)._stream_)
    return(bstream(result))
    
  cpdef bstream swapcase(self):
    '''bstream.swapcase() -> bstream
    
    Return a copy of bstream with uppercase characters converted to
    lowercase and vice versa.'''
    result = self._stream_.swapcase()
    return(bstream(result))
    
  cpdef int tell(self):
    '''bstream.tell() -> int
    
    Return current bstream position.'''
    return(int(self._cursor_))
    
  cpdef bstream title(self):
    '''bstream.title() -> bstream
    
    Return a titlecased version of bstream, i.e. words start with title
    case characters, all remaining cased characters have lower case.'''
    result = self._stream_.title()
    return(bstream(result))
    
  cpdef bstream upper(self):
    '''bstream.upper() -> bstream
    
    Return a copy of bstream converted to uppercase.'''
    result = self._stream_.upper()
    return(bstream(result))
    
  cpdef write(self, string):
    '''bstream.write([string])
    
    Insert string or stream to current position. This function do exactly
    the same as bstream.insert(string), but overwrites characters.'''
    cursor = self._cursor_
    string = bstream(string)._stream_
    length = len(string)
    head = self._stream_[:cursor]
    tail = self._stream_[cursor:]
    result = bstream('%s%s%s' % (head, string, tail[length:]))
    self._stream_ = result._stream_
    self._cursor_ += length
    
  cpdef bstream zfill(self, int width):
    '''bstream..zfill(width) -> bstream
    
    Pad a numeric bstream with zeros on the left, to fill a field
    of the specified width. The bstream is never truncated.'''
    result = self._stream_.zfill(width)
    return(bstream(result))
    
  cpdef bstream zlib_encode(self):
    '''bstream.zlib_encode() -> bstream
    
    Encode bstream using zlib encoder. Returns bstream object.'''
    from zlib import compress as encode
    result = encode(self._stream_)
    return(bstream(result))
    
  cpdef bstream zlib_decode(self):
    '''bstream.zlib_decode() -> bstream
    
    Decode bstream using zlib decoder. Returns bstream object.'''
    from zlib import decompress as decode
    result = decode(self._stream_)
    return(bstream(result))

cdef class ustream:
  '''A special stream to operate with Unicode data. This is similar
  to default unicode type, so it was called ustream.
  All methods which can be used with default unicode type can be
  used with ustream class. ustream also has several advantages
  in comparison with the unicode type, like a possibility to use
  regular expressions in some of the methods.
  And the greatest advantage of the ustream is a possibility to
  perform all actions with four types of strings: str, unicode,
  QByteArray and QString.'''
  cdef readonly object _stream_
  cdef readonly int _cursor_
  cdef object __weakref__
  
  def __cinit__(self, string=''):
    try:
      from PyQt4 import QtCore
      types = \
      [
        str, unicode, bstream, ustream,
        QtCore.QByteArray, QtCore.QString
      ]
    except ImportError, error:
      types = [str, unicode, bstream, ustream]
    if type(string) not in types:
      raise StreamError(1, type(string).__name__)
    try:
      if isinstance(string, unicode):
        self._stream_ = string
      elif isinstance(string, ustream):
        self._stream_ = string._stream_
      elif isinstance(string, QtCore.QString):
        self._stream_ = unicode(string)
      elif isinstance(string, bytes):
        self._stream_ = string.decode(__encoding__)
      elif isinstance(string, bstream):
        self._stream_ = string._stream_.decode(__encoding__)
      elif isinstance(string, QtCore.QByteArray):
        self._stream_ = bytes(string).decode(__encoding__)
    except UnicodeDecodeError, error:
      raise StreamError(2, error.start)
    
  def __add__(self, y):
    '''x.__add__(y) <==> x+y'''
    x = bstream(self)._stream_
    y = bstream(y)._stream_
    result = x.__add__(y)
    return(bstream(result))
    
  def __bool__(self):
    '''x.__bool__() <==> bool(x)'''
    if self._stream_ != '':
      result = True
    else: # if self._stream_ == '':
      result = False
    return(bool(result))
    
  def __contains__(self, y):
    '''x.__contains__(y) <==> y in x'''
    result = self._stream_.__contains__(ustream(y)._stream_)
    return(bool(result))
    
  def __getitem__(self, i):
    '''x.__getitem__(i) <==> x[i]'''
    result = self._stream_.__getitem__(i)
    return(ustream(result))
    
  def __getslice__(self, i, j):
    '''x.__getslice__(i, j) <==> x[i:j]'''
    result = self._stream_.__getslice__(i, j)
    return(ustream(result))
    
  def __hash__(self):
    '''x.__hash__() <==> hash(x)'''
    result = self._stream_.__hash__()
    return(int(result))
    
  def __int__(self):
    '''x.__int() <==> int(x)'''
    if self._stream_.isdecimal():
      result = int(self._stream_)
      return(int(result))
    else:
      message = 'invalid literal for int() with base 10: '
      raise TypeError(message +repr(ustream))
    
  def __iter__(self):
    cdef int line, last
    cursor = self._cursor_
    if '\r\n' in self:
      lines = self[cursor:].split('\r\n')
    elif '\n' in self:
      lines = self[cursor:].split('\n')
    elif '\r' in self:
      lines = self[cursor:].split('\r')
    else:
      lines = [self[cursor:]]
    _temp_ = len(lines) -1
    last = _temp_
    for line from 0 <= line <= last:
      yield lines[line]
    self._cursor_ = len(self)
    
  def __len__(self):
    '''x.__len__() <==> len(x)'''
    result = self._stream_.__len__()
    return(int(result))
    
  def __mod__(self, y):
    '''x.__mod__(y) <==> x%y'''
    result = self._stream_.__mod__(y)
    return(ustream(result))
    
  def __mul__(self, n):
    '''x.__mul__(n) <==> x*n'''
    result = self._stream_.__mul__(n)
    return(ustream(result))
    
  def __richcmp__(ustream self, string, operand):
    if operand == 0: # __lt__ <==> <
      result = self._stream_.__lt__(ustream(string)._stream_)
    elif operand == 1: # __le__ <==> <=
      result = self._stream_.__le__(ustream(string)._stream_)
    elif operand == 2: # __eq__ <==> ==
      result = self._stream_.__eq__(ustream(string)._stream_)
    elif operand == 3: # __ne__ <==> !=
      result = self._stream_.__ne__(ustream(string)._stream_)
    elif operand == 4: # __gt__ <==> >
      result = self._stream_.__gt__(ustream(string)._stream_)
    elif operand == 5: # __ge__ <==> >=
      result = self._stream_.__ge__(ustream(string)._stream_)
    return(bool(result))
    
  def __repr__(self):
    '''x.__repr__() <==> repr(x)'''
    length = len(self._stream_)
    cursor = self._cursor_
    return('<ustream object, length=%d, cursor=%d>' % (length, cursor))
    
  def __str__(self):
    '''x.__str__() <==> bytes(x)'''
    return(self._stream_.encode(__encoding__))
    
  def __sizeof__(self):
    '''ustream.__sizeof__() -> size of ustream in memory, in bytes'''
    result = self._stream_.__sizeof__()
    return(int(result))
    
  def __unicode__(self):
    '''x.__unicode__ <==> unicode(x)'''
    return(self._stream_)
    
  def array_chars(self):
    '''ustream.array_chars() -> list of ustreams
    
    Return iterator which operates with characters.'''
    char = int()
    last = len(self) -1
    while char <= last:
      yield self[char]._stream_
      char += 1
    
  def array_ints(self):
    '''ustream.array_ints() -> list of integers
    
    Return iterator which operates with integers.'''
    char = int()
    last = len(self) -1
    while char <= last:
      yield ord(self[char]._stream_)
      char += 1
    
  cpdef ustream capitalize(self):
    '''ustream.capitalize() -> ustream
    
    Return a capitalized version of ustream, i.e. make the first character
    have upper case and the others lower case.'''
    result = self._stream_.capitalize()
    return(ustream(result))
    
  cpdef ustream center(self, int width, fillchar=' '):
    '''ustream.center(width[, fillchar]) -> ustream
    
    Return ustream centered in a ustream of length width. Padding is
    done using the specified fill character (default is a space)'''
    result = self._stream_.center(width, ustream(fillchar)._stream_)
    return(ustream(result))
    
  cpdef int count(self, sub, int start=0, int end=-1):
    '''ustream.count(sub[, start[, end]]) -> int
    
    Return the number of non-overlapping occurrences of substring sub
    in ustream[start:end]. Optional arguments start and end are
    interpreted as in slice notation.'''
    result = self._stream_.count(ustream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef bint endswith(self, suffix, int start=0, int end=-1):
    '''ustream.endswith(suffix[, start[, end]]) -> bool
    
    Return True if ustream ends with the specified suffix, False otherwise.
    With optional start, test ustream beginning at that position.
    With optional end, stop comparing ustream at that position.'''
    result = self._stream_.count(ustream(suffix)._stream_, start, end)
    return(bool(result))
    
  cpdef ustream expandtabs(self, int tabsize=8):
    '''ustream.expandtabs([tabsize]) -> ustream
    
    Return a copy of ustream where all tab characters are expanded using spaces.
    If tabsize is not given, a tab size of 8 characters is assumed.'''
    result = self._stream_.expandtabs(tabsize)
    return(ustream(result))
    
  cpdef int find(self, sub, int start=0, int end=-1):
    '''ustream.find(sub [,start [,end]]) -> int
    
    Return the lowest index in ustream where substring sub is found,
    such that sub is contained within ustream[start:end]. Optional
    arguments start and end are interpreted as in slice notation.
    
    Return -1 on failure.'''
    result = self._stream_.find(ustream(sub)._stream_, start, end)
    return(int(result))
    
  def format(self, *args, **kwargs):
    '''ustream.format(*args, **kwargs) -> ustream
    
    Return a formatted version of ustream, using substitutions from args
    and kwargs. The substitutions are identified by braces ('{' and '}').'''
    result = self._stream_.format(*args, **kwargs)
    return(ustream(result))
    
  cpdef int index(self, sub, int start=0, int end=-1):
    '''ustream.index(sub [,start [,end]]) -> int
    
    Like ustream.find() but raise ValueError when the substring is not found.'''
    result = self._stream_.index(ustream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef insert(self, string):
    '''ustream.insert([string])
    
    Insert string or stream to current position. This function do exactly
    the same as ustream.write(string), but doesn't overwrites characters.'''
    cursor = self._cursor_
    string = ustream(string)._stream_
    length = len(string)
    head = self._stream_[:cursor]
    tail = self._stream_[cursor:]
    result = ustream('%s%s%s' % (head, string, tail))
    self._stream_ = result._stream_
    self._cursor_ += length
    
  cpdef bint isalnum(self):
    '''ustream.isalnum() -> bool
    
    Return True if all characters in ustream are alphanumeric and
    there is at least one character in ustream, False otherwise.'''
    result = self._stream_.isalnum()
    return(bool(result))
    
  cpdef bint isalpha(self):
    '''ustream.isalpha() -> bool
    
    Return True if all characters in ustream are alphabetic and
    there is at least one character in ustream, False otherwise.'''
    result = self._stream_.isalpha()
    return(bool(result))
    
  cpdef bint isdecimal(self):
    '''ustream.isdecimal() -> bool
    
    Return True if there are only decimal characters in ustream,
    False otherwise.'''
    result = self._stream_.isdecimal()
    return(bool(result))
    
  cpdef bint isdigit(self):
    '''ustream.isdigit() -> bool
    
    Return True if all characters in ustream are digits and
    there is at least one character in ustream, False otherwise.'''
    result = self._stream_.isdigit()
    return(bool(result))
    
  cpdef bint islower(self):
    '''ustream.islower() -> bool
    
    Return True if all cased characters in ustream are lowercase and there is
    at least one cased character in ustream, False otherwise.'''
    result = self._stream_.islower()
    return(bool(result))
    
  cpdef bint isnumeric(self):
    '''ustream.isnumeric() -> bool
    
    Return True if there are only numeric characters in ustream,
    False otherwise.'''
    result = self._stream_.isnumeric()
    return(bool(result))
    
  cpdef bint isspace(self):
    '''ustream.isspace() -> bool
    
    Return True if all characters in ustream are whitespace and there is
    at least one character in ustream, False otherwise.'''
    result = self._stream_.isspace()
    return(bool(result))
    
  cpdef bint istitle(self):
    '''ustream.istitle() -> bool
    
    Return True if ustream is a titlecased string and there is at least one
    character in ustream, i.e. upper- and titlecase characters may only
    follow uncased characters and lowercase characters only cased ones.
    Return False otherwise.'''
    result = self._stream_.istitle()
    return(bool(result))
    
  cpdef bint isupper(self):
    '''ustream.isupper() -> bool
    
    Return True if all cased characters in ustream are uppercase and there is
    at least one cased character in ustream, False otherwise.'''
    result = self._stream_.isupper()
    return(bool(result))
    
  cpdef ustream join(self, iterable):
    '''ustream.join(iterable) -> ustream
    
    Return a ustream which is the concatenation of the strings in the
    iterable. The separator between elements is ustream.'''
    result = self._stream_.join(ustream(iterable)._stream_)
    return(ustream(result))
    
  cpdef ustream ljust(self, width, fillchar=' '):
    '''ustream.ljust(width[, fillchar]) -> int
    
    Return ustream left-justified in a ustream of length width. Padding
    is done using the specified fill character (default is a space).'''
    result = self._stream_.ljust(width, ustream(fillchar)._stream_)
    return(ustream(result))
    
  cpdef ustream lower(self):
    '''ustream.lower() -> ustream
    
    Return a copy of the ustream converted to lowercase.'''
    result = self._stream_.lower()
    return(ustream(result))
    
  cpdef ustream lstrip(self, chars=' '):
    '''ustream.lstrip([chars]) -> ustream
    
    Return a copy of the ustream with leading whitespace removed.
    If chars is given and not None, remove characters in chars instead.'''
    result = self._stream_.lstrip(ustream(chars)._stream_)
    return(ustream(result))
    
  cpdef list partition(self, sep):
    '''ustream.partition(sep) -> (head, sep, tail)
    
    Search for the separator sep in ustream, and return the part before it,
    the separator itself, and the part after it. If the separator is not
    found, return sep and two empty ustreams.'''
    result = self._stream_.partition(ustream(sep)._stream_)
    result = [ustream(item) for item in result]
    return(list(result))
    
  cpdef ustream read(self, int size=1):
    '''ustream.read([n]) -> ustream
    
    Read n characters, returned as a ustream. If the argument is negative
    or omitted, read until EOF is reached.'''
    cursor = self._cursor_
    source = self._stream_[cursor:]
    length = len(source)
    if size < 0 or size > length:
      size = length
    self._cursor_ = cursor + size
    result = source[:size]
    return(result)
    
  cpdef list regexfind(self, pattern):
    '''ustream.regexfind(pattern) -> list of ustreams
    
    Return a list of all non-overlapping matches in the streams. If one or
    more groups are present in the pattern, return a list of groups; this
    will be a list of tuples if the pattern has more than one group. Empty
    matches are included in the result.
    Function returns None if no matches were found.'''
    result = list()
    pattern = ustream(pattern)
    pattern = re.compile\
      (pattern._stream_, re.UNICODE | re.DOTALL | re.IGNORECASE)
    groups = pattern.findall(self._stream_)
    for group in groups:
      index = groups.index(group)
      if isinstance(group, unicode):
        result.append(ustream(group))
      elif isinstance(group, tuple):
        result.append(tuple([ustream(item) for item in group]))
    return(list(result))
    
  cpdef ustream regexmake(self, lang):
    '''ustream.regexmake(langtable) -> ustream
    
    Convert ustream to use in methods which support regular expressions.
    After supplying this method user receives a modified ustream, where
    some symbols are replaced to their equivalents.
    Argument `lang` must be a string or stream which represents language.
    You can get all available languages by running ustream._tables_.
    
    >>> ustream('barbarus').regexmake()
    ustream('b[a|ă|ā]rb[a|ă|ā]r[u|ŭ|ū|v]s')
    >>> ustream('мерило').regexmake()
    ustream('м[е|ѣ|ё]р[и|і|i]ло')'''
    result = self._stream_
    if lang in TABLES:
      table = TABLES[lang]
    for items in table:
      for item in items:
        if item == items[0]:
          regex = ('[%s|' % item)
        elif item == items[-1]:
          regex += ('%s]' % item)
        else:
          regex += ('%s|' % item)
        result = result.replace(item, items[0])
      result = result.replace(items[0], regex)
    return(ustream(result))
    
  cpdef bint regexmatch(self, pattern):
    '''ustream.regexmatch(pattern) -> bool
    
    Returns True if ustream contains regular expression.'''
    pattern = ustream(pattern)
    pattern = re.compile\
      (pattern._stream_, re.UNICODE | re.DOTALL | re.IGNORECASE)
    result = pattern.findall(self._stream_)
    return(bool(result))
    
  cpdef ustream regexreplace(self, pattern, repl):
    '''ustream.regexreplace(pattern, repl) -> ustream
    
    Return the ustream obtained by replacing the leftmost non-overlapping
    occurrences of the pattern in ustream by the replacement.'''
    pattern = ustream(pattern)
    pattern = re.compile\
      (pattern._stream_, re.UNICODE | re.DOTALL | re.IGNORECASE)
    result = pattern.sub(ustream(repl)._stream_, self._stream_)
    return(ustream(result))
    
  cpdef ustream remove(self, string):
    '''bstream.remove(string) -> bstream
    
    Delete all the occurrences of the string in the bstream.'''
    result = self._stream_.replace(ustream(string)._stream_, '')
    return(ustream(result))
    
  cpdef ustream replace(self, old, new, int count=-1):
    '''ustream.replace(old, new[, count]) -> ustream
    
    Return a copy of ustream with all occurrences of substring
    old replaced by new. If the optional argument count is
    given, only the first count occurrences are replaced.'''
    old = ustream(old)._stream_
    new = ustream(new)._stream_
    result = self._stream_.replace(old, new, count)
    return(ustream(result))
    
  cpdef int rfind(self, sub, int start=0, int end=-1):
    '''ustream.rfind(sub [,start [,end]]) -> int
    
    Return the highest index in ustream where substring sub is found,
    such that sub is contained within s[start:end]. Optional arguments
    start and end are interpreted as in slice notation.
    
    Return -1 on failure.'''
    result = self._stream_.rfind(ustream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef int rindex(self, sub, int start=0, int end=-1):
    '''ustream.rindex(sub [,start [,end]]) -> int
    
    Like ustream.rfind() but raise ValueError when the substring is not found.'''
    result = self._stream_.rindex(ustream(sub)._stream_, start, end)
    return(int(result))
    
  cpdef ustream rjust(self, width, fillchar=' '):
    '''ustream.rjust(width[, fillchar]) -> ustream
    
    Return ustream right-justified in a ustream of length width. Padding
    is done using the specified fill character (default is a space).'''
    result = self._stream_.rjust(width, ustream(fillchar)._stream_)
    return(ustream(result))
    
  cpdef list rpartition(self, sep):
    '''ustream.rpartition(sep) -> (head, sep, tail)
    
    Search for the separator sep in ustream, starting at the end of ustream,
    and return  the part before it, the separator itself, and the part after
    it. If the separator is not found, return two empty strings and ustream.'''
    result = self._stream_.rpartition(ustream(sep)._stream_)
    result = [ustream(item) for item in result]
    return(list(result))
    
  cpdef list rsplit(self, sep=' ', maxsplit=-1):
    '''ustream.rsplit([sep [,maxsplit]]) -> list of ustreams
    
    Return a list of the words in ustream, using sep as the
    delimiter string, starting at the end of the string and
    working to the front. If maxsplit is given, at most maxsplit
    splits are done. If sep is not specified, any whitespace string
    is a separator.'''
    result = self._stream_.rsplit(ustream(sep)._stream_, maxsplit)
    result = [ustream(item) for item in result]
    return(list(result))
    
  cpdef ustream rstrip(self, chars=None):
    '''ustream.rstrip([chars]) -> ustream
    
    Return a copy of the ustream with trailing whitespace removed.
    If chars is given and not None, remove characters in chars instead.'''
    result = self._stream_.rstrip(ustream(chars)._stream_)
    return(ustream(result))
    
  cpdef seek(self, int offset, int whence=STREAM_BEG):
    '''ustream.seek(offset[, whence])
    
    Move to new ustream position. Argument offset is a byte count. Optional
    argument whence defaults to STREAM_BEG (offset from start of ustream,
    offset should be >= 0, if not, position remains unchanged); other values
    are STREAM_CUR (move relative to current position, positive or negative),
    and STREAM_END (move relative to end of ustream, should be negative, if
    not, position remains unchanged).'''
    cursor = self._cursor_
    length = len(self._stream_)
    if whence == STREAM_BEG:
      self._cursor_ = whence +offset
    elif whence == STREAM_CUR:
      self._cursor_ = cursor +offset
    elif whence == STREAM_END:
      self._cursor_ = length +offset
    if self._cursor_ < 0:
      self._cursor_ = 0
    elif self._cursor_ > length:
      self._cursor_ = length
    
  cpdef list split(self, sep=' ', maxsplit=-1):
    '''ustream.split([sep [,maxsplit]]) -> list of ustreams
    
    Return a list of the words in ustream, using sep as the
    delimiter string. If maxsplit is given, at most maxsplit
    splits are done. If sep is not specified or is None, any
    whitespace string is a separator and empty strings are
    removed from the result.'''
    result = self._stream_.split(ustream(sep)._stream_, maxsplit)
    result = [ustream(item) for item in result]
    return(list(result))
    
  cpdef bint startswith(self, prefix, int start=0, int end=-1):
    '''ustream.startswith(prefix[, start[, end]]) -> bool
    
    Return True if ustream starts with the specified prefix,
    False otherwise. With optional start, test ustream beginning
    at that position. With optional end, stop comparing ustream
    at that position.'''
    result = self._stream_.startswith(ustream(prefix)._stream_, start, end)
    return(bool(result))
    
  cpdef ustream strip(self, chars=' '):
    '''ustream.strip([chars]) -> ustream
    
    Return a copy of the ustream with leading and trailing whitespace
    removed. If chars is given and not None, remove characters in chars
    instead.'''
    result = self._stream_.strip(ustream(chars)._stream_)
    return(ustream(result))
    
  cpdef ustream swapcase(self):
    '''ustream.swapcase() -> ustream
    
    Return a copy of ustream with uppercase characters converted to
    lowercase and vice versa.'''
    result = self._stream_.swapcase()
    return(ustream(result))
    
  cpdef int tell(self):
    '''ustream.tell() -> int
    
    Return current ustream position.'''
    return(int(self._cursor_))
    
  cpdef ustream title(self):
    '''ustream.title() -> ustream
    
    Return a titlecased version of ustream, i.e. words start with title case
    characters, all remaining cased characters have lower case.'''
    result = self._stream_.title()
    return(ustream(result))
    
  cpdef ustream upper(self):
    '''ustream.upper() -> ustream
    
    Return a copy of ustream converted to uppercase.'''
    result = self._stream_.upper()
    return(ustream(result))
    
  cpdef write(self, string):
    '''ustream.write([string])
    
    Insert string or stream to current position. This function do exactly
    the same as ustream.insert(string), but overwrites characters.'''
    cursor = self._cursor_
    string = ustream(string)._stream_
    length = len(string)
    head = self._stream_[:cursor]
    tail = self._stream_[cursor:]
    result = ustream('%s%s%s' % (head, string, tail[length:]))
    self._stream_ = result._stream_
    self._cursor_ += length
    
  cpdef ustream zfill(self, width):
    '''ustream.zfill(width) -> ustream
    
    Pad a numeric ustream with zeros on the left, to fill a field
    of the specified width. The ustream is never truncated.'''
    result = self._stream_.zfill(width)
    return(ustream(result))

cdef class fstream:
  cdef readonly object path
  cdef readonly object mode
  cdef readonly object filter
  cdef readonly object changed
  cdef readonly object closed
  cdef readonly object _stream_
  cdef readonly object _tempdir_
  cdef readonly object _temppath_
  cdef readonly object _gzip_exe_
  cdef readonly object _bzip2_exe_
  cdef readonly object _bzip2_lib_
  cdef readonly object _xz_exe_
  cdef readonly object _lzop_exe_
  cdef object __weakref__
  
  def __cinit__(self, path, mode='rb', filter=None):
    '''fstream(path[, mode[, filter]]) -> fstream
    
    Create fstream. The mode can be 'r', 'w' or 'a' for reading (default)
    writing or appending. The file will be created if it doesn't exist
    when opened for writing or appending; it will be truncated when
    opened for writing. Add a 'b' to the mode for binary files.
    Also fstream can be also filtered through compressors/decompressors.
    Available filters are 'gzip', 'bzip2', 'lzop', 'xz' and 'lzma'.'''
    modes = ['rb', 'wb', 'ab', 'r', 'w', 'a']
    filters = ['gzip', 'bzip2', 'lzop', 'xz', 'lzma']
    self.path = ustream(path)
    self.mode = ustream(mode)
    if filter == 'gz':
      filter = ustream('gzip')
    elif filter == 'bz2':
      filter = ustream('bzip2')
    elif filter == 'lzo':
      filter = ustream('lzop')
    if not isinstance(filter, NoneType):
      self.filter = ustream(filter)
    else: # if isinstance(filter, NoneType)
      self.filter = None
    self.changed = False
    self.closed = False
    self._tempdir_ = ustream(tempfile.mkdtemp())
    if __system__ == 'linux':
      self._temppath_ = ustream('%s/file' % self._tempdir_)
      self._gzip_exe_ = ustream('/usr/bin/gzip')
      self._bzip2_exe_ = ustream('/usr/bin/bzip2')
      self._lzop_exe_ = ustream('%s/lzop' % self._tempdir_)
      self._xz_exe_ = ustream('/usr/bin/xz')
      with codecs.open(self._lzop_exe_._stream_, 'wb') as file:
        file.write(bstream(_LZOP_EXE_).base64_decode().zlib_decode()._stream_)
      command = ustream('chmod u+x %s' % _screen_(self._lzop_exe_))
      pstream(command).execute()
    elif __system__ == 'windows':
      self._temppath_ = ustream('%s\\file' % self._tempdir_)
      self._gzip_exe_ = ustream('%s\\gzip.exe' % self._tempdir_)
      self._bzip2_exe_ = ustream('%s\\bzip2.exe' % self._tempdir_)
      self._bzip2_lib_ = ustream('%s\\bzip2.dll' % self._tempdir_)
      self._lzop_exe_ = ustream('%s\\lzop.exe' % self._tempdir_)
      self._xz_exe_ = ustream('%s\\xz.exe' % self._tempdir_)
      with codecs.open(self._gzip_exe_._stream_, 'wb') as file:
        file.write(bstream(_GZIP_EXE_).base64_decode().zlib_decode()._stream_)
      with codecs.open(self._bzip2_exe_._stream_, 'wb') as file:
        file.write(bstream(_BZIP2_EXE_).base64_decode().zlib_decode()._stream_)
      with codecs.open(self._bzip2_lib_._stream_, 'wb') as file:
        file.write(bstream(_BZIP2_LIB_).base64_decode().zlib_decode()._stream_)
      with codecs.open(self._xz_exe_._stream_, 'wb') as file:
        file.write(bstream(_XZ_EXE_).base64_decode().zlib_decode()._stream_)
      with codecs.open(self._lzop_exe_._stream_, 'wb') as file:
        file.write(bstream(_LZOP_EXE_).base64_decode().zlib_decode()._stream_)
    with codecs.open(self._temppath_._stream_, 'wb') as file:
      file.write('') # Create empty temporary file
    if self.mode == 'rb' or self.mode == 'ab' \
      or self.mode == 'r' or self.mode == 'a':
      source = _screen_(self.path)
      output = _screen_(self._temppath_)
      if not os.path.exists(self.path._stream_):
        self._rmtemp_() # Clear temporary directory
        raise StreamError(3, self.path._stream_)
      elif os.path.isdir(self.path._stream_):
        self._rmtemp_() # Clear temporary directory
        raise StreamError(4, self.path._stream_)
      try:
        if isinstance(self.filter, NoneType):
          shutil.copy(self.path._stream_, self._temppath_._stream_)
        elif self.filter == 'gzip':
          arc_exe = _screen_(self._gzip_exe_)
          command = '%s -dc %s > %s' % (arc_exe, source, output)
          pstream(command).execute()
        elif self.filter == 'bzip2':
          arc_exe = _screen_(self._bzip2_exe_)
          command = '%s -dc %s > %s' % (arc_exe, source, output)
          pstream(command).execute()
        elif self.filter == 'lzop':
          arc_exe = _screen_(self._lzop_exe_)
          command = '%s -dc %s > %s' % (arc_exe, source, output)
          pstream(command).execute()
        elif self.filter == 'xz':
          arc_exe = _screen_(self._xz_exe_)
          command = '%s --format=xz -dc %s > %s' % (arc_exe, source, output)
          pstream(command).execute()
        elif self.filter == 'lzma':
          arc_exe = _screen_(self._xz_exe_)
          command = '%s --format=lzma -dc %s > %s' % (arc_exe, source, output)
          pstream(command).execute()
      except StreamError, error:
        self._rmtemp_() # Clear temporary directory
        raise StreamError(7, bytes(self.filter))
    else: # self.mode == 'wb' or self.mode == 'w'
      with codecs.open(self.path._stream_, 'w') as file:
        file.write('')
    if 'b' in self.mode: # open fstream in binary mode
      self._stream_ = \
        codecs.open(self._temppath_._stream_, self.mode._stream_)
    else: # open fstream in string mode
      self._stream_ = \
        codecs.open(self._temppath_._stream_, self.mode._stream_, __encoding__)
    if self.mode not in modes:
      self._rmtemp_() # Clear temporary directory
      raise StreamError(5, self.mode._stream_)
    if not isinstance(filter, NoneType) \
      and self.filter not in filters:
        self._rmtemp_() # Clear temporary directory
        raise StreamError(6, self.filter._stream_)
      
  def __enter__(self):
    return(self) # Return initialized object
    
  def __exit__(self, exc_type, exc_val, exc_tb):
    self.close() # Close using internal function
  
  def __iter__(self):
    return(self)
  
  def __next__(self):
    '''x.next() -> the next value, or raise StopIteration
    
    Object which user receives after supplying this method is based on mode
    which is used to open the fstream. Function returns bstream object if
    fstream was opened in binary mode, otherwise it returns ustream.'''
    if not self.closed and 'b' in self.mode: # binary mode
      result = self._stream_.next()
      return(bstream(result))
    elif not self.closed and 'b' not in self.mode: # binary mode
      result = self._stream_.next()
      return(ustream(result))
    else:
      raise StreamError(8, '__next__')
    
  def __repr__(self):
    '''x.__repr__() <==> repr(x)'''
    path = repr(self.path._stream_)
    mode = repr(self.mode._stream_)
    cursor = self.tell()
    if not isinstance(self.filter, NoneType):
      filter = repr(self.filter._stream_)
    else:
      filter = 'None'
    if not self.closed:
      return('<opened fstream object, path=%s, mode=%s, filter=%s, cursor=%d>'
        % (path, mode, filter, cursor))
    else:
      return('<closed fstream object, path=%s, mode=%s, filter=%s, cursor=%d>'
        % (path, mode, filter, cursor))
    
  cpdef _rmtemp_(self):
    os.remove(self._temppath_._stream_)
    os.remove(self._lzop_exe_._stream_)
    if __system__ == 'windows':
      os.remove(self._gzip_exe_._stream_)
      os.remove(self._bzip2_exe_._stream_)
      os.remove(self._bzip2_lib_._stream_)
      os.remove(self._xz_exe_._stream_)
    os.rmdir(self._tempdir_._stream_)
    
  cpdef close(self):
    '''fstream.close(stream)
    
    Close opened fstream. fstream.close() method may be called more than once
    without error; the first time it is called it removes all the temporary
    files and temporary directory.'''
    self.closed = True
    self._stream_.close()
    if self.changed:
      source = _screen_(self._temppath_)
      output = _screen_(self.path)
      if isinstance(self.filter, NoneType):
        shutil.copy(self._temppath_._stream_, self.path._stream_)
      elif self.filter == 'gzip':
        arc_exe = _screen_(self._gzip_exe_)
        command = '%s -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'bzip2':
        arc_exe = _screen_(self._bzip2_exe_)
        command = '%s -zc %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'lzop':
        arc_exe = _screen_(self._lzop_exe_)
        command = '%s -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'xz':
        arc_exe = _screen_(self._xz_exe_)
        command = '%s --format=xz -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'lzma':
        arc_exe = _screen_(self._xz_exe_)
        command = '%s --format=lzma -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      self._rmtemp_() # Clear temporary directory
    
  cpdef flush(self):
    '''fstream.flush()
    
    Flush the internal I/O buffer.'''
    if not self.closed and self.changed:
      self._stream_.flush()
      source = _screen_(self._temppath_)
      output = _screen_(self.path)
      if isinstance(self.filter, NoneType):
        shutil.copy(self._temppath_._stream_, self.path._stream_)
      elif self.filter == 'gzip':
        arc_exe = _screen_(self._gzip_exe_)
        command = '%s -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'bzip2':
        arc_exe = _screen_(self._bzip2_exe_)
        command = '%s -zc %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'lzop':
        arc_exe = _screen_(self._lzop_exe_)
        command = '%s -c %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'xz':
        arc_exe = _screen_(self._xz_exe_)
        command = '%s --format=xz -zc %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
      elif self.filter == 'lzma':
        arc_exe = _screen_(self._xz_exe_)
        command = '%s --format=lzma -zc %s > %s' % (arc_exe, source, output)
        pstream(command).execute()
    
  cpdef read(self, int size=-1):
    '''fstream.read([size]) -> bstream or ustream
    
    If the size argument is negative or omitted, read until EOF is reached.
    Notice that when in non-blocking mode, less data than what was requested
    may be returned, even if no size parameter was given.
    Object which user receives after supplying this method is based on mode
    which is used to open the fstream. Function returns bstream object if
    fstream was opened in binary mode, otherwise it returns ustream.'''
    if not self.closed:
      result = self._stream_.read(size)
      if 'b' in self.mode: # binary mode
        return(bstream(result))
      else: # string mode
        return(ustream(result))
    else:
      raise StreamError(8, 'read')
    
  def readline(self, size=-1):
    '''fstream.readline([size]) -> next line from the fstream
    
    Retain newline. A non-negative size argument limits the maximum
    number of bytes to return (an incomplete line may be returned then).
    Return an empty stream at EOF.
    Object which user receives after supplying this method is based on mode
    which is used to open the fstream. Function returns bstream object if
    fstream was opened in binary mode, otherwise it returns ustream.'''
    if not self.closed:
      result = self._stream_.readline(size)
      if 'b' in self.mode: # binary mode
        return(bstream(result))
      else: # string mode
        return(ustream(result))
    else:
      raise StreamError(8, 'readline')
    
  def readlines(self, size=-1):
    '''fstream.readlines([size]) -> list of streams
    
    Call fstream.readline() repeatedly and return a list of the lines to
    read. The optional size argument, if given, is an approximate bound
    on the total number of bytes in the lines returned.
    Object which user receives after supplying this method is based on mode
    which is used to open the fstream. Function returns bstream object if
    fstream was opened in binary mode, otherwise it returns ustream.'''
    if not self.closed:
      result = self._stream_.readlines(size)
      if 'b' in self.mode: # binary mode
        result = [bstream(item) for item in result]
        return(list(result))
      else: # string mode
        result = [ustream(item) for item in result]
        return(list(result))
    else:
      raise StreamError(8, 'readlines')
    
  def seek(self, offset, whence=0):
    '''fstream.seek(offset[, whence])
    
    Move to new fstream position. Argument offset is a byte count. Optional
    argument whence defaults to 0 (offset from start of fstream, offset should
    be >= 0); other values are 1 (move relative to current position, positive
    or negative), and 2 (move relative to end of fstream, usually negative,
    although many platforms allow seeking beyond the end of a fstream).
    If the fstream is opened in text mode, only offsets returned by tell() are
    legal. Use of other offsets causes undefined behavior.'''
    if not self.closed:
      self._stream_.seek(offset, whence)
    else:
      raise StreamError(8, 'seek')
    
  def tell(self):
    '''tell() -> current fstream position as integer'''
    if not self.closed:
      result = self._stream_.tell()
      return(int(result))
    else:
      raise StreamError(8, 'tell')
    
  def truncate(self, size=-1):
    '''fstream.truncate([size])
    
    Truncate the fstream to at most size bytes. Size defaults to the current
    fstream position, as returned by fstream.tell().'''
    if not self.closed:
      if size == -1:
        size = self._stream_.tell()
      self._stream_.truncate(size)
    else:
      raise StreamError(8, 'truncate')
    
  def write(self, stream):
    '''fstream.write(stream)
    
    Write stream to fstream. Note that due to buffering, flush() or close()
    may be needed before the fstream on disk reflects the data written.
    Due to some encoding problems for Windows, this method was designed
    to receive only bstream and ustream objects, which have no problems
    with encoding on all platforms.'''
    if not self.closed:
      if 'b' in self.mode:
        self._stream_.write(bstream(stream)._stream_)
      elif 'b' not in self.mode:
        self._stream_.write(ustream(stream)._stream_)
      self.changed = True
    else:
      raise StreamError(8, 'write')
    
  def writelines(self, sequence):
    '''fstream.writelines(sequence)
    
    Write the list of streams to the fstream. Note that newlines are not added.
    The sequence must be a list of strings and/or streams.'''
    if not self.closed:
      if 'b' in self.mode: # binary mode
        result = [bstream(item)._stream_ for item in sequence]
      else: # string mode
        result = [ustream(item)._stream_ for item in sequence]
      self._stream_.writelines(result)
      self.changed = True
    else:
      raise StreamError(8, 'writelines')

cdef class pstream:
  '''A pipeline stream to run an external applications. Subprocess can be
  used to execute an applications or to get stdout as bstream.
  This class was created as an alternative to default Python subprocess
  module which has some problems with encoding on Windows. It doesn't have
  all the possibilities of the original module, but yet can be useful.'''
  cdef readonly object _command_
  
  def __cinit__(self, args):
    '''pstream.__cinit__(args) -> pstream
    
    Create pstream. This function receives one argument, which must be a full
    command or a list of streams which are later used as options.'''
    try:
      from PyQt4 import QtCore
      types = \
      [
        str, unicode, bstream, ustream,
        QtCore.QByteArray, QtCore.QString
      ]
    except ImportError, error:
      types = [str, unicode, bstream, ustream]
    if isinstance(args, list):
      args = [ustream(arg) for arg in args]
      for arg in args:
        if arg == args[0]:
          self._command_ = \
            '%s' % arg._stream_.encode(__console__)
        else:
          self._command_ += \
            ' %s' % arg._stream_.encode(__console__)
    if type(args) in types:
      self._command_ = \
        ustream(args)._stream_.encode(__console__)
    
  def __repr__(self):
    '''x.__repr__() <==> repr(x)'''
    return('<pstream object, command=%s>' % repr(self._command_))
    
  def pipeline(self, process):
    '''pstream.pipeline(process) -> pstream
    
    Connect pstream with another pstream through pipeline. Programs can be run
    together such that one program reads the output from another with no need
    for an explicit intermediate file.
    
    >>> pstream('echo one\\ntwo\\nfour\\nfive').pipeline(pstream('grep f'))
    <pstream 'echo one\\ntwo\\nfour\\nfive | grep f'>'''
    if not isinstance(process, pstream):
      raise StreamError(10, type(process).__name__)
    result = '%s | %s' % (self._command_, process.command)
    return(pstream(result))
    
  def execute(self, exception=True, stdin=''):
    '''pstream.execute([exception[, stdin]) -> bstream or tuple
    
    Try to execute the specified command. If command was run successfully,
    function returns stdout as a bstream object. If an error occurred and
    the exception flag is True (default), StreamError is raised, otherwise
    function returns tuple from return code and stderr as a bstream object.'''
    import subprocess
    pipe = subprocess.PIPE
    process = \
      subprocess.Popen(self._command_, shell=True,
      stdin=pipe, stdout=pipe, stderr=pipe)
    if stdin != '':
      stdout, stderr = process.communicate(bstream(stdin)._stream_)
    else:
      stdout, stderr = process.communicate()
    retcode = int(process.returncode)
    stdout = bstream(stdout)
    stderr = bstream(stderr)
    if __system__ == 'windows':
      stdout = stdout.replace('\r\n', '\n')
      stderr = stderr.replace('\r\n', '\n')
    if retcode == 0:
      return(bstream(stdout))
    elif retcode != 0 and exception:
      raise StreamError(9, retcode)
    elif retcode != 0 and not exception:
      return(tuple([int(retcode), bstream(stderr)]))
