from contextlib import contextmanager, closing
from enum import Enum
from StringIO import StringIO
from abc import ABCMeta, abstractmethod


class Exportable(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def dump(self, f):
        pass

    def dumps(self):
        with closing(StringIO()) as f:
            self.dump(f)
            return f.getvalue()


class RestartPolicy(Enum):
    always = 'true'
    never = 'false'
    unexpected = 'unexpected'


class StopSignal(Enum):
    term = 'TERM'
    hup = 'HUP'
    int = 'INT'
    quit = 'QUIT'
    kill = 'KILL'
    usr1 = 'USR1'
    usr2 = 'USR2'


class LogMode(Enum):
    auto = 'AUTO'
    none = 'NONE'
    file = 'FILE'


class LogUnit(Enum):
    b = ''
    kb = 'KB'
    mb = 'MB'
    gb = 'GB'


class LogFile(object):
    def __init__(self, mode=LogMode.auto, location=None):
        if location is not None:
            self._mode = LogMode.file
            self._location = location
        else:
            self._mode = mode
            self._location = None

    @property
    def mode(self):
        return self._mode

    @property
    def location(self):
        return self._location

    def __str__(self):
        if self.location is None:
            return self.mode.value
        else:
            return self.location


class LogSize(object):
    def __init__(self, size=0, unit=LogUnit.b):
        self._size = size
        self._unit = unit

    @property
    def size(self):
        return self._size

    @property
    def unit(self):
        return self._unit

    def __str__(self):
        return '{0}{1}'.format(self.size, self.unit.value)


class Group(Exportable):
    def __init__(self, **opts):
        self._name = opts.get('name', None)
        self._programs = opts.get('programs', [])
        self._priority = opts.get('priority', 999)

    @property
    def name(self):
        return self._name

    @property
    def programs(self):
        return self._programs

    @property
    def priority(self):
        return self._priority

    def dump(self, f):
        for p in self.programs:
            p.dump(f)
            f.write('\n')

        f.write('[group:{0}]\n'.format(self.name))
        f.write('programs={0}\n'.format(','.join(x.name for x in self.programs)))
        f.write('priority={0:d}\n'.format(self.priority))
        f.write('\n')


class Program(Exportable):
    def __init__(self, **opts):
        self._name = opts.get('name', None)
        self._command = opts.get('command', None)
        self._process_name = opts.get('process_name', None)
        self._numprocs = opts.get('numprocs', 1)
        self._numprocs_start = opts.get('numprocs_start', 0)
        self._priority = opts.get('priority', 999)
        self._autostart = opts.get('autostart', True)
        self._autorestart = opts.get('autorestart', RestartPolicy.always)
        self._startsecs = opts.get('startsecs', 1)
        self._startretries = opts.get('startretries', 3)
        self._exitcodes = opts.get('exitcodes', [0, 2])
        self._stopsignal = opts.get('stopsignal', StopSignal.term)
        self._stopwaitsecs = opts.get('stopwaitsecs', 10)
        self._stopasgroup = opts.get('stopasgroup', False)
        self._killasgroup = opts.get('killasgroup', False)
        self._user = opts.get('user', None)
        self._redirect_stderr = opts.get('redirect_stderr', False)
        self._stdout_logfile = opts.get('stdout_logfile', LogFile())
        self._stdout_logfile_maxbytes = opts.get('stdout_logfile_maxbytes',
                                                 LogSize(size=50,
                                                         unit=LogUnit.mb))
        self._stdout_logfile_backups = opts.get('stdout_logfile_backups', 10)
        self._stdout_capture_maxbytes = opts.get('stdout_capture_maxbytes',
                                                 LogSize())
        self._stdout_events_enabled = opts.get('stdout_events_enabled', False)
        self._stderr_logfile = opts.get('stderr_logfile', LogFile())
        self._stderr_logfile_maxbytes = opts.get('stderr_logfile_maxbytes',
                                                 LogSize(size=50,
                                                         unit=LogUnit.mb))
        self._stderr_logfile_backups = opts.get('stderr_logfile_backups', 10)
        self._stderr_capture_maxbytes = opts.get('stderr_capture_maxbytes',
                                                 LogSize())
        self._stderr_events_enabled = opts.get('stderr_events_enabled', False)
        self._environment = opts.get('environment', {})
        self._directory = opts.get('directory', None)
        self._umask = opts.get('umask', None)
        self._serverurl = opts.get('serverurl', 'AUTO')

    @property
    def name(self):
        return self._name

    @property
    def command(self):
        return self._command

    @property
    def process_name(self):
        if self._process_name is None:
            if self._numprocs > 1:
                return '%(program_name)s-%(process_num)s'
            else:
                return '%(program_name)s'

        return self._process_name

    @property
    def numprocs(self):
        return self._numprocs

    @property
    def numprocs_start(self):
        return self._numprocs_start

    @property
    def priority(self):
        return self._priority

    @property
    def autostart(self):
        return self._autostart

    @property
    def autorestart(self):
        return self._autorestart

    @property
    def startsecs(self):
        return self._startsecs

    @property
    def startretries(self):
        return self._startretries

    @property
    def exitcodes(self):
        return self._exitcodes

    @property
    def stopsignal(self):
        return self._stopsignal

    @property
    def stopwaitsecs(self):
        return self._stopwaitsecs

    @property
    def stopasgroup(self):
        return self._stopasgroup

    @property
    def killasgroup(self):
        return self._killasgroup

    @property
    def user(self):
        return self._user

    @property
    def redirect_stderr(self):
        return self._redirect_stderr

    @property
    def stdout_logfile(self):
        return self._stdout_logfile

    @property
    def stdout_logfile_maxbytes(self):
        return self._stdout_logfile_maxbytes

    @property
    def stdout_logfile_backups(self):
        return self._stdout_logfile_backups

    @property
    def stdout_capture_maxbytes(self):
        return self._stdout_capture_maxbytes

    @property
    def stdout_events_enabled(self):
        return self._stdout_events_enabled

    @property
    def stderr_logfile(self):
        return self._stderr_logfile

    @property
    def stderr_logfile_maxbytes(self):
        return self._stderr_logfile_maxbytes

    @property
    def stderr_logfile_backups(self):
        return self._stderr_logfile_backups

    @property
    def stderr_capture_maxbytes(self):
        return self._stderr_capture_maxbytes

    @property
    def stderr_events_enabled(self):
        return self._stderr_events_enabled

    @property
    def environment(self):
        return self._environment

    @property
    def directory(self):
        return self._directory

    @property
    def umask(self):
        return self._umask

    @property
    def serverurl(self):
        return self._serverurl

    def dump(self, f):
        f.write('[program:{0}]\n'.format(self.name))
        f.write('command={0}\n'.format(self.command))
        f.write('process_name={0}\n'.format(self.process_name))
        f.write('numprocs={0:d}\n'.format(self.numprocs))
        f.write('numprocs_start={0:d}\n'.format(self.numprocs_start))
        f.write('priority={0:d}\n'.format(self.priority))
        f.write('autostart={0}\n'.format('true' if self.autostart else 'false'))
        f.write('autorestart={0}\n'.format(self.autorestart.value))
        f.write('startsecs={0:d}\n'.format(self.startsecs))
        f.write('startretries={0:d}\n'.format(self.startretries))
        if len(self.exitcodes) > 0:
            f.write('exitcodes={0}\n'.format(
                ','.join('{0:d}'.format(x) for x in self.exitcodes)))
        f.write('stopsignal={0}\n'.format(self.stopsignal.value))
        f.write('stopwaitsecs={0:d}\n'.format(self.stopwaitsecs))
        f.write('stopasgroup={0}\n'.format('true' if self.stopasgroup else
                                           'false'))
        f.write('killasgroup={0}\n'.format('true' if self.killasgroup else
                                           'false'))
        f.write('user={0}\n'.format(self.user))
        f.write('redirect_stderr={0}\n'.format('true' if self.redirect_stderr
                                               else 'false'))
        f.write('stdout_logfile={0}\n'.format(str(self.stdout_logfile)))
        f.write('stdout_logfile_maxbytes={0}\n'.format(str(
            self.stdout_logfile_maxbytes)))
        f.write('stdout_logfile_backups={0}\n'.format(str(
            self.stdout_logfile_backups)))
        f.write('stdout_capture_maxbytes={0}\n'.format(str(
            self.stdout_capture_maxbytes)))
        f.write('stdout_events_enabled={0}\n'.format(
            'true' if self.stdout_events_enabled else 'false'))
        if not self.redirect_stderr:
            f.write('stderr_logfile={0}\n'.format(str(self.stderr_logfile)))
            f.write('stderr_logfile_maxbytes={0}\n'.format(str(
                self.stderr_logfile_maxbytes)))
            f.write('stderr_logfile_backups={0}\n'.format(str(
                self.stderr_logfile_backups)))
            f.write('stderr_capture_maxbytes={0}\n'.format(str(
                self.stderr_capture_maxbytes)))
            f.write('stderr_events_enabled={0}\n'.format(
                'true' if self.stderr_events_enabled else 'false'))
        if len(self.environment) > 0:
            f.write('environment={0}\n'.format(
                ','.join('{0}="{1}"'.format(k, v) for k, v in self.environment.iteritems())))
        if self.directory is not None:
            f.write('directory={0}\n'.format(self.directory))
        if self.umask is not None:
            f.write('umask=0{0:o}\n'.format(self.umask))
        f.write('serverurl={0}\n'.format(self.serverurl))
        f.write('\n')


class EditableProgram(Program):
    def __init__(self, **opts):
        super(EditableProgram, self).__init__(**opts)

    @Program.command.setter
    def command(self, value):
        self._command = value

    @Program.process_name.setter
    def process_name(self, value):
        self._process_name = value

    @Program.numprocs.setter
    def numprocs(self, value):
        self._numprocs = value

    @Program.numprocs_start.setter
    def numprocs_start(self, value):
        self._numprocs_start = value

    @Program.priority.setter
    def priority(self, value):
        self._priority = value

    @Program.autostart.setter
    def autostart(self, value):
        self._autostart = value

    @Program.autorestart.setter
    def autorestart(self, value):
        self._autorestart = value

    @Program.startsecs.setter
    def startsecs(self, value):
        self._startsecs = value

    @Program.startretries.setter
    def startretries(self, value):
        self._startretries = value

    @Program.exitcodes.setter
    def exitcodes(self, value):
        self._exitcodes = value

    def exitcode(self, value):
        self._exitcodes.append(value)

    @Program.stopsignal.setter
    def stopsignal(self, value):
        self._stopsignal = value

    @Program.stopwaitsecs.setter
    def stopwaitsecs(self, value):
        self._stopwaitsecs = value

    @Program.stopasgroup.setter
    def stopasgroup(self, value):
        self._stopasgroup = value

    @Program.killasgroup.setter
    def killasgroup(self, value):
        self._killasgroup = value

    @Program.user.setter
    def user(self, value):
        self._user = value

    @Program.redirect_stderr.setter
    def redirect_stderr(self, value):
        self._redirect_stderr = value

    @Program.stdout_logfile.setter
    def stdout_logfile(self, value):
        self._stdout_logfile = value

    @Program.stdout_logfile_maxbytes.setter
    def stdout_logfile_maxbytes(self, value):
        self._stdout_logfile_maxbytes = value

    @Program.stdout_logfile_backups.setter
    def stdout_logfile_backups(self, value):
        self._stdout_logfile_backups = value

    @Program.stdout_capture_maxbytes.setter
    def stdout_capture_maxbytes(self, value):
        self._stdout_capture_maxbytes = value

    @Program.stdout_events_enabled.setter
    def stdout_events_enabled(self, value):
        self._stdout_events_enabled = value

    @Program.stderr_logfile.setter
    def stderr_logfile(self, value):
        self._stderr_logfile = value

    @Program.stderr_logfile_maxbytes.setter
    def stderr_logfile_maxbytes(self, value):
        self._stderr_logfile_maxbytes = value

    @Program.stderr_logfile_backups.setter
    def stderr_logfile_backups(self, value):
        self._stderr_logfile_backups = value

    @Program.stderr_capture_maxbytes.setter
    def stderr_capture_maxbytes(self, value):
        self._stderr_capture_maxbytes = value

    @Program.stderr_events_enabled.setter
    def stderr_events_enabled(self, value):
        self._stderr_events_enabled = value

    @Program.environment.setter
    def environment(self, value):
        self._environment = value

    def env_variable(self, key, value):
        self._environment[key] = value

    @Program.directory.setter
    def directory(self, value):
        self._directory = value

    @Program.umask.setter
    def umask(self, value):
        self._umask = value

    @Program.serverurl.setter
    def serverurl(self, value):
        self._serverurl = value


class EditableGroup(Group):
    def __init__(self, **opts):
        super(EditableGroup, self).__init__(**opts)

    @contextmanager
    def program(self, name):
        p = EditableProgram(name=name)
        yield p
        self._programs.append(p)

    @Group.priority.setter
    def priority(self, value):
        self._priority = value


@contextmanager
def group(name):
    g = EditableGroup(name=name)
    yield g


@contextmanager
def program(name):
    p = EditableProgram(name=name)
    yield p
