from contextlib import contextmanager, closing
from enum import Enum
from StringIO import StringIO
from abc import ABCMeta, abstractmethod
import ConfigParser


def format_boolean(b):
    return 'true' if b else 'false'


class Exportable(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def dump(self, f):
        pass

    def dumps(self):
        with closing(StringIO()) as f:
            self.dump(f)
            return f.getvalue()


class ConfigSection(object):
    def __init__(self, name, base):
        self._name = name
        self._base = base

        self.base.exportable.add_section(self.name)

    @property
    def name(self):
        return self._name

    @property
    def base(self):
        return self._base

    def entry(self, key, value):
        self.base.exportable.set(self.name, key, value)


class ConfigBase(Exportable):
    def __init__(self):
        self._exportable = ConfigParser.RawConfigParser()
        self._sections = []

    @property
    def exportable(self):
        return self._exportable

    @contextmanager
    def section(self, name):
        s = ConfigSection(name, self)
        yield s
        self._sections.append(s)

    def dump(self, f):
        self.exportable.write(f)


class RestartPolicy(Enum):
    always = 'true'
    never = 'false'
    unexpected = 'unexpected'


class StopSignal(Enum):
    term = 'TERM'
    hup = 'HUP'
    int = 'INT'
    quit = 'QUIT'
    kill = 'KILL'
    usr1 = 'USR1'
    usr2 = 'USR2'


class LogMode(Enum):
    auto = 'AUTO'
    none = 'NONE'
    file = 'FILE'


class LogUnit(Enum):
    b = ''
    kb = 'KB'
    mb = 'MB'
    gb = 'GB'


class LogFile(object):
    def __init__(self, mode=LogMode.auto, location=None):
        if location is not None:
            self._mode = LogMode.file
            self._location = location
        else:
            self._mode = mode
            self._location = None

    @property
    def mode(self):
        return self._mode

    @property
    def location(self):
        return self._location

    def __str__(self):
        if self.location is None:
            return self.mode.value
        else:
            return self.location


class LogSize(object):
    def __init__(self, size=0, unit=LogUnit.b):
        self._size = size
        self._unit = unit

    @property
    def size(self):
        return self._size

    @property
    def unit(self):
        return self._unit

    def __str__(self):
        return '{0}{1}'.format(self.size, self.unit.value)


class GroupBase(Exportable):
    def __init__(self, **opts):
        self._name = opts.get('name', None)
        self._programs = opts.get('programs', [])
        self._priority = opts.get('priority', 999)

    @property
    def name(self):
        return self._name

    @property
    def programs(self):
        return self._programs

    @property
    def priority(self):
        return self._priority

    def _export(self, config):
        with config.section('group:{0}'.format(self.name)) as s:
            s.entry('programs', ','.join(x.name for x in self.programs))
            s.entry('priority', str(self.priority))

    def dump(self, f):
        config = ConfigBase()
        for p in self.programs:
            p._export(config)
        self._export(config)
        config.dump(f)


class ProgramBase(Exportable):
    def __init__(self, **opts):
        self._name = opts.get('name', None)
        self._command = opts.get('command', None)
        self._process_name = opts.get('process_name', None)
        self._numprocs = opts.get('numprocs', 1)
        self._numprocs_start = opts.get('numprocs_start', 0)
        self._priority = opts.get('priority', 999)
        self._autostart = opts.get('autostart', True)
        self._autorestart = opts.get('autorestart', RestartPolicy.always)
        self._startsecs = opts.get('startsecs', 1)
        self._startretries = opts.get('startretries', 3)
        self._exitcodes = opts.get('exitcodes', [0, 2])
        self._stopsignal = opts.get('stopsignal', StopSignal.term)
        self._stopwaitsecs = opts.get('stopwaitsecs', 10)
        self._stopasgroup = opts.get('stopasgroup', False)
        self._killasgroup = opts.get('killasgroup', False)
        self._user = opts.get('user', None)
        self._redirect_stderr = opts.get('redirect_stderr', False)
        self._stdout_logfile = opts.get('stdout_logfile', LogFile())
        self._stdout_logfile_maxbytes = opts.get('stdout_logfile_maxbytes',
                                                 LogSize(size=50,
                                                         unit=LogUnit.mb))
        self._stdout_logfile_backups = opts.get('stdout_logfile_backups', 10)
        self._stdout_capture_maxbytes = opts.get('stdout_capture_maxbytes',
                                                 LogSize())
        self._stdout_events_enabled = opts.get('stdout_events_enabled', False)
        self._stderr_logfile = opts.get('stderr_logfile', LogFile())
        self._stderr_logfile_maxbytes = opts.get('stderr_logfile_maxbytes',
                                                 LogSize(size=50,
                                                         unit=LogUnit.mb))
        self._stderr_logfile_backups = opts.get('stderr_logfile_backups', 10)
        self._stderr_capture_maxbytes = opts.get('stderr_capture_maxbytes',
                                                 LogSize())
        self._stderr_events_enabled = opts.get('stderr_events_enabled', False)
        self._environment = opts.get('environment', {})
        self._directory = opts.get('directory', None)
        self._umask = opts.get('umask', None)
        self._serverurl = opts.get('serverurl', 'AUTO')

    @property
    def name(self):
        return self._name

    @property
    def command(self):
        return self._command

    @property
    def process_name(self):
        if self._process_name is None:
            if self._numprocs > 1:
                return '%(program_name)s-%(process_num)s'
            else:
                return '%(program_name)s'

        return self._process_name

    @property
    def numprocs(self):
        return self._numprocs

    @property
    def numprocs_start(self):
        return self._numprocs_start

    @property
    def priority(self):
        return self._priority

    @property
    def autostart(self):
        return self._autostart

    @property
    def autorestart(self):
        return self._autorestart

    @property
    def startsecs(self):
        return self._startsecs

    @property
    def startretries(self):
        return self._startretries

    @property
    def exitcodes(self):
        return self._exitcodes

    @property
    def stopsignal(self):
        return self._stopsignal

    @property
    def stopwaitsecs(self):
        return self._stopwaitsecs

    @property
    def stopasgroup(self):
        return self._stopasgroup

    @property
    def killasgroup(self):
        return self._killasgroup

    @property
    def user(self):
        return self._user

    @property
    def redirect_stderr(self):
        return self._redirect_stderr

    @property
    def stdout_logfile(self):
        return self._stdout_logfile

    @property
    def stdout_logfile_maxbytes(self):
        return self._stdout_logfile_maxbytes

    @property
    def stdout_logfile_backups(self):
        return self._stdout_logfile_backups

    @property
    def stdout_capture_maxbytes(self):
        return self._stdout_capture_maxbytes

    @property
    def stdout_events_enabled(self):
        return self._stdout_events_enabled

    @property
    def stderr_logfile(self):
        return self._stderr_logfile

    @property
    def stderr_logfile_maxbytes(self):
        return self._stderr_logfile_maxbytes

    @property
    def stderr_logfile_backups(self):
        return self._stderr_logfile_backups

    @property
    def stderr_capture_maxbytes(self):
        return self._stderr_capture_maxbytes

    @property
    def stderr_events_enabled(self):
        return self._stderr_events_enabled

    @property
    def environment(self):
        return self._environment

    @property
    def directory(self):
        return self._directory

    @property
    def umask(self):
        return self._umask

    @property
    def serverurl(self):
        return self._serverurl

    def _export(self, config):
        with config.section('program:{0}'.format(self.name)) as s:
            s.entry('command', self.command)
            s.entry('process_name', self.process_name)
            s.entry('numprocs', str(self.numprocs))
            s.entry('numprocs_start', str(self.numprocs_start))
            s.entry('priority', str(self.priority))
            s.entry('autostart', format_boolean(self.autostart))
            s.entry('autorestart', self.autorestart.value)
            s.entry('startsecs', str(self.startsecs))
            s.entry('startretries', str(self.startretries))
            if len(self.exitcodes) > 0:
                s.entry('exitcodes', ','.join(str(x) for x in self.exitcodes))
            s.entry('stopsignal', self.stopsignal.value)
            s.entry('stopwaitsecs', str(self.stopwaitsecs))
            s.entry('stopasgroup', format_boolean(self.stopasgroup))
            s.entry('killasgroup', format_boolean(self.killasgroup))
            s.entry('user', self.user)
            s.entry('redirect_stderr', format_boolean(self.redirect_stderr))
            s.entry('stdout_logfile', str(self.stdout_logfile))
            s.entry('stdout_logfile_maxbytes', str(self.stdout_logfile_maxbytes))
            s.entry('stdout_logfile_backups', str(self.stdout_logfile_backups))
            s.entry('stdout_capture_maxbytes', str(self.stdout_capture_maxbytes))
            s.entry('stdout_events_enabled', format_boolean(self.stdout_events_enabled))
            if not self.redirect_stderr:
                s.entry('stderr_logfile', str(self.stderr_logfile))
                s.entry('stderr_logfile_maxbytes', str(self.stderr_logfile_maxbytes))
                s.entry('stderr_logfile_backups', str(self.stderr_logfile_backups))
                s.entry('stderr_capture_maxbytes', str(self.stderr_capture_maxbytes))
                s.entry('stderr_events_enabled', format_boolean(self.stderr_events_enabled))
            if len(self.environment) > 0:
                s.entry('environment', ','.join('{0}="{1}"'.format(k, v)
                        for k, v in self.environment.iteritems()))
            if self.directory is not None:
                s.entry('directory', self.directory)
            if self.umask is not None:
                s.entry('umask', '0{0:o}'.format(self.umask))
            s.entry('serverurl', self.serverurl)

    def dump(self, f):
        config = ConfigBase()
        self._export(config)
        config.dump(f)


class Program(ProgramBase):
    def __init__(self, **opts):
        super(Program, self).__init__(**opts)

    @ProgramBase.command.setter
    def command(self, value):
        self._command = value

    @ProgramBase.process_name.setter
    def process_name(self, value):
        self._process_name = value

    @ProgramBase.numprocs.setter
    def numprocs(self, value):
        self._numprocs = value

    @ProgramBase.numprocs_start.setter
    def numprocs_start(self, value):
        self._numprocs_start = value

    @ProgramBase.priority.setter
    def priority(self, value):
        self._priority = value

    @ProgramBase.autostart.setter
    def autostart(self, value):
        self._autostart = value

    @ProgramBase.autorestart.setter
    def autorestart(self, value):
        self._autorestart = value

    @ProgramBase.startsecs.setter
    def startsecs(self, value):
        self._startsecs = value

    @ProgramBase.startretries.setter
    def startretries(self, value):
        self._startretries = value

    @ProgramBase.exitcodes.setter
    def exitcodes(self, value):
        self._exitcodes = value

    def exitcode(self, value):
        self._exitcodes.append(value)

    @ProgramBase.stopsignal.setter
    def stopsignal(self, value):
        self._stopsignal = value

    @ProgramBase.stopwaitsecs.setter
    def stopwaitsecs(self, value):
        self._stopwaitsecs = value

    @ProgramBase.stopasgroup.setter
    def stopasgroup(self, value):
        self._stopasgroup = value

    @ProgramBase.killasgroup.setter
    def killasgroup(self, value):
        self._killasgroup = value

    @ProgramBase.user.setter
    def user(self, value):
        self._user = value

    @ProgramBase.redirect_stderr.setter
    def redirect_stderr(self, value):
        self._redirect_stderr = value

    @ProgramBase.stdout_logfile.setter
    def stdout_logfile(self, value):
        self._stdout_logfile = value

    @ProgramBase.stdout_logfile_maxbytes.setter
    def stdout_logfile_maxbytes(self, value):
        self._stdout_logfile_maxbytes = value

    @ProgramBase.stdout_logfile_backups.setter
    def stdout_logfile_backups(self, value):
        self._stdout_logfile_backups = value

    @ProgramBase.stdout_capture_maxbytes.setter
    def stdout_capture_maxbytes(self, value):
        self._stdout_capture_maxbytes = value

    @ProgramBase.stdout_events_enabled.setter
    def stdout_events_enabled(self, value):
        self._stdout_events_enabled = value

    @ProgramBase.stderr_logfile.setter
    def stderr_logfile(self, value):
        self._stderr_logfile = value

    @ProgramBase.stderr_logfile_maxbytes.setter
    def stderr_logfile_maxbytes(self, value):
        self._stderr_logfile_maxbytes = value

    @ProgramBase.stderr_logfile_backups.setter
    def stderr_logfile_backups(self, value):
        self._stderr_logfile_backups = value

    @ProgramBase.stderr_capture_maxbytes.setter
    def stderr_capture_maxbytes(self, value):
        self._stderr_capture_maxbytes = value

    @ProgramBase.stderr_events_enabled.setter
    def stderr_events_enabled(self, value):
        self._stderr_events_enabled = value

    @ProgramBase.environment.setter
    def environment(self, value):
        self._environment = value

    def env_variable(self, key, value):
        self._environment[key] = value

    @ProgramBase.directory.setter
    def directory(self, value):
        self._directory = value

    @ProgramBase.umask.setter
    def umask(self, value):
        self._umask = value

    @ProgramBase.serverurl.setter
    def serverurl(self, value):
        self._serverurl = value


class Group(GroupBase):
    def __init__(self, **opts):
        super(Group, self).__init__(**opts)

    @contextmanager
    def program(self, name):
        p = Program(name=name)
        yield p
        self._programs.append(p)

    @GroupBase.priority.setter
    def priority(self, value):
        self._priority = value


@contextmanager
def group(name):
    g = Group(name=name)
    yield g


@contextmanager
def program(name):
    p = Program(name=name)
    yield p
