"""
**********
Provenance
**********

The Activity object represents the data processing steps used to produce a data
entity in Synapse. Using `W3C provenance ontology <http://www.w3.org/2011/prov/wiki/Main_Page>`_ 
terms, a result is generated by a combination of code and data artifacts which are
**used**. Code is said to be **executed**.

To create an activity::

    act = Activity(name='clustering', description='hierarchical clustering', used=['syn1234','syn1235'], executed='syn4567')

Alternatively, you can build an activity up piecemeal::

    act = Activity(name='clustering', description='hierarchical clustering')
    act.used(['syn12345', 'syn12346'])
    act.executed('syn12347')

The activity can be passed in when storing an Entity to link set the Entity's provenance::

    syn.store(my_entity, activity=act)


~~~~~~~~
Activity
~~~~~~~~

.. autoclass:: synapseclient.activity.Activity
   :members:
   
.. automethod:: synapseclient.activity.is_used_entity
.. automethod:: synapseclient.activity.is_used_url

"""

import collections

from synapseclient.utils import is_url, id_of, is_synapse_entity


def is_used_entity(x):
    """
    Does the given object represent a UsedEntity?
    """
    if not isinstance(x, collections.Mapping):
        return False
    if 'reference' not in x:
        return False
    for k in x:
        if k not in ('reference', 'wasExecuted', 'concreteType',):
            return False
    if 'targetId' not in x['reference']:
        return False
    for k in x['reference']:
        if k not in ('targetId', 'targetVersionNumber',):
            return False
    return True


def is_used_url(x):
    """
    Does the given object represent a UsedURL?
    """
    if not isinstance(x, collections.Mapping):
        return False
    if 'url' not in x:
        return False
    for k in x:
        if k not in ('url', 'name', 'wasExecuted', 'concreteType',):
            return False
    return True


class Activity(dict):
    """
    Represents the provenance of a Synapse entity.
    
    :param name:        TODO_Sphinx
    :param description: TODO_Sphinx
    :param used:        Either a list of reference objects 
                        (e.g. ``[{'targetId':'syn123456', 'targetVersionNumber':1}]``) 
                        or a list of Synapse Entities or Entity IDs
    :param data:        A dictionary representation of an Activity, 
                        with fields 'name', 'description' and 'used' 
                        (a list of reference objects)
        
    See also: `Provenance in Synapse <https://sagebionetworks.jira.com/wiki/display/PLFM/Analysis+Provenance+in+Synapse>`_ 
        and the `W3C's provenance ontology <http://www.w3.org/TR/prov-o/>`_
    """

    ## TODO: make constructors from JSON consistent across objects
    def __init__(self, name=None, description=None, used=None, executed=None, data=None):
        # Initialize from a dictionary, as in Activity(data=response.json())
        if data:
            super(Activity, self).__init__(data)
        if name:
            self['name'] = name
        if description:
            self['description'] = description
        if 'used' not in self:
            self['used'] = []
        if used:
            self.used(used)
        if executed:
            self.executed(executed)


    def usedEntity(self, target, targetVersion=None, wasExecuted=False):
        """
        TODO_Sphinx
        
        :param target:        either a synapse entity or entity id (as a string)
        :param targetVersion: optionally specify the version of the entity
        :param wasExecuted:   boolean indicating whether the entity represents code that was executed to produce the result
        """
        
        reference = {'targetId':id_of(target)}
        if targetVersion:
            reference['targetVersionNumber'] = int(targetVersion)
        else:
            try:
                # If we have an Entity, get it's version number
                reference['targetVersionNumber'] = target['versionNumber']
            except (KeyError, TypeError):
                # Count on platform to get the current version of the entity from Synapse
                pass
        self['used'].append({'reference':reference, 'wasExecuted':wasExecuted, 'concreteType':'org.sagebionetworks.repo.model.provenance.UsedEntity'})


    def usedURL(self, url, name=None, wasExecuted=False):
        """
        TODO_Sphinx
        
        :param url:         resource's URL as a string
        :param name:        optionally name the indicated resource, defaults to the URL
        :param wasExecuted: boolean indicating whether the entity represents code that was executed to produce the result
        """
        
        if name is None:
            name = url
        self['used'].append({'url':url, 'name':name, 'wasExecuted':wasExecuted, 'concreteType':'org.sagebionetworks.repo.model.provenance.UsedURL'})


    def used(self, target=None, targetVersion=None, wasExecuted=None, url=None, name=None):
        """
        Add a resource used by the activity.

        This method tries to be as permissive as possible. It accepts a string which might
        be a synapse ID or a URL, a synapse entity, a UsedEntity or UsedURL dictionary or
        a list containing any combination of these.

        In addition, named parameters can be used to specify the fields of either a
        UsedEntity or a UsedURL. If target and optionally targetVersion are specified,
        create a UsedEntity. If url and optionally name are specified, create a UsedURL.

        It is an error to specify both target/targetVersion parameters and url/name
        parameters in the same call. To add multiple UsedEntities and UsedURLs, make a
        separate call for each or pass in a list.

        In case of conflicting settings for wasExecuted both inside an object and with a
        parameter, the parameter wins. For example, this UsedURL will have wasExecuted set
        to False::
            
            activity.used({'url':'http://google.com', 'name':'Goog', 'wasExecuted':True}, wasExecuted=False)

        Entity examples::
        
            activity.used('syn12345')
            activity.used(entity)
            activity.used(target=entity, targetVersion=2)
            activity.used(codeEntity, wasExecuted=True)
            activity.used({'reference':{'target':'syn12345', 'targetVersion':1}, 'wasExecuted':False})

        URL examples::
        
            activity.used('http://mydomain.com/my/awesome/data.RData')
            activity.used(url='http://mydomain.com/my/awesome/data.RData', name='Awesome Data')
            activity.used(url='https://github.com/joe_hacker/code_repo', name='Gnarly hacks', wasExecuted=True)
            activity.used({'url':'https://github.com/joe_hacker/code_repo', 'name':'Gnarly hacks'}, wasExecuted=True)

        List example::
        
            used( [ 'syn12345', 'syn23456', entity,
                    {'reference':{'target':'syn100009', 'targetVersion':2}, 'wasExecuted':True},
                    'http://mydomain.com/my/awesome/data.RData' ] )
        """

        # Check for allowed combinations of parameters and generate specific error message
        # based on the context. For example, if we specify a URL, it's illegal to specify
        # a version.
        def check_for_invalid_parameters(context=None, params={}):
            err_msg = 'Error in call to Activity.used()'
            
            if context == 'list':
                illegal_params = ('targetVersion', 'url', 'name',)
                context_msg = 'list of used resources'
            elif context == 'dict':
                illegal_params = ('targetVersion', 'url', 'name',)
                context_msg = 'dictionary representing a used resource'
            elif context == 'url_param':
                illegal_params = ('target', 'targetVersion',)
                context_msg = 'URL'
            elif context == 'url_string':
                illegal_params = ('targetVersion',)
                context_msg = 'URL'
            elif context == 'entity':
                illegal_params = ('url', 'name',)
                context_msg = 'Synapse entity'
            else:
                illegal_params = ()
                context_msg = '?'

            for param in illegal_params:
                if param in params and params[param] is not None:
                    raise Exception('%s: It is an error to specify the \'%s\' parameter in combination with a %s.' % (err_msg, str(param), context_msg))

        # List
        if isinstance(target, list):
            check_for_invalid_parameters(context='list', params=locals())
            for item in target:
                self.used(item, wasExecuted=wasExecuted)
            return

        # Used Entity
        elif is_used_entity(target):
            check_for_invalid_parameters(context='dict', params=locals())
            resource = target
            if 'concreteType' not in resource:
                resource['concreteType'] = 'org.sagebionetworks.repo.model.provenance.UsedEntity'

        # Used URL
        elif is_used_url(target):
            check_for_invalid_parameters(context='dict', params=locals())
            resource = target
            if 'concreteType' not in resource:
                resource['concreteType'] = 'org.sagebionetworks.repo.model.provenance.UsedURL'

        #  Synapse Entity
        elif is_synapse_entity(target):
            check_for_invalid_parameters(context='entity', params=locals())
            reference = {'targetId':target['id']}
            if 'versionNumber' in target:
                reference['targetVersionNumber'] = target['versionNumber']
            ## if targetVersion is specified as a parameter, it overrides the version in the object
            if targetVersion:
                reference['targetVersionNumber'] = int(targetVersion)
            resource = {'reference':reference, 'concreteType':'org.sagebionetworks.repo.model.provenance.UsedEntity'}

        # URL parameter
        elif url:
            check_for_invalid_parameters(context='url_param', params=locals())
            resource = {'url':url, 'name':name if name else target, 'concreteType':'org.sagebionetworks.repo.model.provenance.UsedURL'}

        # URL as a string
        elif is_url(target):
            check_for_invalid_parameters(context='url_string', params=locals())
            resource = {'url':target, 'name':name if name else target, 'concreteType':'org.sagebionetworks.repo.model.provenance.UsedURL'}

        # If it's a string and isn't a URL, assume it's a Synapse Entity ID
        elif isinstance(target, basestring):
            check_for_invalid_parameters(context='entity', params=locals())
            reference = {'targetId':target}
            if targetVersion:
                reference['targetVersionNumber'] = int(targetVersion)
            resource = {'reference':reference, 'concreteType':'org.sagebionetworks.repo.model.provenance.UsedEntity'}

        else:
            raise Exception('Unexpected parameters in call to Activity.used().')

        # Set wasExecuted
        if wasExecuted is None:
            # Default to False
            if 'wasExecuted' not in resource:
                resource['wasExecuted'] = False
        else:
            # wasExecuted parameter overrides setting in an object
            resource['wasExecuted'] = wasExecuted

        # Add the used resource to the activity
        self['used'].append(resource)


    def executed(self, target=None, targetVersion=None, url=None, name=None):
        """
        Add a code resource that was executed during the activity.
        See :py:func:`synapseclient.activity.Activity.used`
        """
        
        self.used(target=target, targetVersion=targetVersion, url=url, name=name, wasExecuted=True)
