from urlparse import urljoin
import requests
import convertible

from config import Service


class ServiceUrls:
    def __init__(self, device_domain, service, mapping):
        self.domain_port = "{}:{}".format(device_domain, mapping.external_port)
        self.full_url = "{}://{}/{}".format(service.protocol, self.domain_port, service.url)


class Dns:

    def __init__(self, api_url, main_domain_name, domain_config, service_config, port_mapper):
        self.api_url = api_url
        self.main_domain_name = main_domain_name
        self.domain_config = domain_config
        self.service_config = service_config
        self.port_mapper = port_mapper

    def check_http_error(self, response):
        if not response.status_code == 200:
            error = convertible.from_json(response.text)
            raise Exception(error.message)

    def acquire(self, email, password, user_domain):
        data = {'email': email, 'password': password, 'user_domain': user_domain}
        url = urljoin(self.api_url, "/domain/acquire")
        response = requests.post(url, data)
        self.check_http_error(response)
        domain = convertible.from_json(response.text)
        self.domain_config.save(domain)
        return domain

    def drop(self):
        self.domain_config.remove()

    def add_service(self, name, protocol, type, port, url=None):
        self.port_mapper.add(port)
        new_service = Service(name, protocol, type, port, url)
        self.service_config.add(new_service)

    def get_service(self, name):
        return self.service_config.get(name)

    def get_service_by_port(self, port):
        return self.service_config.get(port)

    def remove_service(self, name):
        service = self.get_service(name)
        self.service_config.remove(name)
        self.port_mapper.remove(service.port)

    def full_name(self):
        domain_config = self.domain_config.load()
        return 'device.{}.{}'.format(domain_config.user_domain, self.main_domain_name)

    def service_urls(self, name):
        service = self.get_service(name)
        mapping = self.port_mapper.get(service.port)
        return ServiceUrls(self.full_name(), service, mapping)

    def sync(self):
        self.port_mapper.sync()

        services = self.service_config.load()
        mappings = self.port_mapper.list()
        domain = self.domain_config.load()
        external_ip = self.port_mapper.external_ip()
        if len(mappings) == 0:
            raise Exception("no port mappings defined (need at least one), not updating dns")
        if not domain:
            raise Exception("no token saved, need to call set_dns or get_dns_token first")
        if not external_ip:
            raise Exception("unable to get external ip")

        for service in services:
            mapping = self.port_mapper.get(service.port)
            if mapping:
                service.port = mapping.external_port
            else:
                services.remove(service)

        data = {'update_token': domain.token, 'ip': external_ip, 'services': services}

        url = urljoin(self.api_url, "/domain/update")

        response = requests.post(url, convertible.to_json(data))

        self.check_http_error(response)