import unittest
import responses
from mock import MagicMock

from insider.dns import Dns
from insider.config import Port, Domain, Service

from insider.port_mapper import PortMapper

from convertible import reformat

from helpers import get_port_config, get_domain_config, get_service_config

def get_upnpc(external_ip):
    upnpc = MagicMock()
    upnpc.external_ip = MagicMock(return_value=external_ip)
    return upnpc

def get_port_mapper(port_config, upnpc):
    return PortMapper(port_config, upnpc)

class TestDns(unittest.TestCase):

    def assertSingleRequest(self, expected_request):
        expected_request = reformat(expected_request)
        self.assertEquals(1, len(responses.calls))
        the_call = responses.calls[0]
        self.assertEquals(expected_request, the_call.request.body)

    @responses.activate
    def test_sync_success(self):
        service_config = get_service_config([
            Service("ownCloud", "http", "_http._tcp", 80, url="owncloud"),
            Service("SSH", "https", "_http._tcp", 81, url=None)
        ])
        port_config = get_port_config([Port(80, 10000), Port(81, 10001)])
        upnpc = get_upnpc(external_ip='192.167.44.52')
        port_mapper = get_port_mapper(port_config, upnpc)
        domain_config = get_domain_config(Domain('boris', 'some_update_token'))

        responses.add(responses.POST,
                      "http://api.domain.com/domain/update",
                      status=200,
                      body="{'message': 'Domain was updated'}",
                      content_type="application/json")

        dns = Dns('http://api.domain.com', 'domain.com', domain_config, service_config, port_mapper)
        dns.sync()

        expected_request = '''
{
    "services": [
        {"name": "ownCloud", "protocol": "http", "type": "_http._tcp", "port": 10000, "url": "owncloud"},
        {"name": "SSH", "protocol": "https", "type": "_http._tcp", "port": 10001, "url": null}
    ],
    "ip": "192.167.44.52",
    "token": "some_update_token"
}
'''

        self.assertSingleRequest(expected_request)

    @responses.activate
    def test_sync_server_error(self):
        service_config = get_service_config([Service("ownCloud", "http", "_http._tcp", 80, url="owncloud")])
        port_config = get_port_config([Port(80, 10000)])
        upnpc = get_upnpc(external_ip='192.167.44.52')
        port_mapper = get_port_mapper(port_config, upnpc)
        domain_config = get_domain_config(Domain('boris', 'some_update_token'))

        responses.add(responses.POST,
                      "http://api.domain.com/domain/update",
                      status=400,
                      body='{"message": "Unknown update token"}',
                      content_type="application/json")

        dns = Dns('http://api.domain.com', 'domain.com', domain_config, service_config, port_mapper)

        with self.assertRaises(Exception) as context:
            dns.sync()

        self.assertEquals(context.exception.message, "Unknown update token")

    @responses.activate
    def test_link_success(self):
        domain_config = get_domain_config(None)

        responses.add(responses.POST,
                      "http://api.domain.com/domain/acquire",
                      status=200,
                      body='{"user_domain": "boris", "update_token": "some_update_token"}',
                      content_type="application/json")

        dns = Dns('http://api.domain.com', 'domain.com', domain_config, service_config=None, port_mapper=None)
        result = dns.acquire('boris@mail.com', 'pass1234', 'boris')

        self.assertIsNotNone(result)
        self.assertEquals(result.user_domain, "boris")
        self.assertEquals(result.update_token, "some_update_token")

        # expected_request = '{"password": "pass1234", "email": "boris@mail.com", "user_domain": "boris"}'
        # self.assertSingleRequest(expected_request)

        domain = domain_config.load()
        self.assertIsNotNone(domain)
        self.assertEquals(domain.user_domain, "boris")
        self.assertEquals(domain.update_token, "some_update_token")

    @responses.activate
    def test_link_server_error(self):
        domain_config = get_domain_config(None)

        responses.add(responses.POST,
                      "http://api.domain.com/domain/acquire",
                      status=403,
                      body='{"message": "Authentication failed"}',
                      content_type="application/json")

        dns = Dns('http://api.domain.com', 'domain.com', domain_config, service_config=None, port_mapper=None)
        with self.assertRaises(Exception) as context:
            result = dns.acquire('boris@mail.com', 'pass1234', 'boris')

        self.assertEquals(context.exception.message, "Authentication failed")

        domain = domain_config.load()
        self.assertIsNone(domain)

    def test_add_service(self):
        service_config = get_service_config([])
        port_config = get_port_config([])
        upnpc = get_upnpc(external_ip='192.167.44.52')
        port_mapper = get_port_mapper(port_config, upnpc)

        domain_config = get_domain_config(None)

        dns = Dns('http://api.domain.com', 'domain.com', domain_config, service_config, port_mapper)
        dns.add_service("ownCloud", "http", "_http._tcp", 80, url="owncloud")

        services = service_config.load()
        self.assertEquals(1, len(services))
        service = services[0]
        self.assertEquals("ownCloud", service.name)
        self.assertEquals("_http._tcp", service.type)
        self.assertEquals(80, service.port)
        self.assertEquals("owncloud", service.url)

        mappings = port_config.load()
        self.assertEquals(1, len(mappings))
        mapping = mappings[0]
        self.assertEquals(80, mapping.local_port)

    def test_get_service(self):
        service_config = get_service_config([])
        port_config = get_port_config([])
        upnpc = get_upnpc(external_ip='192.167.44.52')
        port_mapper = get_port_mapper(port_config, upnpc)

        domain_config = get_domain_config(None)

        dns = Dns('http://api.domain.com', 'domain.com', domain_config, service_config, port_mapper)
        dns.add_service("ownCloud", "http", "_http._tcp", 80, url="owncloud")

        service = dns.get_service("ownCloud")

        self.assertIsNotNone(service)
        self.assertEquals("ownCloud", service.name)
        self.assertEquals("_http._tcp", service.type)
        self.assertEquals(80, service.port)
        self.assertEquals("owncloud", service.url)

    def test_get_not_existing_service(self):
        service_config = get_service_config([])
        port_config = get_port_config([])
        upnpc = get_upnpc(external_ip='192.167.44.52')
        port_mapper = get_port_mapper(port_config, upnpc)

        domain_config = get_domain_config(None)

        dns = Dns('http://api.domain.com', 'domain.com', domain_config, service_config, port_mapper)

        service = dns.get_service("ownCloud")

        self.assertIsNone(service)
