#!/usr/bin/python
# -*- coding: iso-8859-1 -*-
###############################################################################
# Tool functions
#
import os.path
import fnmatch
import re
import shutil
from stat import S_IWRITE
import msvcrt
import tempfile
from zipfile import ZipFile, ZIP_DEFLATED

_debugLevel = 1 # 0:quiet, 1:verbose, 2:debug 

def printVerbose(msg):
    if _debugLevel>=1:
        print msg

def printDebug(msg):
    if _debugLevel>=2:
        print msg


###############################################################################
# Read SVN repository information from working copies.
#
def getSvnWCRevInfo(svnWCRevPathName, workingCopyPath,
                    verbose=False):
    """"""
    template = """
    revision: $WCREV$
    revisionDate: $WCDATE=%Y-%m-%d %H:%M:%S$
    buildDate: $WCNOW=%Y-%m-%d %H:%M:%S$
    mixedRevision: $WCMIXED?True:False$
    localModifications: $WCMODS?True:False$
    repositoryUrl: $WCURL$
    updateRevisionRange: $WCRANGE$
    """
    infoDict = {"wcPath": workingCopyPath }
    
    tmplFileHandle, tmplFilePath = tempfile.mkstemp(suffix=".tmpl")
    tmplFile = os.fdopen(tmplFileHandle, "w")
    resultFilePath = tmplFilePath + ".out"
    
    try:
        tmplFile.write(template)
        tmplFile.close()
        
        command = 'CALL "%s" "%s" "%s" "%s"' % (svnWCRevPathName, 
                                                workingCopyPath,
                                                tmplFilePath,
                                                resultFilePath)
        printDebug(command)
        print 
        errorLevel = os.system(command)
        print 
        if errorLevel > 0:
            print "%s returned ERRORLEVEL %i." % (svnWCRevPathName, errorLevel) 
            return None

        resultFile = open(resultFilePath, "r")
        for line in resultFile:
            if line.strip():
                key, val = line.split(":", 1)
                val = val.strip()
                if val == "True":
                    val = True
                elif val == "False":
                    val = False
                infoDict[key.strip()] = val
        resultFile.close()
    finally:
#        printDebug "Removing %s..." % tmplFilePath
        os.remove(tmplFilePath)
#        printDebug "Removing %s..." % resultFilePath
        os.remove(resultFilePath)

    if verbose:
        print("SVN info for %s:" % workingCopyPath)
        for k, v in infoDict.items():
            print("  %20s: %s" % (k, v))
    return infoDict


###############################################################################
# Replace macros in textfiles
#
def replaceSvnRevInfo(svnWCRevPathName, workingCopyPath, sourceFilePath,
                      versionTag, buildTag):
    """"""
    printVerbose("Replace SVN revision info")
    infoDict = getSvnWCRevInfo(svnWCRevPathName, workingCopyPath, verbose=False)
    if not infoDict:
        raise RuntimeError()
    
    tmpFileHandle, tmpFilePath = tempfile.mkstemp(suffix=".tmp")
    tmpFile = os.fdopen(tmpFileHandle, "w")
    
    nReplace = 0
    
    # If localModifications is set: append "(modified)"
    revisionSummary = "%s, %s" % (infoDict["revision"], infoDict["revisionDate"])
    if infoDict["localModifications"]:
        revisionSummary += ". Modified %s" % infoDict["buildDate"]

    # Add som build tags (not from SVN, but from make-script)
    infoDict["versionTag"] = versionTag
    infoDict["buildTag"] = buildTag
    
    macroList= [
                (re.compile(r"(.*)\$Rev:(.*?)\$(.*)"),
                 r"\1$Rev: %s$\3" % infoDict["revision"]),
                (re.compile(r"(.*)\$Date:(.*?)\$(.*)"),
                 r"\1$Date: %s$\3" % infoDict["revisionDate"]), 
                (re.compile(r"(.*)\$BuildDate:(.*?)\$(.*)"),
                 r"\1$BuildDate: %s$\3" % infoDict["buildDate"]), 
                (re.compile(r"(.*)\$Revision:(.*?)\$(.*)"),
                 r"\1$Revision: %s$\3" % revisionSummary), 
                 # Build info (not from SVN):
                (re.compile(r"(.*)\$Version:(.*?)\$(.*)"),
                 r"\1$Version: %s$\3" % infoDict["versionTag"]), 
                (re.compile(r"(.*)\$Build:(.*?)\$(.*)"),
                 r"\1$Build: %s$\3" % infoDict["buildTag"]), 
                ]

    try:
        sourceFile = open(sourceFilePath, "r")
        fileMatch = False
        iLine = 0
        for line in sourceFile:
            iLine += 1
            matched = False
            for macro in macroList:
                line, n = re.subn(macro[0], macro[1], line)
                nReplace += n
                if n:
                    matched = True
            if matched:
                if not fileMatch:
                    fileMatch = True
                    printVerbose("  %s:" % sourceFilePath)
                printVerbose("    #%04i: %s" % (iLine, line))
            tmpFile.write(line)
        sourceFile.close()
        tmpFile.close()
        
        backup = False
        if backup:
            bakFilePath = "%s.bak" % sourceFilePath
            if os.path.exists(bakFilePath):
                os.remove(bakFilePath)
            shutil.move(sourceFilePath, bakFilePath)
        else:
            os.remove(sourceFilePath)

        shutil.move(tmpFilePath, sourceFilePath)

    except Exception, e:
        print "replaceSvnRevInfo failed", e
        raise
#    finally:
#        printDebug "Removing %s..." % tmplFilePath
#        os.remove(tmplFilePath)

    return nReplace
    

def replaceJsLines(filePathList, macroDict={}, mode=None):
    """Replace marked lines in text files.

    TODO: allow $REPLACE(mode) for mode dependent replaces

    @example
    Given bar.js
        function foo() {
            alert('Hello developer'); // $REPLACE:    alert('Dear customer');
            alert('Development version'); // $REPLACE:    alert('Revision: $Rev:$');
        };
    Calling
        replaceJsLines("*.js", {'Rev': '1234'}):
    Yields bar.js
        function foo() {
            alert('Dear customer');
            alert('Revision: 1234');
        };
    """
    printVerbose("Replace JS lines with release version")
    
    if isinstance(filePathList, basestring):
        filePathList = [ filePathList ]
    
    tmpFileHandle, tmpFilePath = tempfile.mkstemp(suffix=".tmp")
    tmpFile = os.fdopen(tmpFileHandle, "w")
    
    replacedLines = replacedFiles = 0
    
    macroList= [
                (re.compile(r"(.*)\$REPLACE(.*?):(.*)"),
                 r"\3"), 
                ]
    
    for filePath in filePathList:
        try:
            sourceFile = open(filePath, "r")
            fileMatch = False
            iLine = 0
            for line in sourceFile:
                iLine += 1
                matched = False
                for macro in macroList:
                    line, n = re.subn(macro[0], macro[1], line)
                    replacedLines += n
                    if n:
                        matched = True
                if matched:
                    if not fileMatch:
                        replacedFiles += 1
                        fileMatch = True
                        printVerbose("  %s:" % filePath)
                    printVerbose("    #%04i: %s" % (iLine, line))
                tmpFile.write(line)
            sourceFile.close()
            tmpFile.close()
            
            backup = False
            #backup = True
            if backup:
                bakFilePath = "%s.bak" % filePath
                if os.path.exists(bakFilePath):
                    os.remove(bakFilePath)
                shutil.move(filePath, bakFilePath)
            else:
                os.remove(filePath)
    
            shutil.move(tmpFilePath, filePath)
    
        except Exception, e:
            print "replaceJsLines failed", e
            raise
    #    finally:
    #        printDebug "Removing %s..." % tmplFilePath
    #        os.remove(tmplFilePath)
    printDebug("Replaced %s lines in %s files." % (replacedLines, replacedFiles))

    return replacedFiles


###############################################################################
# File tools
#
_pattern_type = type(re.compile("", 0))


def fileIterator(rootPath, patternOrRex, recursive=True):
    """Return a list of absolute file paths for matching files.

    TODO: this is breadth first; should be an easier version with depth first
    fi = fileIterator(r"c:\prj\cpp", r".*\.cpp$")
    for f in fi:
        print f
    """
    if not os.path.isdir(rootPath):
        raise TypeError()
    assert recursive
    if isinstance(patternOrRex, _pattern_type):
        rex = patternOrRex
    else:
        rex = re.compile(patternOrRex)
    dirs = [rootPath]
    # while we has dirs to scan
    while len(dirs) :
        nextDirs = []
        for parent in dirs :
            # scan each dir
            for f in os.listdir( parent ) :
                # if there is a dir, then save for next ittr
                # if it  is a file then yield it (we'll return later)
                ff = os.path.join( parent, f )
                if os.path.isdir( ff ) :
                    nextDirs.append( ff )
                elif rex.match(f):
                    yield ff
        # once we've done all the current dirs then
        # we set up the next itter as the child dirs 
        # from the current itter.
        dirs = nextDirs



#def breathFirstfileIterator(rootPath, patternOrRex, recursive=True):
#    """Return a list of absolute file paths for matching files.
#    """
#    if not os.path.isdir(rootPath):
#        raise TypeError()
#    if isinstance(patternOrRex, _pattern_type):
#        rex = patternOrRex
#    else:
#        rex = re.compile(patternOrRex)
#    dirs = [rootPath]
#    # while we has dirs to scan
#    while len(dirs) :
#        nextDirs = []
#        for parent in dirs :
#            # scan each dir
#            for f in os.listdir( parent ) :
#                # if there is a dir, then save for next ittr
#                # if it  is a file then yield it (we'll return later)
#                ff = os.path.join( parent, f )
#                if os.path.isdir( ff ) :
#                    nextDirs.append( ff )
#                else :
#                    yield ff
#        # once we've done all the current dirs then
#        # we set up the next itter as the child dirs 
#        # from the current itter.
#        dirs = nextDirs



#def fileIterator(rootPath, patternOrRex, recursive=True, includeFiles=True, includeDirs=False):
#    """Return a list of absolute file paths for matching files."""
#    if not os.path.isdir(rootPath):
#        raise TypeError()
#    if isinstance(patternOrRex, _pattern_type):
#        rex = patternOrRex
#    else:
#        rex = re.compile(p)
#    pathList = []
#    
#    for root, dirs, files in os.walk(rootPath):
#        if includeFiles:
#            for name in files:
#                path = os.path.join(root, name)
##                relPath = path[len(folderPath)+1:]
#                pathList.append(path)
#        if includeDirs or recursive:
#            for name in dirs:
#                path = os.path.join(root, name)
#                if includeDirs:
#                    pathList.append(path)
#                pathList.append()
#            
#    for name in os.listdir(rootPath):
#        fullpath=os.path.join(rootPath, name)
#
#        if dirMode:
#            if os.path.isdir(fullpath):
#                if rex.match(name):
#                    def onError(function, path, excinfo):
#                        if function is os.remove:
#                            try:
#                                # os.remove cannot remove readonly files on windows
#                                os.chmod(path, S_IWRITE)
#                                os.remove(path)
#                                return
#                            except OSError, _e:
#                                pass
#                        if printMode >= 1:
#                            print "Error removing directory: path=%s, excinfo=%s" % (path, excinfo)
#                    shutil.rmtree(fullpath, False, onError)
#                    if printMode >= 2:
#                        print "Removed directory '%s'" % fullpath
#                else:
#                    _removePattern(fullpath, rex, dirMode)
#        else:
#            if os.path.isdir(fullpath):
#                _removePattern(fullpath, rex, dirMode)
#            elif os.path.isfile(fullpath) and rex.match(name):
#                try:
#                    # os.remove cannot remove readonly files on windows
#                    os.chmod(fullpath, S_IWRITE)
#                    os.remove(fullpath)
#                    if printMode >= 2:
#                        print "Removed file '%s'" % fullpath
#                except OSError, (_errno, strerror):
#                    if printMode >= 1:
#                        print "Error removing '%s': %s" % (fullpath, strerror)
#    return res


def _removePattern(rootPath, patternListOrRex, dirMode=False, printMode=1):
    """Remove matching files in the directory tree.
    
    patternListOrRex may of type 
      - compiled regular expression
      - string starting with 're:' (will be compiled to an regular expression)
      - other string (uses glob syntax)
    or a list thereof.
    """
    if not os.path.isdir(rootPath):
        return
    # Handle list items
    if type(patternListOrRex) == type([]):
        for p in patternListOrRex:
            _removePattern(rootPath, p, dirMode)
        return

    # Convert string to regular expression 
    rex = patternListOrRex
    if isinstance(rex, _pattern_type):
        pass
    elif isinstance(rex, basestring):
        if rex.startswith("re:"):
            rex = rex.lstrip("re:")
        else:
            # translate glob syntax to regular expression
            rex = fnmatch.translate(rex)
        if printMode >= 2:
            print "remove pattern: '%s'" % rex
        rex = re.compile(rex)
    assert type(rex) == _pattern_type
#    print "_removePattern(%s, %s, dirMode=%s):" % (rootPath, rex, dirMode)
    for name in os.listdir(rootPath):
        fullpath=os.path.join(rootPath, name)

        if dirMode:
            if os.path.isdir(fullpath):
                if rex.match(name):
                    def onError(function, path, excinfo):
                        if function is os.remove:
                            try:
                                # os.remove cannot remove readonly files on windows
                                os.chmod(path, S_IWRITE)
                                os.remove(path)
                                return
                            except OSError, _e:
                                pass
                        if printMode >= 1:
                            print "Error removing directory: path=%s, excinfo=%s" % (path, excinfo)
                    shutil.rmtree(fullpath, False, onError)
                    if printMode >= 1:
                        print "Removed directory '%s'" % fullpath
                else:
                    _removePattern(fullpath, rex, dirMode)
        else:
            if os.path.isdir(fullpath):
                _removePattern(fullpath, rex, dirMode)
            elif os.path.isfile(fullpath) and rex.match(name):
                try:
                    # os.remove cannot remove readonly files on windows
                    os.chmod(fullpath, S_IWRITE)
                    os.remove(fullpath)
                    if printMode >= 1:
                        print "Removed file '%s'" % fullpath
                except OSError, (_errno, strerror):
                    if printMode >= 1:
                        print "Error removing '%s': %s" % (fullpath, strerror)


def removeFilePattern(rootPath, patternList):
    """Remove all matching files in or under the rootPath directory.

    A list of glob patterns or glob pattern strings
    (Strings starting with 're:' use regular expression syntax).
    Examples:
        "*.bak"  Removes all *.bak files
        "re:.*\.bak"     Removes all *.bak and *.bake files
        "re:.*\.bak$"    Removes all *.bak files
    """
    _removePattern(rootPath, patternList, dirMode=False)


def removeDirPattern(rootPath, patternList):
    """Remove all matching subdirectories.

    A list of glob patterns or glob pattern strings
    (Strings starting with 're:' use regular expression syntax).
    Examples:
        "re:[\_|.]svn$"  Removes all '.svn' and '_svn' directories
    """
    _removePattern(rootPath, patternList, dirMode=True)


def removeDir(rootPath):
    """Like os.rmtree, but doesn't fail on readonly files."""
    def onError(function, path, excinfo):
        if function is os.remove:
            try:
                # os.remove cannot remove readonly files on windows
                os.chmod(path, S_IWRITE)
                os.remove(path)
                return
            except OSError, _e:
                pass
            print "Error removing directory: path=%s, excinfo=%s" % (path, excinfo)
    shutil.rmtree(rootPath, False, onError)


###############################################################################
# Get buildinfo from subversion
#
#revisionInfoDict = getSvnWCRevInfo(svnWcRevPath, srcPath)

#getSvnWCRevInfo(svnWcRevPath, os.path.join(srcPath, "doc", "dynatree-doc.html"))
#getSvnWCRevInfo(svnWcRevPath, os.path.join(srcPath, "doc", "howto.js"))

#if revisionInfoDict["localModifications"]:
#    raise RuntimeWarning("%s has local modifications. You should commit first.")



###############################################################################
# Copy project folder to temp dir and remove SVN stuff
#
def exportWorkingCopy(wcPath, targetPath, excludeList=[]):
    """Copy project folder to temp dir and remove SVN stuff."""
    removeDir(targetPath)
    shutil.copytree(wcPath, targetPath)
    
    removeFilePattern(targetPath, ".project")      # Eclipse project settings
    removeFilePattern(targetPath, ".pydevproject") # PyDev project settings
    removeDirPattern(targetPath, "re:[\.|_]svn")   # '.svn' and '_svn' folders
    removeFilePattern(targetPath, excludeList)
    


###############################################################################
# Patch $Rev:$ macros
#
#replaceSvnRevInfo(svnWcRevPath, 
#                  srcPath, 
#                  os.path.join(tempPath, "src", "jquery.dynatree.js"))
#replaceSvnRevInfo(svnWcRevPath, 
#                  os.path.join(srcPath, "doc", "dynatree-doc.html"), 
#                  os.path.join(tempPath, "doc", "dynatree-doc.html"))


###############################################################################
# Create minified version
#
def minify(srcPath, targetPath, disclaimer=None):
    import jsmin
    
    jsm = jsmin.JavascriptMinify()
    ins = open(srcPath, "r")
    outs = open(targetPath, "w")
    if disclaimer:
        print >> outs, disclaimer
    jsm.minify(ins, outs)
    
    ins.close()
    outs.close()


###############################################################################
# Create a ZIP file
#
def createZip(folderPath, zipPath):
    zip = ZipFile(zipPath, "w", ZIP_DEFLATED)
    for root, _dirs, files in os.walk(folderPath):
        for name in files:
            path = os.path.join(root, name)
            relPath = path[len(folderPath)+1:]
            zip.write(path, relPath)
    
    zip.close()


def ensure_dir(filePath):
    """Create all dirs neccessary to store a file."""
    d = os.path.dirname(filePath)
    if not os.path.exists(d):
        os.makedirs(d)


def unzip(file, dir):
    zip = ZipFile(file)
    for name in zip.namelist():
        if name.endswith("/"):
            os.mkdir(os.path.join(dir, name)) # unzip empty folder
        else:
            filePath = os.path.join(dir, name)
            ensure_dir(filePath)
            outfile = open(filePath, "wb")
            outfile.write(zip.read(name))
            outfile.close()



###############################################################################
# Create a tarball
#
def createTar(folderPath, tarPath):
    import tarfile

    cwd = os.getcwd()

    # Change cwd, so gz contains relative paths:
    os.chdir(os.path.dirname(tarPath))
    tar = tarfile.open(os.path.basename(tarPath), "w:gz")

    # Change cwd, so tar contains relative paths:
    os.chdir(folderPath)
    tar.add(".", recursive=True)

    tar.close()
        
    os.chdir(cwd)


###############################################################################
# 
#
def _hexString(s):
    return "[%s]" % ", ".join([ "x%02X" % ord(c) for c in s ])


def fixTabs(srcPath, targetPath, tabSize=4, tabbify=False, 
            originalTabSize=None, lineSeparator=None, backup=True):
    """Unify leading spaces and tabs and strip trailing whitespace."""
    printVerbose("fixTabs %s" % srcPath)
    srcPath = os.path.abspath(srcPath)
    originalTabSize = originalTabSize or tabSize
#    lineSeparator = lineSeparator or os.linesep
    replaceMode = (targetPath.lower() == srcPath.lower())
    if replaceMode or not targetPath:
        targetPath = srcPath + ".tabbed" 
    # Open with 'U', so we get file.newline 
    fin = open(srcPath, "Ur")
    # Open with 'b', so we can have our own line endings
    fout = open(targetPath, "wb")
    lines = []
    lineNo = 0
    changedLines = 0
    for line in fin:
        lineNo += 1
        line = line.rstrip(" \t"+chr(0x0A) +chr(0x0D))
        s = ""
        indent = 0
        chars = 0
        for c in line:
            if c in (" ", " "): # Space, shift-space
                chars += 1
                indent += 1
            elif c == "\t":
                chars += 1
                indent = tabSize * ((indent + tabSize) / tabSize)
            else:
                break

        if tabbify:
            s = "\t" * (indent / tabSize) + " " * (indent % tabSize) + line[chars:]
        else:
            s = " " * indent + line[chars:]

#        print >>fout, s
        #fout.write(s + "*" + lineSeparator)
        lines.append(s)
        if s != line:
            changedLines += 1
            printDebug("    #%04i: %s" % (lineNo, line.replace(" ", ".").replace("\t", "<tab>"))) 
            printDebug("         : %s" % s.replace(" ", ".").replace("\t", "<tab>")) 
    
    try:
        inputSeparators = fin.newlines
    except Exception, _e:  
        inputSeparators = []
    fin.close()
    if not lineSeparator and inputSeparators and len(inputSeparators) == 1:
        lineSeparator = inputSeparators[0]
        printVerbose("    Setting line separator to input format (%s)" % (_hexString(lineSeparator)))
    if not lineSeparator:
        lineSeparator = os.linesep
    fout.writelines(lineSeparator.join(lines))
    fout.close()
    
    srcSize = os.path.getsize(srcPath)
    targetSize = os.path.getsize(targetPath)

    if replaceMode:
        if backup:
            bakFilePath = "%s.bak" % srcPath
            if os.path.exists(bakFilePath):
                os.remove(bakFilePath)
            shutil.move(srcPath, bakFilePath)
        else:
            os.remove(srcPath)
        shutil.move(targetPath, srcPath)
    printVerbose("    Changed %s lines (size %s -> %s bytes)" % (changedLines, srcSize, targetSize))


###############################################################################
# Wait for user input
#

def runCommand(cmd, cmdLine=None, useStart=False):
    if useStart:
        command = 'START "%s"' % cmd
    else:
        command = '"%s"' % cmd
    if cmdLine:
        command += " " + cmdLine
    print command
    errorLevel = os.system(command)
    if errorLevel > 0:
        print "%s returned ERRORLEVEL %i." % (command, errorLevel) 


#def startCommand(cmd, cmdLine=None):
#    command = 'START "%s"' % cmd
#    if cmdLine:
#        command += " " + cmdLine
#    print command
#    errorLevel = os.system(command)
#    if errorLevel > 0:
#        print "%s returned ERRORLEVEL %i." % (command, errorLevel) 


def waitForKey(message):
        print message + " ",
        return msvcrt.getch()


def promptYesNo(message, enterKey="", escapeKey=""):
    while True:
        if enterKey.lower() in ("j", "y"):
            choice = "[Y, n]"
        elif enterKey.lower() == "n":
            choice = "[y, N]"
        else:
            choice = "[y, n]"
        print message + " " + choice + " ",
        
        c = msvcrt.getch()
        
        if ord(c) == 3: # break
            raise "Break"
        elif ord(c) == 13: # enter
            c = enterKey
        elif ord(c) == 27: # escape
            c = escapeKey
        
        if c.lower() in ["y", "j", "z"]:
            print c
            return True
        elif c.lower() in ["n"]:
            print c
            return False


if __name__ == "__main__":
    #createTar(r"c:\prj\cpp", r"c:\temp\tartest.tgz")
    raise RuntimeError("Don't call directly")
