#! /usr/bin/env python
# -*- coding: utf-8 -*-

"""
    mass rename - massive renaming of same type files in a directory

    python2 script

    Copyright (c) 2011-2014 Raphaël SEBAN <motus@laposte.net>

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation, either version 3 of
    the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.

    If not, see http://www.gnu.org/licenses/
"""

# lib imports

from __future__ import print_function
import re
import os
import os.path as OP
import glob
import math
import locale
import argparse

# func defs

def ufname (fname):

    return unicode(fname, locale.getdefaultlocale()[1], "replace")

# end def

def rename_tmp2file(dirlist, tempfile_ext, verbose=False):

    # best before looping

    _tail = -len(tempfile_ext)

    for _file in dirlist:

        _file = ufname(_file)

        _dest = _file[:_tail]

        if OP.exists(_dest):

            raise IOError(

                u"file already exists: '{uri}'".format(uri=_dest)
            )

            # fatal error

            exit(1)

        else:

            if verbose:

                print(

                    u"[TEMP] restoring '{0}' to '{1}'"

                    .format(_file , _dest)
                )

            # end if

            os.rename(_file, _dest)

        # end if

    # end for

# end def


# sys.argv[] argument parsing

parser = argparse.ArgumentParser(

    description="This utility tries to rename files massively."
)

parser.add_argument(

    "--start",

    nargs=1,

    default=[1],

    type=int,

    help="starting point for counter index "

        "(default: counter starts at 1). "

        "Example: %(prog)s --start 100"
)

parser.add_argument(

    "--file-ext",

    nargs=1,

    default=["jpg"],

    type=str,

    help="case-insensitive file extension to process "

        "(default: jpg). Example: %(prog)s --file-ext txt"
)

parser.add_argument(

    "radix",

    nargs="*",

    default=["file"],

    type=str,

    help="filename radix without extension nor index count "

        "(default: 'file' ). "

        "Example: %(prog)s mY/.bEst+ hoLiDayS: ON  mars --> "

        "file radix will be 'mY-bEst-hoLiDayS-ON-mars'"
)


# parsing sys.argv

_args = parser.parse_args()


# var inits

_count = max(1, int(_args.start[0]))

_file_radix = re.sub(

    r"(?u)\W+", r"-", ufname("-".join(_args.radix))

).strip("-")

_file_ext = re.sub(r"\W+", r"", _args.file_ext[0]).lower()

_tempfile_ext = "~"


# default values

if not _file_radix:

    _file_radix = u"file"

# end if

if not _file_ext:

    _file_ext = "jpg"

# end if


# file mask inits

_glob_all = "*." + _file_ext

_glob_tmp_all = _glob_all + _tempfile_ext


# interrupted previous process?

_dirlist = glob.glob(_glob_tmp_all)

if _dirlist:

    print("\n*** Resuming interrupted process ***\n")

    rename_tmp2file(_dirlist, _tempfile_ext, verbose=True)

# end if


_dirlist = glob.glob(_glob_all) + glob.glob(_glob_all.upper())

if not _dirlist:

    print("\n[ERROR] no file to process. Exiting.")

    exit(1)

# end if


print("\n*** Starting mass rename process ***")

print(u"\n[INFO] file radix will be '{0}'".format(_file_radix), "\n")

_digits = 1 + int(math.log10(_count + len(_dirlist)))

_new_file_mask = (

    u"{{radix}}-{{index:0{digits}d}}.{{ext}}"

    .format(digits=str(_digits))
)

_dirlist.sort()

for _file in _dirlist:

    #reset to unicode filename

    _file = ufname(_file)

    if OP.isfile(_file):

        _new = _new_file_mask.format(

            radix=_file_radix, index=_count, ext=_file_ext
        )

        print(u"[INFO] renaming '{0}' to '{1}'".format(_file, _new))

        os.rename(_file, _new + _tempfile_ext)

        _count += 1

    elif OP.isdir(_file):

        print(u"[WARNING] found a directory named '{0}'".format(_file))

    # end if

# end for

_dirlist = glob.glob(_glob_tmp_all)

rename_tmp2file(_dirlist, _tempfile_ext)

_dirlist = glob.glob(_glob_all)

_count = len(_dirlist)

if _count <= 30:

    _cmd = {"nt": "dir"}.get(

        os.name.lower(), "ls -ldh1 --color --group-directories-first"
    )

    print("\n*** Listing results: ***")

    print("\ncurrent directory:", os.getcwd(), "\n")

    os.system(_cmd + " " + _glob_all)

    print("\ntotal count (files and dirs):", _count)

# end if

print("\n*** Program end ***\n")

exit(0)
