import re
import copy
import valideer

from . import validators
from .reserved import *


class Script(object):
    def js(self, scope="_"):
        return ""

    def json(self):
        return None

class Event(Script):
    def __init__(self, parser, lineno, args=None, kwargs=None, suite=None, **extra_kargs):
        self.parser = parser
        assert lineno > 0
        self.lineno = str(lineno)
        self.kwargs = kwargs or {}
        self.suite = suite
        self.extra = extra_kargs
        self.siblings = []

        if args and hasattr(self, 'ARGS'):
            if self.ARGS is True:
                for i, a in enumerate(args):
                    self.kwargs[str(i)] = a
            else:
                arg_list = copy.copy(self.ARGS)
                while len(args) > 0:
                    assert len(arg_list) > 0, "Invlid number of inline arguments provided"
                    key = arg_list.pop(0)
                    assert key not in self.kwargs, "Argument already specified in kwargs"
                    self.kwargs[key] = args.pop(0)
        self.validate()

        # add futures
        if self.kwargs:
            parser.futures["_"+str(lineno)] = self.kwargs

        # add the outputs
        if hasattr(self, 'OUTPUTS'):
            parser.output_variables[str(lineno)] = copy.copy(self.OUTPUTS)

    def validate(self):
        if hasattr(self, 'PROPERTIES'):
            if isinstance(self.PROPERTIES, dict):
                # validate its IN variables
                valideer.parse(self.PROPERTIES).validate(self.kwargs)
        elif len(self.kwargs) > 0:
            raise ValueError("Invlid arguments provided.")

    def add(self, *sib):
        for s in sib:
            self.siblings.append(s)
        return self

    def json(self):
        return dict([(key, value.json() if hasattr(value, 'json') else value) for key, value in self.extra.items() if value is not None])

class Program(object):
    def __init__(self, parser, placeholders, task):
        self.parser = parser
        self.placeholders = placeholders
        self.task = task

    def js(self, scope="_"):
        return self.task.js(scope)

    def _json_next(self, dct, event, parent=None):
        if hasattr(event, 'lineno'):
            dct[str(event.lineno)] = dict(event=event.__class__.__name__.lower(),
                                          parent=parent.lineno if parent else None)
            dct[str(event.lineno)].update(event.json())
            if event.suite:
                for e in event.suite:
                    self._json_next(dct, e, event)
        elif isinstance(event, StmtList):
            for _event in event:
                self._json_next(dct, _event, parent)    

    def _dump_json(self, dct):
        return dict([(name, variable.json() if hasattr(variable, 'json') else variable)\
                     for name, variable in dct.items()])

    def json(self):
        dct = {}
        self._json_next(dct, self.task)
        return dict(futures=dict([(str(line), self._dump_json(variables)) for line, variables in self.parser.futures.items()]),
                    outputs=self.parser.outputs,
                    placeholders=self.placeholders.json() if self.placeholders else None,
                    events=dct)

class Placeholders(Script):
    def __init__(self, placeholder):
        self.placeholders = [placeholder]

    def add(self, placeholder):
        self.placeholders.append(placeholder)
        return self

    def json(self):
        return [dict(path=p.path, value=p.value[2:-2]) for p in self.placeholders]

class Placeholder(Script):
    def __init__(self, path, value):
        _value = re.sub(r'\[.*', '', value[2:-2])
        assert _value in PLACEHOLDER_RESERVED_NAMES, "Placeholder value %s is a reserved placeholder value." % _value
        self.path = path
        self.value = value

# ------------------------
# Events
# ------------------------
class Run(Event):
    pass

class Checkout(Event):
    OUTPUTS = {
      "customer": {} # Customer <-- use an uninitiated class here
    }

class Login(Event):
    pass

class Logout(Event):
    pass

class Signup(Event):
    pass

class Addtocart(Event):
    pass

class Storeopens(Event):
    pass

class Storecloses(Event):
    pass

class Subscribe(Event):
    pass

class Unsubscribe(Event):
    pass

class On(Event):
    ARGS = ["date"]
    PROPERTIES = {
      "+date": "timestring"
    }

class Tweet(Event):
    ARGS = ["handler", "status"]
    PROPERTIES = {
        "+handler": "twitter_handler",
        "+status": "string"
    }
    OUTPUTS = {
      "id": "int",
      "text": "text",
      "ok": "bool",
      "user": {
        "name": "text",
        "friends_count": "int",
        "followers_count": "int",
        "protected": "bool"
      }
    }

class Dialog(Event):
    pass

class Reward(Event):
    ARGS = ["user", "points"]
    PROPERTIES = {
      "+user": "user",
      "+points": "number"
    }
    OUTPUT = {
      "customer": {},
      "points": "int"
    }

class Periodically(Event):
    ARGS = ["date"]
    PROPERTIES = {
      "date": "timestring"
    }

class Sms(Event):
    ARGS = ["user", "message"]
    PROPERTIES = {
        "+user": "user",
        "+message": "string",
        "time": "timestring"
    }

class Email(Event):
    ARGS = ["user", "subject", "body"]
    PROPERTIES = {
        "+user": "user",
        "+subject": "string",
        "+body": "html",
        "subscription": "string"
    }

# ------------------------
# System Events
# ------------------------
class WaitUntil(Event):
    pass

class Wait(Event):
    pass

class Log(Event):
    ARGS = True
    PROPERTIES = True

class Pass(Event):
    def __init__(self):
        pass

class Restart(Event):
    pass

class Quit(Event):
    def __init__(self):
        pass

class Foreach(Event):
    pass

# ------------------------
# Statements
# ------------------------
class StmtList(Script):
    def __init__(self, stmt):
        self.siblings = [stmt]

    def add(self, *stmts):
        [self.siblings.append(stmt) for stmt in stmts]
        return self

    def __iter__(self):
        return iter(self.siblings)

class If(Event):
    pass

class Else(Event):
    pass

class Elseif(Event):
    pass

# ------------------------
# Variables Manipulation
# ------------------------
class Set(Event):
    def __init__(self, *a, **k):
        Event.__init__(self, *a, **k)
        assert re.match('^[a-zA-Z_][a-zA-Z]', self.extra["varpath"]), "Invalid variable name. Must start with /^[a-zA-Z_][a-zA-Z]/"

class Unset(Event):
    def __init__(self, *a, **k):
        Event.__init__(self, *a, **k)
        assert re.match('^[a-zA-Z_][a-zA-Z]', self.extra["varpath"]), "Invalid variable name. Must start with /^[a-zA-Z_][a-zA-Z]/"

class Push(Script):
    def __init__(self, *a, **k):
        Event.__init__(self, *a, **k)
        assert self.extra["varpath"].endswith('[]'), "Can only push into :arrays"
        assert re.match(r'^[a-zA-Z_][a-zA-Z]', self.extra["varpath"]), "Invalid variable name. Must start with /^[a-zA-Z_][a-zA-Z]/"

class With(Event):
    pass

# ------------------------
# Juiciness
# ------------------------
class Args(Script):
    def __init__(self, arg):
        self.args = [arg]

    def add(self, arg):
        self.args.append(arg)
        return self

class Kwargs(Script):
    def __init__(self, kwarg):
        self.kwargs = {}
        for k, v in kwarg.items():
            self.kwargs[k[2:]] = v

    def add(self, kwarg):
        for k, v in kwarg.items():
            if k[2:] in self.kwargs:
                raise IndexError("Kwarg %s specified twice" % k)
            else:
                self.kwargs[k[2:]] = v

        return self

# ------------------------
# Variables
# ------------------------
class Varpath(Script):
    def __init__(self, varpath, tags=[], offset=None, where=None, sortby=None, randunique=[], agg=None, time=None):
        self.varpath = '.'.join(varpath) if isinstance(varpath, list) else varpath
        self.tags = []
        self.offset = offset
        self.where = where
        self.sortby = sortby
        self.randunique = randunique
        self.agg = agg
        self.time = time

        # need to define what this element is, or will be once produced
        self.types = set(['twitter_handler'])

        if where:
            # need to check the where expressions: all varpaths exist in `varpaths`
            pass

        if sortby:
            # need to check the `sortby[0]` exists in `varpath`
            pass

    def json(self):
        dct = dict(path=self.varpath,
                   tags=[t[2:] for t in set(self.tags)] if self.tags else None,
                   offset=self.offset,
                   where=self.where.json() if self.where else None,
                   sortby=self.sortby,
                   random=True if 'random' in self.randunique else None,
                   unique=True if 'unique' in self.randunique else None,
                   agg=self.agg,
                   time=self.time)
        return dict([(name, value) for name, value in dct.items() if value is not None])

class String(Script):
    def __init__(self, data):
        self.datas = [data]

    def add(self, data):
        self.datas.append(data)
        return self

    def __repr__(self):
        return "<String %s>" % ",".join(self.types)

    @property
    def types(self):
        types = ["string"]
        for st in self.datas:
            if isinstance(st, Varpath):
                types.append('vartext')
                break
        return set(types)

    def json(self):
        is_complex = False
        for st in self.datas:
            if isinstance(st, Varpath):
                is_complex = True
                break

        if is_complex:
            string = []
            values = {}
            for st in self.datas:
                if isinstance(st, Varpath):
                    values[str(len(values))] = st.json()
                    string.append("{%d}" % (len(values)-1))
                else:
                    string.append(st)
            return dict(string=''.join(string).strip(), values=values)

        else:
            return " ".join([d.strip() for d in self.datas]).strip()

class Number(Script):
    def __init__(self, number):
        self.number = number

    def json(self):
        return self.number

# ------------------------
# Expression
# ------------------------
class Expression(Script):
    def __init__(self, expression):
        self.expressions = [("", expression)]

    @property
    def types(self):
        return set(['number'])

    def add(self, method, expression):
        self.expressions.append( (method, expression) )
        return self

    def json(self, evals=None, values=None):
        if evals == None:
            if len(self.expressions) == 1:
                return self.expressions[0][1].json()
            evals = []
            values = {}
        for mixin, expression in self.expressions:
            evals.append(mixin)
            if isinstance(expression, Expression):
                _d = expression.json(evals, values)
                evals = [_d['expression']]
                values = _d['values']

            elif isinstance(expression, Number):
                d = str(float(expression.number))
                evals.append(d)

            elif hasattr(expression, 'json'):
                d = expression.json()
                if isinstance(d, dict):
                    i = (max(values.keys()) + 1) if values else 0
                    values[i] = d
                    evals.append("{%d}" % (i))
                else:
                    if isinstance(expression, String):
                        d = "'%s'" % d
                    evals.append(d)
            else:
                evals.append(expression)

        return dict(expression=" ".join([e for e in evals if e != ""]),
                    values=values)

class Method(Script):
    def __init__(self, varpath, method, other):
        self.varpath = varpath
        self.method = method
        self.other = other

    def json(self):
        return dict(method=self.method, left=self.varpath.json(), right=self.other.json())

class Regexp(Script):
    def __init__(self, regexp):
        self.regexp = regexp

    def json(self):
        return dict(regex=self.regexp.pattern)
