/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.hdf5.h5ar;

import ch.systemsx.cisd.base.exceptions.IErrorStrategy;
import ch.systemsx.cisd.base.exceptions.IOExceptionUnchecked;
import ch.systemsx.cisd.base.io.IOutputStream;
import ch.systemsx.cisd.base.unix.FileLinkType;
import ch.systemsx.cisd.hdf5.HDF5GenericStorageFeatures;
import ch.systemsx.cisd.hdf5.HDF5OpaqueType;
import ch.systemsx.cisd.hdf5.IHDF5Writer;
import ch.systemsx.cisd.hdf5.IHDF5WriterConfigurator;
import ch.systemsx.cisd.hdf5.h5ar.ArchivingException;
import ch.systemsx.cisd.hdf5.h5ar.ArchivingStrategy;
import ch.systemsx.cisd.hdf5.h5ar.DirectoryIndex;
import ch.systemsx.cisd.hdf5.h5ar.DirectoryIndexProvider;
import ch.systemsx.cisd.hdf5.h5ar.IPathVisitor;
import ch.systemsx.cisd.hdf5.h5ar.LinkRecord;
import ch.systemsx.cisd.hdf5.h5ar.Utils;
import ch.systemsx.cisd.hdf5.io.HDF5IOAdapterFactory;
import java.io.File;
import java.io.FileInputStream;
import java.io.Flushable;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.List;
import java.util.zip.CRC32;
import ncsa.hdf.hdf5lib.exceptions.HDF5Exception;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;

class HDF5ArchiveUpdater {
    private static final String OPAQUE_TAG_FILE = "FILE";
    private static final int SIZEHINT_FACTOR = 5;
    private static final int MIN_GROUP_MEMBER_COUNT_TO_COMPUTE_SIZEHINT = 100;
    private static final int SMALL_DATASET_LIMIT = 4096;
    private final IHDF5Writer hdf5Writer;
    private final DirectoryIndexProvider indexProvider;
    private final IErrorStrategy errorStrategy;
    private final byte[] buffer;

    public HDF5ArchiveUpdater(IHDF5Writer hdf5Writer, DirectoryIndexProvider indexProvider, byte[] buffer) {
        this.hdf5Writer = hdf5Writer;
        this.indexProvider = indexProvider;
        this.errorStrategy = indexProvider.getErrorStrategy();
        this.buffer = buffer;
    }

    public HDF5ArchiveUpdater archive(File path, ArchivingStrategy strategy, int chunkSize, boolean keepNameFromPath, IPathVisitor pathVisitorOrNull) {
        File absolutePath = path.getAbsoluteFile();
        return this.archive(keepNameFromPath ? absolutePath.getParentFile() : absolutePath, absolutePath, strategy, chunkSize, pathVisitorOrNull);
    }

    public IOutputStream archiveFile(String directory, LinkRecord link, boolean compress, int chunkSize) {
        if (link.getLinkType() != FileLinkType.REGULAR_FILE) {
            this.errorStrategy.dealWithError(new ArchivingException("A regular file is expected here."));
        }
        return new H5ARIOutputStream(directory, link, chunkSize, compress);
    }

    public HDF5ArchiveUpdater archive(String directory, LinkRecord link, InputStream inputOrNull, boolean compress, int chunkSize, IPathVisitor pathVisitorOrNull) {
        boolean ok = true;
        String normalizedDir = Utils.normalizePath(directory);
        String hdf5ObjectPath = "/".equals(normalizedDir) ? String.valueOf(normalizedDir) + link.getLinkName() : String.valueOf(normalizedDir) + "/" + link.getLinkName();
        boolean groupExists = this.hdf5Writer.isGroup(normalizedDir);
        if (link.getLinkType() == FileLinkType.DIRECTORY) {
            if (inputOrNull == null) {
                ok = this.archiveDirectory(normalizedDir, link, pathVisitorOrNull);
            } else {
                this.errorStrategy.dealWithError(new ArchivingException("Cannot take InputStream when archiving a directory."));
            }
        } else if (link.getLinkType() == FileLinkType.SYMLINK) {
            if (inputOrNull == null) {
                ok = this.archiveSymLink(normalizedDir, link, pathVisitorOrNull);
            } else {
                this.errorStrategy.dealWithError(new ArchivingException("Cannot take InputStream when archiving a symlink."));
            }
        } else if (link.getLinkType() == FileLinkType.REGULAR_FILE) {
            if (inputOrNull != null) {
                HDF5GenericStorageFeatures compression = compress ? HDF5GenericStorageFeatures.GENERIC_DEFLATE : HDF5GenericStorageFeatures.GENERIC_NO_COMPRESSION;
                try {
                    DataSetInfo info = this.copyToHDF5(inputOrNull, hdf5ObjectPath, compression, chunkSize);
                    link.setCrc32(info.crc32);
                    link.setSize(info.size);
                    if (pathVisitorOrNull != null) {
                        pathVisitorOrNull.visit(hdf5ObjectPath);
                    }
                }
                catch (IOException ex) {
                    ok = false;
                    this.errorStrategy.dealWithError(new ArchivingException(hdf5ObjectPath, ex));
                }
                catch (HDF5Exception ex) {
                    ok = false;
                    this.errorStrategy.dealWithError(new ArchivingException(hdf5ObjectPath, ex));
                }
            } else {
                this.errorStrategy.dealWithError(new ArchivingException("Need to have InputStream when archiving a regular file."));
            }
        } else {
            this.errorStrategy.dealWithError(new ArchivingException("Don't know how to archive file link type " + link.getLinkType()));
            ok = false;
        }
        if (ok) {
            this.updateIndicesOnThePath(hdf5ObjectPath, link, groupExists);
        }
        return this;
    }

    public HDF5ArchiveUpdater archive(File root, File path, ArchivingStrategy strategy, int chunkSize, IPathVisitor pathVisitorOrNull) {
        boolean ok;
        File absolutePath;
        File absoluteRoot = root.getAbsoluteFile();
        String hdf5ObjectPath = HDF5ArchiveUpdater.getRelativePath(absoluteRoot, absolutePath = path.getAbsoluteFile());
        String hdf5GroupPath = FilenameUtils.getFullPathNoEndSeparator(hdf5ObjectPath);
        boolean groupExists = hdf5GroupPath.length() == 0 ? true : this.hdf5Writer.isGroup(hdf5GroupPath);
        int crc32 = 0;
        LinkRecord linkOrNull = LinkRecord.tryCreate(absolutePath, true, this.errorStrategy);
        if (linkOrNull != null && linkOrNull.isSymLink()) {
            ok = this.archiveSymLink("", linkOrNull, absolutePath, pathVisitorOrNull);
        } else if (absolutePath.isDirectory()) {
            ok = this.archiveDirectory(absoluteRoot, absolutePath, strategy, chunkSize, pathVisitorOrNull);
        } else if (absolutePath.isFile()) {
            LinkRecord pseudoLinkForChecksum = new LinkRecord();
            ok = this.archiveFile(absolutePath, hdf5ObjectPath, pseudoLinkForChecksum, strategy.doCompress(hdf5ObjectPath), chunkSize, pathVisitorOrNull);
            crc32 = pseudoLinkForChecksum.getCrc32();
        } else {
            ok = false;
            this.errorStrategy.dealWithError(new ArchivingException(absolutePath, new IOException("Path corresponds to neither a file nor a directory.")));
        }
        if (ok) {
            this.updateIndicesOnThePath(absoluteRoot, absolutePath, crc32, groupExists, strategy.doStoreOwnerAndPermissions());
        }
        return this;
    }

    private void updateIndicesOnThePath(File root, File path, int crc32, boolean immediateGroupOnly, boolean storeOwnerAndPermissions) {
        String rootAbsolute = root.getAbsolutePath();
        File pathProcessing = path;
        int crc32Processing = crc32;
        do {
            File dirProcessingOrNull;
            String dirAbsolute;
            String string = dirAbsolute = (dirProcessingOrNull = pathProcessing.getParentFile()) != null ? dirProcessingOrNull.getAbsolutePath() : "";
            if (dirProcessingOrNull == null || !dirAbsolute.startsWith(rootAbsolute)) break;
            String hdf5GroupPath = HDF5ArchiveUpdater.getRelativePath(rootAbsolute, dirAbsolute);
            DirectoryIndex index = this.indexProvider.get(hdf5GroupPath, false);
            LinkRecord linkOrNull = LinkRecord.tryCreate(pathProcessing, storeOwnerAndPermissions, this.errorStrategy);
            if (linkOrNull != null) {
                linkOrNull.setCrc32(crc32Processing);
                crc32Processing = 0;
                index.updateIndex(linkOrNull);
            }
            pathProcessing = dirProcessingOrNull;
        } while (!immediateGroupOnly);
    }

    private void updateIndicesOnThePath(String path, LinkRecord link, boolean immediateGroupOnly) {
        String pathProcessing;
        String string = pathProcessing = path.startsWith("/") ? path : "/" + path;
        if ("/".equals(pathProcessing)) {
            return;
        }
        int crc32 = link.getCrc32();
        long size = link.getSize();
        long lastModified = link.getLastModified();
        short permissions = link.getPermissions();
        int uid = link.getUid();
        int gid = link.getGid();
        FileLinkType fileLinkType = link.getLinkType();
        String linkTargetOrNull = link.tryGetLinkTarget();
        do {
            String hdf5GroupPath = Utils.getParentPath(Utils.normalizePath(pathProcessing));
            DirectoryIndex index = this.indexProvider.get(hdf5GroupPath, false);
            String hdf5FileName = FilenameUtils.getName(pathProcessing);
            LinkRecord linkProcessing = new LinkRecord(hdf5FileName, linkTargetOrNull, fileLinkType, size, lastModified, uid, gid, permissions, crc32);
            index.updateIndex(linkProcessing);
            fileLinkType = FileLinkType.DIRECTORY;
            crc32 = 0;
            size = -1L;
            linkTargetOrNull = null;
            pathProcessing = hdf5GroupPath;
        } while (!immediateGroupOnly && pathProcessing.length() != 0);
    }

    private boolean archiveDirectory(String parentDirectory, LinkRecord link, IPathVisitor pathVisitorOrNull) {
        String GroupPath = String.valueOf(parentDirectory) + "/" + link.getLinkName();
        try {
            this.hdf5Writer.createGroup(GroupPath);
            return true;
        }
        catch (HDF5Exception ex) {
            this.errorStrategy.dealWithError(new ArchivingException(GroupPath, ex));
            return false;
        }
    }

    private boolean archiveDirectory(File root, File dir, ArchivingStrategy strategy, int chunkSize, IPathVisitor pathVisitorOrNull) {
        File[] fileEntries = dir.listFiles();
        if (fileEntries == null) {
            this.errorStrategy.dealWithError(new ArchivingException(dir, new IOException("Cannot read directory")));
            return false;
        }
        String hdf5GroupPath = HDF5ArchiveUpdater.getRelativePath(root, dir);
        if (!"/".equals(hdf5GroupPath)) {
            try {
                if (this.hdf5Writer.getFileFormat() != IHDF5WriterConfigurator.FileFormat.STRICTLY_1_8 && fileEntries.length > 100) {
                    int totalLength = HDF5ArchiveUpdater.computeSizeHint(fileEntries);
                    this.hdf5Writer.createGroup(hdf5GroupPath, totalLength * 5);
                } else {
                    this.hdf5Writer.createGroup(hdf5GroupPath);
                }
            }
            catch (HDF5Exception ex) {
                this.errorStrategy.dealWithError(new ArchivingException(hdf5GroupPath, ex));
            }
        }
        List<LinkRecord> linkEntries = DirectoryIndex.convertFilesToLinks(fileEntries, strategy.doStoreOwnerAndPermissions(), this.errorStrategy);
        if (pathVisitorOrNull != null) {
            pathVisitorOrNull.visit(hdf5GroupPath);
        }
        Iterator<LinkRecord> linkIt = linkEntries.iterator();
        int i = 0;
        while (i < fileEntries.length) {
            File file = fileEntries[i];
            LinkRecord link = linkIt.next();
            if (link == null) {
                linkIt.remove();
            } else {
                String absoluteEntry = file.getAbsolutePath();
                if (link.isDirectory()) {
                    if (strategy.doExclude(absoluteEntry, true)) {
                        linkIt.remove();
                    } else {
                        boolean ok = this.archiveDirectory(root, file, strategy, chunkSize, pathVisitorOrNull);
                        if (!ok) {
                            linkIt.remove();
                        }
                    }
                } else if (strategy.doExclude(absoluteEntry, false)) {
                    linkIt.remove();
                } else if (link.isSymLink()) {
                    boolean ok = this.archiveSymLink(hdf5GroupPath, link, file, pathVisitorOrNull);
                    if (!ok) {
                        linkIt.remove();
                    }
                } else if (link.isRegularFile()) {
                    String hdf5ObjectPath = HDF5ArchiveUpdater.getRelativePath(root, file);
                    boolean ok = this.archiveFile(file, hdf5ObjectPath, link, strategy.doCompress(hdf5ObjectPath), chunkSize, pathVisitorOrNull);
                    if (!ok) {
                        linkIt.remove();
                    }
                } else {
                    this.errorStrategy.dealWithError(new ArchivingException(file, new IOException("Path corresponds to neither a file nor a directory.")));
                }
            }
            ++i;
        }
        boolean verbose = pathVisitorOrNull != null;
        DirectoryIndex index = this.indexProvider.get(hdf5GroupPath, verbose);
        index.updateIndex(linkEntries);
        return true;
    }

    private boolean archiveSymLink(String hdf5GroupPath, LinkRecord link, IPathVisitor pathVisitorOrNull) {
        if (link.tryGetLinkTarget() == null) {
            this.errorStrategy.dealWithError(new ArchivingException(link.getLinkName(), new IOException("Link target not given for symbolic link.")));
            return false;
        }
        return this.archiveSymLink(hdf5GroupPath, link, link.tryGetLinkTarget(), pathVisitorOrNull);
    }

    private boolean archiveSymLink(String hdf5GroupPath, LinkRecord link, File file, IPathVisitor pathVisitorOrNull) {
        String linkTarget = LinkRecord.tryReadLinkTarget(file);
        if (linkTarget == null) {
            this.errorStrategy.dealWithError(new ArchivingException(file, new IOException("Cannot read link target of symbolic link.")));
            return false;
        }
        return this.archiveSymLink(hdf5GroupPath, link, linkTarget, pathVisitorOrNull);
    }

    private boolean archiveSymLink(String hdf5GroupPath, LinkRecord link, String linkTarget, IPathVisitor pathVisitorOrNull) {
        try {
            String hdf5LinkPath = String.valueOf(hdf5GroupPath) + "/" + link.getLinkName();
            this.hdf5Writer.createSoftLink(linkTarget, hdf5LinkPath);
            if (pathVisitorOrNull != null) {
                pathVisitorOrNull.visit(hdf5LinkPath);
            }
        }
        catch (HDF5Exception ex) {
            this.errorStrategy.dealWithError(new ArchivingException(String.valueOf(hdf5GroupPath) + "/" + link.getLinkName(), ex));
            return false;
        }
        return true;
    }

    private static int computeSizeHint(File[] entries) {
        int totalLength = 0;
        File[] fileArray = entries;
        int n = entries.length;
        int n2 = 0;
        while (n2 < n) {
            File entry = fileArray[n2];
            totalLength += entry.getName().length();
            ++n2;
        }
        return totalLength;
    }

    private boolean archiveFile(File file, String hdf5ObjectPath, LinkRecord link, HDF5GenericStorageFeatures features, int chunkSize, IPathVisitor pathVisitorOrNull) throws ArchivingException {
        boolean ok = true;
        try {
            DataSetInfo info = this.copyToHDF5(file, hdf5ObjectPath, features, chunkSize);
            link.setSize(info.size);
            link.setCrc32(info.crc32);
            if (pathVisitorOrNull != null) {
                pathVisitorOrNull.visit(hdf5ObjectPath);
            }
        }
        catch (IOException ex) {
            ok = false;
            this.errorStrategy.dealWithError(new ArchivingException(file, ex));
        }
        catch (HDF5Exception ex) {
            ok = false;
            this.errorStrategy.dealWithError(new ArchivingException(hdf5ObjectPath, ex));
        }
        return ok;
    }

    private static String getRelativePath(File root, File entry) {
        return HDF5ArchiveUpdater.getRelativePath(root.getAbsolutePath(), entry.getAbsolutePath());
    }

    private static String getRelativePath(String root, String entry) {
        String path = entry.substring(root.length());
        if (path.length() == 0) {
            return "/";
        }
        return FilenameUtils.separatorsToUnix(path);
    }

    private DataSetInfo copyToHDF5(File source, String objectPath, HDF5GenericStorageFeatures compression, int chunkSize) throws IOException {
        FileInputStream input = FileUtils.openInputStream(source);
        try {
            DataSetInfo dataSetInfo = this.copyToHDF5(input, objectPath, compression, chunkSize);
            return dataSetInfo;
        }
        finally {
            IOUtils.closeQuietly(input);
        }
    }

    private int getEffectiveChunkSize(int chunkSize) {
        return chunkSize <= 0 || chunkSize > this.buffer.length ? this.buffer.length : chunkSize;
    }

    private DataSetInfo copyToHDF5(InputStream input, String objectPath, HDF5GenericStorageFeatures compression, int chunkSize) throws IOException {
        int effectiveBufferLength = this.getEffectiveChunkSize(chunkSize);
        CRC32 crc32 = new CRC32();
        HDF5GenericStorageFeatures features = compression;
        int n = this.fillBuffer(input, effectiveBufferLength);
        if (n < effectiveBufferLength) {
            if (n <= 4096 || !features.isDeflating()) {
                features = HDF5GenericStorageFeatures.GENERIC_CONTIGUOUS;
            }
            HDF5OpaqueType type = this.hdf5Writer.createOpaqueByteArray(objectPath, OPAQUE_TAG_FILE, n, features);
            this.hdf5Writer.writeOpaqueByteArrayBlockWithOffset(objectPath, type, this.buffer, n, 0L);
            crc32.update(this.buffer, 0, n);
            return new DataSetInfo(n, (int)crc32.getValue());
        }
        HDF5OpaqueType type = this.hdf5Writer.createOpaqueByteArray(objectPath, OPAQUE_TAG_FILE, 0L, effectiveBufferLength, compression);
        long count = 0L;
        while (n != -1) {
            this.hdf5Writer.writeOpaqueByteArrayBlockWithOffset(objectPath, type, this.buffer, n, count);
            count += (long)n;
            crc32.update(this.buffer, 0, n);
            n = this.fillBuffer(input, effectiveBufferLength);
        }
        return new DataSetInfo(count, (int)crc32.getValue());
    }

    private int fillBuffer(InputStream input, int bufferLength) throws IOException {
        int ofs = 0;
        int len = bufferLength;
        int count = 0;
        int n = 0;
        while (len > 0 && -1 != (n = input.read(this.buffer, ofs, len))) {
            ofs += n;
            len -= n;
            count += n;
        }
        return count;
    }

    private static class DataSetInfo {
        final long size;
        final int crc32;

        DataSetInfo(long size, int crc32) {
            this.size = size;
            this.crc32 = crc32;
        }
    }

    private final class H5ARIOutputStream
    implements IOutputStream,
    Flushable {
        private final IOutputStream delegate;
        private final String directory;
        private final String path;
        private final LinkRecord link;
        private final CRC32 crc32 = new CRC32();
        private long size = 0L;

        public H5ARIOutputStream(String directory, LinkRecord link, int chunkSize, boolean compress) {
            this.directory = Utils.normalizePath(directory);
            this.path = "/".equals(this.directory) ? String.valueOf(this.directory) + link.getLinkName() : String.valueOf(this.directory) + "/" + link.getLinkName();
            this.link = link;
            HDF5GenericStorageFeatures creationStorageFeature = compress ? HDF5GenericStorageFeatures.GENERIC_DEFLATE : HDF5GenericStorageFeatures.GENERIC_NO_COMPRESSION;
            this.delegate = HDF5IOAdapterFactory.asIOutputStream(HDF5ArchiveUpdater.this.hdf5Writer, this.path, creationStorageFeature, HDF5ArchiveUpdater.this.getEffectiveChunkSize(chunkSize), HDF5ArchiveUpdater.OPAQUE_TAG_FILE);
            HDF5ArchiveUpdater.this.indexProvider.get(directory, false).addFlushable(this);
        }

        public void write(int b) throws IOExceptionUnchecked {
            this.crc32.update(b);
            ++this.size;
            this.delegate.write(b);
        }

        public void write(byte[] b) throws IOExceptionUnchecked {
            this.crc32.update(b);
            this.size += (long)b.length;
            this.delegate.write(b);
        }

        public void write(byte[] b, int off, int len) throws IOExceptionUnchecked {
            this.crc32.update(b, off, len);
            this.size += (long)len;
            this.delegate.write(b, off, len);
        }

        public void flush() throws IOExceptionUnchecked {
            this.link.setCrc32((int)this.crc32.getValue());
            this.link.setSize(this.size);
            boolean updateImmediateGroupOnly = HDF5ArchiveUpdater.this.hdf5Writer.isGroup(this.directory);
            HDF5ArchiveUpdater.this.updateIndicesOnThePath(this.path, this.link, updateImmediateGroupOnly);
            this.delegate.flush();
        }

        public void synchronize() throws IOExceptionUnchecked {
            this.delegate.synchronize();
        }

        public void close() throws IOExceptionUnchecked {
            this.flush();
            this.delegate.close();
            HDF5ArchiveUpdater.this.indexProvider.get(this.path, false).removeFlushable(this);
        }
    }
}

