"""
This module computes the Tate bilinear pairing.
"""

from . import f3m
from . import f33m
from . import f36m
from . import ecc

_params = (
    "5 4 5 31 0906 1b00",
    "7 2 6 43 2a40 2209",
    "11 2 15 25411 0d02b004 33034c04",
    "17 16 15 30091 681000150100 8d7400000801",
    "19 2 16 53923 16c305c10802 d0210027d407",
    "23 8 21 1476463 05e26c580c10 015b10c68005",
    "25 6 21 1530601 d08965002a601001 18205201e09da800",
    "29 4 26 37464463 8229d81819120302 01404103ec178210",
    "31 20 47 88239050462461 c2e1c817151a0368 00226150c614060b",
    "37 6 56 64326272436179833 c0c05008053311868108 03095c2302f062a00819",
    "47 32 72 3798402051319990403467 9cb040318c31004a1c06024e 811493090d0944ab08841026",
    "53 22 82 2769035095381770614187391 0100208048000876e58909b25b10 e22640605a06150c88b314047108",
    "73 30 104 11009154363058726777293427189333 52c72110801543ccf40024100021766094330301 498226395d134434e4012205c140006418080000",
    "89 64 118 181932967220635112252099081035759771 40a085451a131350c7092001080e481a8044e0831836c400 63346448008e00128011030084811330d5311a482f80d001",
    "97 12 151 2726865189058261010774960798134976187171462721 b412dc4d18cd020e000e6151000aac2390a432bd41bc40800801 c860506071801ad00b20d31300029b830586342427e01e20e000",
    "121 40 155 31577918281911659253350468036837879363403758167 c092d4424aa23bd00484983644394a0013282ba5145100093819064828042000 90914a040d4090d014002b1a50f712002f020081f0136828886994e003086500",
    "137 136 155 33098280119090191028775580055082175056428495623 50902711c34003446214c0045007930cb401206e88e81c29e49084c3071a05786ca34200 0a8a7827270001080340418291581b24100141700108d8f8266020223054468540d1c200",
    "139 80 157 99228442119121453369139167190862266059691529391 16090a8044405810b8080952c48a39b40305e9f244468828228a461564203855460a6800 40144e7063c093a15218452116a086000006110ba108881b000024c7820a60051087a801",
    "163 80 256 84268735918094105836318246511533764121140010481130741067443071103148817701717 024d81931451fa88043cd10000583810022040980074005c4ce922055779c30c571c80400e119fa92204 09a21201d880228186218845002a19c08c0d0e3002904120ee2109905c40c6422a191562345240200401",
    "167 92 262 6825767609365622572741777967434234893829444442089535769018035530821792872561467 0038d4a32e82a040704020008c100254019012c004bcc1235c11794d188c3ac77a30ed85a33847ac0871 901c34c0f8dc4802ba3040546844f0118706c048216243c21b062325f000c29f2012aa02aa68500b0244",
    "193 12 272 5419993450562850297857167297670010193827783696301157744581770569855881629479145037 3030ea080d6d51a520299b956402490504d003021053bec600c2861456a0820a40cc40404a9a5d2298f103e0890b20402001 ac052348a010a1ad341e376316314ad80d390a05000202900010c218260844061088014018010eb521b004749268c9d12001",
    "239 24 293 11645921167942370591874819266315038503211534885578185054924562964549681039843332188583217 884c048130890c4aa2503180599aa0007306964028103e1c9c00672b2c1a52a1715ec7021011550f0217002554980429219311cc808321da90c08200 1409669102422c3c9546a318128511552ca64b089a135d0c5b22236c281208d4804c9c01118308985c82c94028228208101644ec02b1245044008640",
    "253 12 300 1252664247298565483627798763061050195550010540089250803231992405086311062075236767656632029 20e051d2ae1680cd1084830684f1244125608183062713700948305002120812880f082441410102674114093a005208da1648742150000722140e8fdda42201 2432bbc21d0388902a3a8330021214f1b26629d800980f682114991a2c6e8212d9800010c2bc356590c05c844125020c4111d200e761f0115aca2284d1002100",
    "269 262 332 6098123594898015235178889861709486393553373026665997484538345610949391417131368575719515683194766411 208e202a0d8784004412810704b80c6c103a7e802020c1d000024764a872000628010e210fd1c24071eea82c380872437390a90400270e881a2e0fa1980044800c81940c c818006bdc3992ab0d006401d2804c822e044012380320183c00610675029744b1162722b48401446140305e980e2558130d9009bccd82bcc0a343d70e50021900b04401",
    "277 24 391 4712846864217433368196950515441583378910085434731725171167315433422009511668953281307698995253933810936913898860681463 202710140004a7567ba89052b8049048209e2514a38a1480f000059100049a0132030add104a889b31182084462b0906920486c100d06a5024a87d0d00500e7f1b04aa851814 b0ced6430224acc826b0bd407282210ef619664838d504193000078073324a30440a044220010c0d195211080940318d391031092201370220bbc60d53d84e04cd154c902019",
    "289 216 413 12110237121388586273633515584110013226458147617120215864841293225721772921309935053138239509622675684739528032647726444916937 99292900a024e1a00a6e8c0003050b5a2561d62508114066412123a194040045144c8b400066c0d6b94cc81c06d00012478898a005489c2010e2ae1e903e56041a61c16b38c100601600 35b90b33010aaa10091dc0f9b12605a402726a1813d309d0902a0850685bf64e08686489018a02b48c16605425f222150206809800280194c2a000100c2b01862c04a00021e5901b1200",
    "313 126 460 1740152565458953073634994052791514658464681966876165910633848150891357424728224859850966359875510494253226650526622032998680170543382642037 820558419820252630a0d4204b38107849d2ee5164089220a330ab8c222c44000a882499099d040014da21b004004ad9c95d2807140407069005118e82f3680a0406543311500859f463016094401101 e00201980043a39204a1690504028c828b506929a166900b01f0614a6206580d0e2402a00d160e00174d86445f841c449b06927aca547314102880024c984560420d02b115f0833060d3551da008a001",
    "395 338 485 80172097064154181257340545445945701478615643539554910656655431171167598268341527430200810544156625333601812351266052856520678455274751591367269291 0842404016820cc2d4c1414802d7106b83cbfc0e0080d82c0e1a0a761ad0849e1b335309192bc004811526c244f445010800433da51fa00540042124ae252128ec9048140170063425522061f50065056b61a4cc847622d00c3142a05931a10aaaca4602 62aa4841012c21a043fc363a600545c89e4e684830882a01c5b0803619f23b5b0002801140a48200049450b0c486783839079810a3b4ee914214bc01814012d2b027203012824445d52802087b88c40d04048fd122e48a03290ff24aad4e13218347c600",
    "413 22 494 44792676566930626747128588835039760252218989372927310770910976361699556551237605799643618249972834300903220124751255253755196166139481881539923184049 9209c143180781449420152d4e2c06069140c74a0022c205ddb24520d43276005441c08109749518003504dc411208b0d61e000c4152123c031044812a1c880020c27969428538a58f5509c00201220e0300086b2924043616006060fec0d3202864364208200201 b54438e2111c9e423783208275c89428008b082a860013c58800304050005d068828898b026ab1481103d2410010804101506f1440934104c0e3602440008b340a170057cd60769411d5043a12f5cf9d0eaf80f033040410341502a0e88425929fca792e0c828002",
    "421 24 558 766095903370946936328977113720428544867444971502516671915420662414613284821795003200938190677676484768890593305562786566799531495126788130969596470061001070877913721493 0051582959000318bc120830614e605735d0c2f4b88d2c10840b507020440c1a1aa50b5a954232000e4d7352012d024ca08851a51e6b24a08602d50ca64368324a18119188002c38084132c2614334ac8cd61820e18552c0a1603104b9e0b00081865029925d12a44a00 b24e841093404255c80c12360a8f0a820843c23c1e051aa02342514005953182910041a21831405a221c04041400023a09fa319900080051a408b7012a31e3010151608018f7802cc1413ac00508b58615582284700c8c3255c546b98089e1a15a8924c1019405c20209",
    "443 188 597 286994573195522671783569752843814954931005517693442866628385002514472327521108606853847529710200440188889761335818189573634744921114245155602346332378537306079049242196812199055723 214acc4414ab0f108814da83404b8940662800e4c804c6200560080512800c1815c1300b5b8828cd20d80210540615601785161703900804c6b510a24144f020642300542d2440080057d60324711001520a4208203453e6422ac9a404731520c02584ab22c8ea89e84a81e8c4281502 618078dc3da401680141408f8415808f0c88c2a88a944009c33a3d5213e7539701c4d6650ba5c5247858553406248da08106e48000b471010a2b050142188a049e36a3303b686e407035001224092a8024c142a9c40080009a0a0190f442220182a22082d9c812126c30085fab0a8604",
    "467 48 645 108220469499363631995525712756135494735252733492048868417164002000654321383482753640072319529019505742300964525569770933946381504691909098938045089999753901375631613294579329433690943459352138231 68e504039d8050aca580950e8a04304268621333c4400f2905cd20c3080113003a5d46540a6d082202613c808a048221c10f28b064280c5622a000060828a8405d0451403242b101880918861da4c42b8110061a100b04a4f0c86f40823120e182074d88024269306b209c1ea003021b125309c41b06 ecf03009021c8500a000f132a642040c0a6000aa2251911c43303b019256e820c4d6a3a068127a4c5143ee22aa1a622401104d324ec18089c8410200090044e1a270ed55ec008811983b03009e0c404508624138c900f40421108c21010c45910581032a08009910818d4874a8924501027c7013b805",
    "493 4 655 123209830885067439469541117476726593415493307419479916852122774108282502720664530088563526403396399075460482335100190835116991459881196401183702809789568034105117984556933759942709262108642530208151 0c8a62550004264284088c0fc308304bf1214c04ce88cc418a980d9802480d00c8010ac620280044056b220418a22cec04d24125d0811022a58b004f4a0251108c8a965119912150310038e482040a16a0fa3165338461077066f997526b321c64315f4584808080c5c8c51c0103d20db812061e689d4a301120b40c 14608954694d5bd541e05a1119936d10906280f19025ef00c1c20f006291d88211cb40488c8517231481d057f91e3a8060228256811000008197e1134a024280700a00a2a408921284ac224882232d04720e659810182a00e03c840a0210a804b337005a880888722b0802818418131d70a1764fa8d40e4804c8350d",
    "503 104 697 545523657676112447260904563578912738373307867219686215849632469801471112426878939776725222290437653718473962733760874627315930933126581248465899651120481066111839081575164964589811985885719017214938514563804313 40a40721c51131c216329065c46c71a44300655926943648214811801b0484246b6ac0303720020a80805a81894c9bda9c08026844e8022100a4200621c02430485816002c8014e08163181802825b20661826010b492350b7aa0280b33b8984942e8d0819a4d53371844c06806424238525133206b08e704952a8460101 200596b801598f0b4c100e026d418440b8298d2b8c8216d073c086c888424338162b288610347d50e849e04e7018958006d4194514814840000070200b19650f6a01456ea44000322ed0000096522c03066000730d21248831001120083c06a8d45419e202828507a4020184c14238280066880b00b13eac2c09c8948410",
    "509 358 804 102239946202586852409809887418093021457150612495255706614733003327526279081563687830782748305746187060264985869283524441819589592750998086186315250781067131293823177124077445718802216415539934838376431091001197641295264650596195201747790167311 04715a2020234d3690c19a10230122bfd06856281310250c667a0c08a73ae910f0374c625000e00338432a1d08498813134023014d98010079f9f82708543f00c38e04190b44a089662a20a0d06a58402e0409406ca49252080540d5108514ea0248218daf9e18240194846070224704001f10e60000e2d3800006d0522b001f 438ce884172cb502928629a10f3460a586d02063813abd45000d0e414042425cb1b0907c1a22022bd620100496970110094a0d20848018680b0d9cb2a836000014420539800048e461691658c0c9895071225b90604540b2535081a0ad80b882080a208144c4990021cc6d09294088c474853288085904174012014c45c93d12",
    "523 260 814 77905128763524077566304920728728526484926909016380546727953305431329463044841100911312909173360926330643390026799609624979615542902934603433968227214565025284052971369334036241928344439119900117985875313257678242146902382177199561000124403566721 461198287180c982ce26010c01aa064ca42e50206083a80e998068ac34448dca21005c0088020b00445b20834248c2500b66403400e9e64620c488822c5401c4100401ea4315840f06043181eca1c845d1324a910450856857a0067585124323022094210199509840c8b0845c2ca5821828e4990709061208010400350841a19a200601 bb64eb20822af05e22130a4ca0962b020834345643a00792f1808889084814c85b860c008421da3d3045749012786dbaa0e2acc80bdc9904410821825e044bc00601401b101f05540ea0c86c34024148c4997748cb08a84fa02004411056a1936b16a011a35e629800828eb88b4f21831245110942321422064306a1d8790162b029c100",
    "557 382 822 22678199448850737741470023015474464957229393423594505255973411898087609752859197360536491103616512827211567572812303499136898211491729662160857574093598860125843809492644953118512668398303205062080581182126851585493880341845392329518114488694364631 98228304a5e51d700041f4071e254083eca043256910012ac009229c0d1190c286a35dbc5408aa7312a0ec58445968864cdf08dc880a010a0899564a0e020200ac302c45c2044641445a520ae007d91603300112b264111c14d0126306913cc29561508268017850024222b6010cc00003048a200729a300450004359200b5620101d0654114018c82200100 805800889324c03746b100f1ad0c8041001074614bf71449040129108c0c8b6f6488959010e60a983060603283861aa3a0150b5523fb682b06443043e88484a5481712a3450804a0cf67489902c8190cb40002625c8eb14003108408c39091eac06542304480825702638c01446102869b0c7400c54441e2200084048500a882cc841019515012a060583804",
    "593 112 911 8787559841310502548112202964508393415022198003343919775555629297577875431685877534501861059841910816121826316070725098197804946846072868704004042819334740500073251048524810119935754492326811092034218456907415187519158187004726202613531723766536810294204598390767415388972923 2632243ab8480003d8041128c2150182285c96b2029552aaa814114262004c3c583505d6025c061bd4436180e900606220204a803218e017011100a42a82a4a0203ab4436c100400080200c949c14503b56684225b2414244af468c6820105c5402d4152e14ca9996601408402ca2064a210602b2c885c12ee9c180ed8b011c48513082422b45994404b401f85409c93e53bbef3ad00 97d719a1002a10419044dab571414f100c28c3dc2c0a1442c4ca9ca4b148f9b621100722111241326040a2a14a4912447449001028803b348e32002c252aa41844a698442c052162c0e8004828a2046a51212a0f8a010a8ca2808d02063c0390b1e005281043004233000884c0a0400ea0b2490f284d0e1102a9890990b482020400837040710042914a4008002689d0e84681250200",
    )

def _algo4a(t, u):
    '''computing of $(-t^2 +u*s -t*p -p^2)^3$
    The algorithm is by J.Beuchat et.al, in the paper of "Algorithms and Arithmetic Operators for Computing
    the $eta_T$ Pairing in Characteristic Three", algorithm 4 in the appendix
    '''
    c0 = f3m.cubic(t) # c0 == t^3
    c1 = f3m.cubic(u)
    f3m.neg(c1, c1) # c1 == -u^3
    m0 = f3m.mult(c0, c0) # m0 == c0^2
    v0 = f3m.zero()
    f3m.neg(m0, v0) # v0 == -c0^2
    f3m.sub(v0, c0, v0) # v0 == -c0^2 -c0
    f3m._add2(v0) # v0 == -c0^2 -c0 -1
    v1 = c1
    v2 = f3m.one()
    f3m.sub(v2, c0, v2) # v2 == 1 -c0
    return [[v0, v1], [v2, f3m.zero()], [f3m.two(), f3m.zero()]]

def _algo4(xp, yp, xq, yq):
    re = f3m._m % 12
    xp = f3m._clone(xp)
    f3m._add1(xp) # xp == xp + b
    yp = f3m._clone(yp)
    if re == 1 or re == 11: 
        f3m.neg(yp, yp) # yp == -\mu*b*yp, \mu == 1 when re==1, or 11
    xq = f3m.cubic(xq) # xq == xq^3
    yq = f3m.cubic(yq) # yq == yq^3
    t = f3m.zero(); f3m.add(xp, xq, t) # t == xp+xq
    nt = f3m.zero()
    f3m.neg(t, nt) # nt == -t
    nt2 = f3m.mult(t, nt) # nt2 == -t^2
    v2 = f3m.mult(yp, yq) # v2 == yp*yq
    v1 = f3m.mult(yp, t) # v1 == yp*t
    if re == 7 or re == 11: # \lambda == 1
        nyp = f3m.zero(); f3m.neg(yp, nyp) # nyp == -yp
        nyq = f3m.zero(); f3m.neg(yq, nyq) # nyq == -yq
        a1 = [[v1, nyq], [nyp, f3m.zero()], [f3m.zero(), f3m.zero()]]
        # a1 == \lambda*yp*t -\lambda*yq*s -\lambda*yp*p
    else: # \lambda == -1
        f3m.neg(v1, v1) # v1 == -yp*t
        a1 = [[v1, yq], [yp, f3m.zero()], [f3m.zero(), f3m.zero()]]
    a2 = [[nt2, v2], [nt, f3m.zero()], [f3m.two(), f3m.zero()]] 
    # a2 == -t^2 +yp*yq*s -t*p -p^2
    R = f36m.mult(a1, a2)
    for _ in range((f3m._m - 1) // 2):
        R = f36m.cubic(R)
        xq = f3m.cubic(xq)
        xq = f3m.cubic(xq)
        f3m._add2(xq) # xq <= xq^9-b
        yq = f3m.cubic(yq)
        yq = f3m.cubic(yq)
        f3m.neg(yq, yq) # yq <= -yq^9
        f3m.add(xp, xq, t) # t == xp+xq
        f3m.neg(t, nt) # nt == -t
        nt2 = f3m.mult(t, nt) # nt2 == -t^2
        u = f3m.mult(yp, yq) # u == yp*yq
        S = [[nt2, u], [nt, f3m.zero()], [f3m.two(), f3m.zero()]]
        R = f36m.mult(R, S)
    return R

def _algo5(xp, yp, xq, yq):
    '''computing the $\eta_T$ pairing
    This is the algorithm 5 in the paper above. 
    
    :param xp: the x coordinate of element $P=[xp, yp]$
    :type xp: list
    :param yp: the y coordinate of element $P=[xp, yp]$
    :type yp: list
    :param xq: the x coordinate of element $R=[xq, yq]$
    :type xq: list
    :param yq: the y coordinate of element $R=[xq, yq]$
    :type yq: list
    :returns: \eta_T(P,Q)
    '''
    re = f3m._m % 12
    xp = f3m._clone(xp)
    f3m._add1(xp) # xp == xp + b
    yp = f3m._clone(yp)
    if re == 1 or re == 11: 
        f3m.neg(yp, yp) # yp == -\mu*b*yp, \mu == 1 when re==1, or 11
    xq = f3m.cubic(xq) # xq == xq^3
    yq = f3m.cubic(yq) # yq == yq^3
    t = f3m.zero(); f3m.add(xp, xq, t) # t == xp+xq
    nt = f3m.zero()
    f3m.neg(t, nt) # nt == -t
    nt2 = f3m.mult(t, nt) # nt2 == -t^2
    v2 = f3m.mult(yp, yq) # v2 == yp*yq
    v1 = f3m.mult(yp, t) # v1 == yp*t
    if re == 7 or re == 11: # \lambda == 1
        nyp = f3m.zero(); f3m.neg(yp, nyp) # nyp == -yp
        nyq = f3m.zero(); f3m.neg(yq, nyq) # nyq == -yq
        a1 = [[v1, nyq], [nyp, f3m.zero()], [f3m.zero(), f3m.zero()]]
        # a1 == \lambda*yp*t -\lambda*yq*s -\lambda*yp*p
    else: # \lambda == -1
        f3m.neg(v1, v1) # v1 == -yp*t
        a1 = [[v1, yq], [yp, f3m.zero()], [f3m.zero(), f3m.zero()]]
    a2 = [[nt2, v2], [nt, f3m.zero()], [f3m.two(), f3m.zero()]] 
    # a2 == -t^2 +yp*yq*s -t*p -p^2
    R = f36m.mult(a1, a2)
    nu = f3m.zero()
    for _ in range((f3m._m - 1) // 4):
        R = f36m.cubic(R)
        R = f36m.cubic(R) # R <= R^9
        xq = f3m.cubic(xq)
        xq = f3m.cubic(xq)
        f3m._add2(xq) # xq <= xq^9-b
        yq = f3m.cubic(yq)
        yq = f3m.cubic(yq) # yq <= yq^9
        f3m.add(xp, xq, t) # t == xp+xq
        u = f3m.mult(yp, yq) # u == yp*yq
        f3m.neg(u, nu) # nu == -yp*yq
        S = _algo4a(t, nu) # S == (-t^2 -u*s -t*p -p^2)^3
        xq = f3m.cubic(xq)
        xq = f3m.cubic(xq)
        f3m._add2(xq) # xq <= xq^9-b
        yq = f3m.cubic(yq)
        yq = f3m.cubic(yq) # yq <= yq^9
        f3m.add(xp, xq, t) # t == xp+xq
        u = f3m.mult(yp, yq) # u == yp*yq
        f3m.neg(t, nt) # nt == -t
        nt2 = f3m.mult(t, nt) # nt2 == -t^2
        S2 = [[nt2, u], [nt, f3m.zero()], [f3m.two(), f3m.zero()]] # S2 == -t^2 +u*s -t*p -p^2
        S = f36m.mult(S, S2)
        R = f36m.mult(R, S)
    return R

def _algo6(u):
    '''computation of $U ^ {3^{3m} - 1}$
    This is the algorithm 6 in the paper above. '''
    u0, u1 = u[0]
    u2, u3 = u[1]
    u4, u5 = u[2]
    v0 = [u0, u2, u4]
    v1 = [u1, u3, u5]
    m0 = f33m.mult(v0, v0)
    m1 = f33m.mult(v1, v1)
    m2 = f33m.mult(v0, v1)
    a0 = f33m.zero()
    f33m.sub(m0, m1, a0)
    a1 = f33m.zero()
    f33m.add(m0, m1, a1)
    i = f33m.inverse(a1)
    v0 = f33m.mult(a0, i)
    v1 = f33m.mult(m2, i)
    a0, a1, a2 = v0
    a3, a4, a5 = v1
    return [[a0, a3], [a1, a4], [a2, a5]]

def _algo7(u):
    '''computation of $U ^ {3^m+1}$, $U \in T_2(F_{3^3M})$
    This is the algorithm 7 in the paper above.'''
    u0, u1 = u[0]
    u2, u3 = u[1]
    u4, u5 = u[2]
    a0, a1, a2, a3, a4, a5, a6 = f3m.zero(), f3m.zero(), f3m.zero(), f3m.zero(), f3m.zero(), f3m.zero(), f3m.zero()
    f3m.add(u0, u1, a0)
    f3m.add(u2, u3, a1)
    f3m.sub(u4, u5, a2)
    m0 = f3m.mult(u0, u4)
    m1 = f3m.mult(u1, u5)
    m2 = f3m.mult(u2, u4)
    m3 = f3m.mult(u3, u5)
    m4 = f3m.mult(a0, a2)
    m5 = f3m.mult(u1, u2)
    m6 = f3m.mult(u0, u3)
    m7 = f3m.mult(a0, a1)
    m8 = f3m.mult(a1, a2)
    f3m.add(m5, m6, a3)
    f3m.sub(a3, m7, a3)
    f3m.neg(m2, a4)
    f3m.sub(a4, m3, a4)
    f3m.sub(m3, m2, a5)
    f3m.sub(m1, m0, a6)
    f3m.add(a6, m4, a6)
    v0, v1, v2, v3, v4, v5 = f3m.zero(), f3m.zero(), f3m.zero(), f3m.zero(), f3m.zero(), f3m.zero()
    if f3m._m % 6 == 1:
        f3m.add(m0, m1, v0)
        f3m.add(v0, a4, v0)
        f3m._add1(v0)
        f3m.sub(m5, m6, v1)
        f3m.add(v1, a6, v1)
        f3m.sub(a4, a3, v2)
        f3m.add(m8, a5, v3)
        f3m.sub(v3, a6, v3)
        f3m.add(a3, a4, v4)
        f3m.neg(v4, v4)
        f3m.add(m8, a5, v5)
    else: # f3m._m % 6 == 5
        f3m.add(m0, m1, v0)
        f3m.sub(v0, a4, v0)
        f3m._add1(v0)
        f3m.sub(m6, m5, v1)
        f3m.add(v1, a6, v1)
        v2 = a3
        f3m.add(m8, a5, v3)
        f3m.add(v3, a6, v3)
        f3m.add(a3, a4, v4)
        f3m.neg(v4, v4)
        f3m.add(m8, a5, v5)
        f3m.neg(v5, v5)
    return [[v0, v1], [v2, v3], [v4, v5]]

def _algo8(u):
    '''computing U^M, M=(3^{3m}-1)*(3^m+1)*(3^m+1-\mu*b*3^{(m+1)//2})
    This is the algorithm 8 in the paper above. '''
    v = _algo6(u)
    v = _algo7(v)
    w = v
    for _ in range((f3m._m + 1) // 2):
        w = f36m.cubic(w)
    v = _algo7(v)
    m = f3m._m
    if m % 12 in [1, 11]:
        w0, w1 = w[0]
        w2, w3 = w[1]
        w4, w5 = w[2]
        f3m.neg(w1, w1)
        f3m.neg(w3, w3)
        f3m.neg(w5, w5)
        w = [[w0, w1], [w2, w3], [w4, w5]]
    return f36m.mult(v, w)

def pairing(x1, y1, x2, y2):
    '''computing the Tate bilinear pairing
    
    :param x1: the x coordinate of element $P=[x1, y1]$
    :type x1: list
    :param y1: the y coordinate of element $P=[x1, y1]$
    :type y1: list
    :param x2: the x coordinate of element $R=[x2, y2]$
    :type x2: list
    :param y2: the y coordinate of element $R=[x2, y2]$
    :type y2: list
    :returns: the result
    '''
    if (f3m._m - 1)//2 % 2 == 0:
        f = _algo5
    else:
        f = _algo4
    v = f(x1, y1, x2, y2)
    return _algo8(v)

def init(bits_of_order_of_G1):
    '''init whole module. making sure the order of G1 is at least $bits_of_order_of_G1$'''
    for p in _params:
        p = p.split()
        bit_num = int(p[2])
        if bit_num >= bits_of_order_of_G1:
            m, t, _, order = map(int, p[:4])
            f3m._set_param(m, t)
            ecc._order = order
            x, y = p[4:]
            ecc._x = f3m._from_str(x)
            ecc._y = f3m._from_str(y)
            return
    raise NotImplementedError('max supported bit num is 911')
