# -*- coding: utf-8 -*-

# This file is subject to the terms and conditions defined in
# file 'LICENSE', which is part of this source code package.
#       Copyright (c) 2011 R Pratap Chakravarthy


"""
Implements the run-time engine a stack machine interpreting instructions
generated by :class:`InstrGen`.

Standard libraries and references made available in a template script.
----------------------------------------------------------------------

``_m``,
    Reference to :class:`StackMachine` instance used to generate the final
    HTML text.

``this``,
    Every template script can be viewed as an object instance which can be 
    referenced using ``this``. In case of template scripts making use of 
    inheritance feature, ``this`` will always refer to the template script
    at the end of the inheritance chain.

``local``,
    For non-inheriting template scripts ``this`` and ``local`` refer to the
    same object. In case of template scripts using inheritance feature,
    unlike ``this`` symbol which refers to the template script at the end of
    the inheritance chain, ``local`` will always refer to the template script
    object in which it is used.

``parent``,
    In case of inheriting scripts, ``parent`` will refer to the base template
    from which ``local`` template script derives.

``next``,
    In case of inheriting scripts, ``next`` will refer to the deriving
    template script.

All names ``this``, ``local``, ``parent``, ``next`` refer to the same type of
object, template-module. Having a refence to template-module allows developers
to access global variables and functions defined in the module.

In case if a template script is part of an inheritance chain, then attributes
references on the template-module will propogate towards the top of the chain
until an attributed is found in one of the base module.

``_compiler``,
    Refers to :class:`TTLCompiler` plugin instance.

``_context``,
    Refers to the context dictionary.

``_ttlhash``,
    Refers to the hash value generated using template text.

``_ttlfile``,
    File name of template script.

``__compiler``,
    Temporary variable that is optionally created for compiling and importing
    library templates. Refers to a :class:`TTLCompiler` plugin instance.

``__ttlcode``,
    Temporary variable that is optionally created while importing library
    templates.

``<Implemented-interfaces>``,
    When a template script implements an interface, interface name and plugin
    name will also be made available in the global context.

Other than this every template script will have the following standard imports

.. code-block :: python

    import imp
    from   io                   import StringIO
    from   pluggdapps.plugin    import Plugin, implements
    import pluggdapps.utils     as h
    from   tayra                import BaseTTLPlugin
"""

import re, imp

import pluggdapps.utils     as h
from   pluggdapps.plugin    import pluginname

from   tayra.lexer      import TTLLexer
from   tayra.interfaces import ITayraTags, ITayraEscapeFilter, \
                               ITayraFilterBlock

class StackMachine( object ) :
    """Stack machine instruction interpretr.  Intructions are method calls on
    this object. Supported instructions are,
    
    indent, append, extend, pushbuf, popbuf, popbuftext, handletag, evalexprs,
    inherit
    """

    def __init__( self, compiler ):
        self.compiler = compiler
        self.encoding = compiler.encoding
        self.tagplugins = [ compiler.query_plugin( ITayraTags, name )
                                for name in compiler['use_tag_plugins'] ]
        self.escfilters = [
            compiler.query_plugin( ITayraEscapeFilter, name ) 
            for name in compiler['escape_filters'] ]
        self.filterblocks = compiler.query_plugins( ITayraFilterBlock )
        self.filterblocks = { pluginname(x) : x for x in self.filterblocks }

    def _init( self, ifile ):
        self.bufstack = [ [] ]
        self.ifile = ifile
        self.htmlindent = ''

    #---- Stack machine instructions

    def indent( self ) :
        return self.append( self.htmlindent )

    def upindent( self, up='' ) :
        self.htmlindent += up
        return self.htmlindent

    def downindent( self, down='' ) :
        self.htmlindent = self.htmlindent[:-len(down)]
        return self.htmlindent

    def append( self, value ) :
        self.bufstack[-1].append( value )
        return value

    def extend( self, value ) :
        if isinstance(value, list) :
            self.bufstack[-1].extend( value )
        else :
            raise Exception( 'Unable to extend context stack' )

    def pushbuf( self, buf=None ) :
        buf = []
        self.bufstack.append( buf )
        return buf

    def popbuf( self ) :
        return self.bufstack.pop(-1)

    def popbuftext( self ) :
        x = ''.join( self.popbuf() )
        return x

    regex_tag = re.compile( 
        r'(\{[^\}]*\})|(%s=%s)|(%s)|([^ \t\r\n]+)' % (
            TTLLexer.attrname, TTLLexer.attrvalue, TTLLexer.attrvalue ))

    def handletag( self, contents, tagbegin, indent=False, nl='' ):
        tagbegin = tagbegin.replace('\n', ' ')[1:-1]    # remove < and >
        try    : tagname, tagbegin = tagbegin.split(' ', 1)
        except : tagname, tagbegin = tagbegin, ''
        
        styles, attributes, tokens = [], [], []
        for m in self.regex_tag.finditer( tagbegin ) :
            if not m : continue
            parts = m.groups()
            parts[0] and styles.append( parts[0][1:-1].strip() )
            parts[1] and attributes.append( parts[1] )
            tokens.extend( parts[2:] )
        tokens = list( filter( None, tokens ))

        for plugin in self.tagplugins :
            html = plugin.handle( self, tagname, tokens, styles, attributes,
                                  contents )
            if html == None : continue
            self.append( html )
            break
        else :
            raise Exception("Unable to handle tag %r" % tagname)

    def evalexprs( self, text, filters, globals_, locals_ ) :
        out = str( eval( text, globals_, locals_ ))
        if not filters : return out

        filters = h.parsecsv( filters )
        if 'n' in filters : return ou

        for f in filters :
            for p in self.escfilters :
                out1 = p.filter( self, f, out )
                if out1 != None : 
                    out = out1
                    break
        return out

    # def importttl( self, name, pyfile ):
    #     return imp.load_module( name, open(pyfile), pyfile,
    #                             (".py", "r", imp.PY_SOURCE) )

    def inherit( self, ttlloc, childglobals ):
        compiler = self.compiler()
        code = compiler.compilettl( file=ttlloc )
        # inherit module
        parent_context = childglobals['_context']
        parent_context.update({
            'this'      : childglobals['this'],
            'parent'    : None,
            'next'      : childglobals['local'],
        })
        module = compiler.load( code, context=parent_context )
        childglobals['this']._linkparent( Namespace( None, module ))
        childglobals['local'].parent = module
        return module

    # TODO : Can this method be replaced by pluggdapps.utils.lib.hitch ??
    # def hitch( self, obj, cls, interfacefunc, *args, **kwargs ) :
    #     def fnhitched( self, *a, **kw ) :
    #         kwargs.update( kw )
    #         return interfacefunc( self, *(args+a), **kwargs )
    #     return fnhitched.__get__( obj, cls )

    def use( self, interface, pluginname='' ):
        return queryTTLPlugin( self.ttlplugins, interface, pluginname )


class Namespace( object ):
    def __init__( self, parentnm, localmod ):
        self._parentnm = parentnm
        self._localmod = localmod

    def __getattr__( self, name ):
        if self._parentnm :
            return getattr(
                self._localmod, name, getattr( self._parentnm, name, None )
           )
        else :
            return getattr( self._localmod, name, None )
        
    def __setattr__( self, name, value ):
        if name in [ '_parentnm', '_localmod' ] :
            self.__dict__[name] = value
        else :
            setattr( self._localmod, name, value )
        return value

    def _linkparent( self, parentnm ):
        nm, parnm = self, self._parentnm
        while parnm : nm, parnm = parnm, parnm._parentnm
        nm._parentnm = parentnm
        return parentnm

