#!/usr/bin/python
#Last-modified: 02 Oct 2014 20:59:47

#         Module/Scripts Description
#
# Copyright (c) 2008 1.1.0 &lt;Yunfei Wang&gt;
#
# This code is free software; you can redistribute it and/or modify it
# under the terms of the BSD License (see the file COPYING included with
# the distribution).
#
# @status:  experimental
# @version: 1.1.0
# @author:  Yunfei Wang
# @contact: yfwang0405@gmail.com

# ------------------------------------
# Python packages
# ------------------------------------

import sys
import string
import copy

# ------------------------------------
# constants
# ------------------------------------

# ------------------------------------
# Misc functions
# ------------------------------------

# ------------------------------------
# Classes
# ------------------------------------


class Bed3(object):
    ''' Bed3 class contains the common features of genomic intervals. '''
    def __init__(self,bedline):
        ''' Initiation. '''
        if isinstance(bedline,str):
            bedline = bedline.split()            
        self.chrom, self.start, self.end = bedline
        self.start = int(self.start)
        self.end = int(self.end)
    def __len__(self): # len
        ''' Length. '''
        return self.end-self.start
    def extend(self,up=0,down=0):
        ''' Extend upstream and downstream '''
        new_bed = copy.deepcopy(self)
        new_bed.start = max(self.start-up,0)
        new_bed.end = self.end + down
        return new_bed
    def __str__(self): # str
        ''' string representation. '''
        return "{0}	{1}	{2}".format(self.chrom,self.start,self.end)
    def __cmp__(self,B): # cmp
        ''' Compare items. '''
        return cmp(self.chrom, B.chrom) or cmp(self.start, B.start) or cmp(self.end, B.end)
    def is_overlap(self,B):
        '''
        Test if overlapped with B.
        Parameters:
            B: object of Bed3 or its derived class.
        Returns:
            bool
                True if overlapped, otherwise False.
        '''
        if self.chrom != B.chrom:
            return False
        if self.end <= B.start or B.end <= self.start:
            return False
        return True
    def __add__(self,B):
        '''
        Merge current Bed and B together.
        NOTE: B should be overlapped with current Bed.
        Parameters:
            B: object of Bed3 or its derived class.
        Returns:
            new_bed: object of Bed3 or its derived class.
                Merged Bed.
        '''
        new_bed = copy.deepcopy(self)        
        new_bed.start = min(self.start,B.start)
        new_bed.end = max(self.end,B.end)
        return new_bed


class BedGraph(Bed3):
    ''' BedGraph class contains the common features of genomic intervals. '''
    def __init__(self,bedline):
        ''' Initiation. '''
        if isinstance(bedline,str):
            bedline = bedline.split()
        self.chrom, self.start, self.end, self.score = bedline
        self.start = int(self.start)
        self.end = int(self.end)
        self.score = float(self.score)
    def __str__(self): # str
        ''' string representation. '''
        return "{0}	{1}	{2}	{3}".format(self.chrom,self.start,self.end,self.score)


# ------------------------------------
# Main
# ------------------------------------

if __name__=="__main__":
    if len(sys.argv)==1:
        sys.exit("Example:"+sys.argv[0]+" file1 file2... ")
    

