# -*- coding: utf-8 -*-
#  Copyright 2013 Takeshi KOMIYA
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.

import os
import re
import sys
import socket
import signal
import psycopg2
import tempfile
import subprocess
from glob import glob
from time import sleep
from shutil import copytree
from datetime import datetime

__all__ = ['Postgresql']

SEARCH_PATHS = (['/usr/local/pgsql'] +
                glob('/usr/lib/postgresql/*') +  # for Debian/Ubuntu
                glob('/opt/local/lib/postgresql-*'))  # for MacPorts
DEFAULT_SETTINGS = dict(auto_start=2,
                        base_dir=None,
                        initdb=None,
                        initdb_args='-U postgres -A trust',
                        postmaster=None,
                        postmaster_args='-h 127.0.0.1',
                        pid=None,
                        port=None,
                        copy_data_from=None)


class Postgresql(object):
    def __init__(self, **kwargs):
        self.settings = dict(DEFAULT_SETTINGS)
        self.settings.update(kwargs)
        self.pid = None
        self._owner_pid = os.getpid()

        if self.base_dir:
            if self.base_dir[0] != '/':
                self.settings['base_dir'] = os.path.join(os.getcwd(), self.base_dir)
        else:
            self.settings['base_dir'] = tempfile.mkdtemp()
            self._use_tmpdir = True

        if self.initdb is None:
            self.settings['initdb'] = find_program('initdb', ['bin'])

        if self.postmaster is None:
            self.settings['postmaster'] = find_program('postmaster', ['bin'])

        if self.auto_start:
            if self.auto_start >= 2:
                self.setup()

            self.start()

    def __del__(self):
        import os
        if self._owner_pid == os.getpid():
            self.stop()
            self.cleanup()

    def __enter__(self):
        return self

    def __exit__(self, *args):
        import os
        if self.pid and self._owner_pid == os.getpid():
            self.stop()
            self.cleanup()

    def __getattr__(self, name):
        if name in self.settings:
            return self.settings[name]
        else:
            raise AttributeError("'Postgresql' object has no attribute '%s'" % name)

    def dsn(self, **kwargs):
        # "dbname=test host=localhost user=postgres"
        params = dict(kwargs)
        params.setdefault('port', self.port)
        params.setdefault('host', '127.0.0.1')
        params.setdefault('user', 'postgres')
        params.setdefault('dbname', 'test')

        return params

    def url(self, **kwargs):
        params = self.dsn(**kwargs)

        url = ('postgresql://%s@%s:%d/%s' %
               (params['user'], params['host'], params['port'], params['dbname']))

        return url

    def setup(self):
        # copy data files
        if self.copy_data_from:
            try:
                copytree(self.copy_data_from, os.path.join(self.base_dir, 'data'))
                os.chmod(os.path.join(self.base_dir, 'data'), 0o700)
            except Exception as exc:
                raise RuntimeError("could not copytree %s to %s: %r" %
                                   (self.copy_data_from, os.path.join(self.base_dir, 'data'), exc))

        # (re)create directory structure
        for subdir in ['data', 'tmp']:
            try:
                path = os.path.join(self.base_dir, subdir)
                os.makedirs(path)
                os.chmod(path, 0o700)
            except:
                pass

        # initdb
        if not os.path.exists(os.path.join(self.base_dir, 'data', 'PG_VERSION')):
            args = ([self.initdb, '-D', os.path.join(self.base_dir, 'data')] +
                    self.initdb_args.split())

            try:
                subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.STDOUT).communicate()
            except Exception as exc:
                raise RuntimeError("failed to spawn initdb: %r" % exc)

    def start(self):
        if self.pid:
            return  # already started

        if self.port is None:
            self.port = get_unused_port()

        logger = open(os.path.join(self.base_dir, 'tmp', 'postgresql.log'), 'wt')
        pid = os.fork()
        if pid == 0:
            os.dup2(logger.fileno(), sys.__stdout__.fileno())
            os.dup2(logger.fileno(), sys.__stderr__.fileno())

            try:
                os.execl(self.postmaster, self.postmaster,
                         '-p', str(self.port),
                         '-D', os.path.join(self.base_dir, 'data'),
                         '-k', os.path.join(self.base_dir, 'tmp'),
                         '--lc-messages=C',
                         *self.postmaster_args.split())
            except Exception as exc:
                raise RuntimeError('failed to launch postmaster: %r' % exc)
        else:
            logger.close()

            self.pid = pid
            exec_at = datetime.now()
            while True:
                if os.waitpid(pid, os.WNOHANG) != (0, 0):
                    raise RuntimeError("*** failed to launch postmaster ***\n" + self.read_log())

                if re.search('accept connections', self.read_log()):
                    break

                if (datetime.now() - exec_at).seconds > 10.0:
                    raise RuntimeError("*** failed to launch postmaster (timeout) ***\n" + self.read_log())

                sleep(0.1)

            # create test database
            with psycopg2.connect(**self.dsn(dbname='template1')) as conn:
                conn.set_isolation_level(psycopg2.extensions.ISOLATION_LEVEL_AUTOCOMMIT)
                with conn.cursor() as cursor:
                    cursor.execute("SELECT COUNT(*) FROM pg_database WHERE datname='test'")
                    if cursor.fetchone()[0] <= 0:
                        cursor.execute('CREATE DATABASE test')

    def stop(self, _signal=signal.SIGTERM):
        import os
        if self.pid is None:
            return  # not started

        if self._owner_pid != os.getpid():
            return  # could not stop in child process

        try:
            os.kill(self.pid, _signal)
            killed_at = datetime.now()
            while (os.waitpid(self.pid, 0)):
                if (datetime.now() - killed_at).seconds > 10.0:
                    os.kill(self.pid, signal.SIGKILL)
                    raise RuntimeError("*** failed to shutdown postmaster (timeout) ***\n" + self.read_log())

                sleep(0.1)
        except:
            pass

        self.pid = None

    def cleanup(self):
        from shutil import rmtree
        if self._use_tmpdir and os.path.exists(self.base_dir):
            rmtree(self.base_dir)

    def read_log(self):
        try:
            with open(os.path.join(self.base_dir, 'tmp', 'postgresql.log')) as log:
                return log.read()
        except Exception as exc:
            raise RuntimeError("failed to open file:tmp/postgresql.log: %r" % exc)


def find_program(name, subdirs):
    path = get_path_of(name)
    if path:
        return path

    for base_dir in SEARCH_PATHS:
        for subdir in subdirs:
            path = os.path.join(base_dir, subdir, name)
            if os.path.exists(path):
                return path

    raise RuntimeError("command not found: %s" % name)


def get_path_of(name):
    path = subprocess.Popen(['which', name], stdout=subprocess.PIPE, stderr=subprocess.PIPE).communicate()[0]
    if path:
        return path.rstrip().decode('utf-8')
    else:
        return None


def get_unused_port():
    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    sock.bind(('localhost', 0))
    _, port = sock.getsockname()
    sock.close()

    return port
