
"""
Get data from three-quote, multi-line strings. Provides
typically-desired cleanups:
 * remove blank lines (default), but at least first and last blanks
   (which usually appear due to Python formatting)
 * remove common line prefix (default)
 * strip leading/trailing spaces (leading by request, trailing by default)
"""

import os

def lines(text, noblanks=True, dedent=True, lstrip=False, rstrip=True):
    """
    Grab lines from a string. First and last lines are assumed to be uninteresting if blank.
    :param text: text to be processed
    :param dedent: a common prefix should be stripped from each line (default `True`)
    :param noblanks: allow no blank lines at all (default `True`)
    :param lstrip: all left space be stripped from each line (default `False`);
        dedent and lstrip are mutualy exclusive
    :param rstrip: all right space be stripped from each line (default `True`)
    """
        
    textlines = text.expandtabs().splitlines()
    
    # remove blank lines if noblanks
    if noblanks:
        textlines = [ line for line in textlines if line.strip() != '' ]
    else:
        # even if intermediate blank lines ok, first and last are due to Python formatting
        if textlines and textlines[0].strip() == "":
            textlines.pop(0)
        if textlines and textlines[-1].strip() == "":
            textlines.pop()
    
    if dedent and not lstrip:
        prefix = os.path.commonprefix(textlines) # piggyback!
        if prefix:
            prelen = len(prefix)
            textlines = [ line[prelen:] for line in textlines ]

    # perform requested left and right space stripping (must be done
    # late so as to not interfere with dedent's common prefix detection)
    if lstrip and rstrip:
        textlines = [ line.strip() for line in textlines ]
    elif lstrip:
        textlines = [ line.lstrip() for line in textlines ]
    elif rstrip:
        textlines = [ line.rstrip() for line in textlines ]

    return textlines

def textlines(text, **kwargs):
    """
    Like ``lines()``, but returns result as unified text. Useful primarily because
    of the nice cleanups ``lines()`` does.
    """
    return "\n".join(lines(text, **kwargs))
