# vim: ts=8 sts=4 sw=4 si et
import unittest
from thebops.rexxbi import *

DEBUG = 1

class TestWords(unittest.TestCase):
    """
    Systematic test for the subword, delword, word, wordindex functions
    based on testdata generated by openObjectRexx
    """

    simpledata1 = (
            (('follow the white rabbit', 1,),
             ('follow the white rabbit',        # subword
              '',                               # delword
              'follow',                         # word
              1,                                # wordindex
              )),
            (('follow the white rabbit', 2,),
             ('the white rabbit',
              'follow ',
              'the',
              8,
              )),
            (('follow the white rabbit', 3,),
             ('white rabbit',
              'follow the ',
              'white',
              12,
              )),
            (('follow the white rabbit', 4,),
             ('rabbit',
              'follow the white ',
              'rabbit',
              18,
              )),
            (('follow the white rabbit', 5,),
             ('',
              'follow the white rabbit',
              '',
              0,
              )),
             )

    def test_subword1_simple(self):
        """\
        subword(s, start) yields expected results for simple input
        """
        for tup in self.simpledata1:
            args = tup[0]
            expected = tup[1][0]
            res = subword(*args)
            self.assertEqual(expected, res,
                             'subword%s => %r != %r'
                             % (args, res, expected))

    def test_delword1_simple(self):
        """\
        delword(s, start) yields expected results for simple input
        """
        for tup in self.simpledata1:
            args = tup[0]
            expected = tup[1][1]
            res = delword(*args)
            self.assertEqual(expected, res,
                             'delword%s => %r != %r'
                             % (args, res, expected))

    def test_word_simple(self):
        """\
        word(s, nr) yields expected results for simple input
        """
        for tup in self.simpledata1:
            args = tup[0]
            expected = tup[1][2]
            res = word(*args)
            self.assertEqual(expected, res,
                             'word%s => %r != %r'
                             % (args, res, expected))

    def test_wordindex_simple(self):
        """\
        wordindex(s, start) yields expected results for simple input
        """
        for tup in self.simpledata1:
            args = tup[0]
            expected = tup[1][3]
            res = wordindex(*args)
            self.assertEqual(expected, res,
                             'wordindex%s => %r != %r'
                             % (args, res, expected))

    simpledata2 = (
            (('follow the white rabbit', 1, 1),
             ('follow',
              'the white rabbit',
              )),
            (('follow the white rabbit', 1, 2),
             ('follow the',
              'white rabbit',
              )),
            (('follow the white rabbit', 1, 3),
             ('follow the white',
              'rabbit',
              )),
            (('follow the white rabbit', 1, 4),
             ('follow the white rabbit',
              '',
              )),
            (('follow the white rabbit', 1, 5),
             ('follow the white rabbit',
              '',
              )),
            (('follow the white rabbit', 2, 1),
             ('the',
              'follow white rabbit',
              )),
            (('follow the white rabbit', 2, 2),
             ('the white',
              'follow rabbit',
              )),
            (('follow the white rabbit', 2, 3),
             ('the white rabbit',
              'follow ',
              )),
            (('follow the white rabbit', 2, 4),
             ('the white rabbit',
              'follow ',
              )),
            (('follow the white rabbit', 2, 5),
             ('the white rabbit',
              'follow ',
              )),
            (('follow the white rabbit', 3, 1),
             ('white',
              'follow the rabbit',
              )),
            (('follow the white rabbit', 3, 2),
             ('white rabbit',
              'follow the ',
              )),
            (('follow the white rabbit', 3, 3),
             ('white rabbit',
              'follow the ',
              )),
            (('follow the white rabbit', 3, 4),
             ('white rabbit',
              'follow the ',
              )),
            (('follow the white rabbit', 3, 5),
             ('white rabbit',
              'follow the ',
              )),
            (('follow the white rabbit', 4, 1),
             ('rabbit',
              'follow the white ',
              )),
            (('follow the white rabbit', 4, 2),
             ('rabbit',
              'follow the white ',
              )),
            (('follow the white rabbit', 4, 3),
             ('rabbit',
              'follow the white ',
              )),
            (('follow the white rabbit', 4, 4),
             ('rabbit',
              'follow the white ',
              )),
            (('follow the white rabbit', 4, 5),
             ('rabbit',
              'follow the white ',
              )),
            (('follow the white rabbit', 5, 1),
             ('',
              'follow the white rabbit',
              )),
            (('follow the white rabbit', 5, 2),
             ('',
              'follow the white rabbit',
              )),
            (('follow the white rabbit', 5, 3),
             ('',
              'follow the white rabbit',
              )),
            (('follow the white rabbit', 5, 4),
             ('',
              'follow the white rabbit',
              )),
            (('follow the white rabbit', 5, 5),
             ('',
              'follow the white rabbit',
              )),
             )

    def test_subword2_simple(self):
        """\
        subword(s, start, count) yields expected results for simple input
        """
        for tup in self.simpledata2:
            args = tup[0]
            expected = tup[1][0]
            res = subword(*args)
            self.assertEqual(expected, res,
                             'subword%s => %r != %r'
                             % (args, res, expected))

    def test_delword_stripped(self):
        """\
        delword(s, start, count) yields expected results, leading/trailing whitespace stripped
        """
        def normalize(s):
            return s.strip()

        for tup in self.simpledata2:
            args = tup[0]
            expected = tup[1][1]
            res = normalize(delword(*args))
            self.assertEqual(normalize(expected), res,
                             'delword%s.strip() => %r != %r.strip()'
                             % (args, res, expected))

    def test_delword_whitespace(self):
        """\
        delword(s, start, count) yields expected results, whitespace ignored
        """
        def normalize(s):
            return ' '.join(s.split())

        for tup in self.simpledata2:
            args = tup[0]
            expected = tup[1][1]
            res = normalize(delword(*args))
            self.assertEqual(normalize(expected), res,
                             'normalize(delword%s) => %r != normalize(%r)'
                             % (args, res, expected))

    def _test_delword2_simple(self):
        """\
        delword(s, start, count) yields expected results for simple input
        """
        for tup in self.simpledata2:
            args = tup[0]
            expected = tup[1][1]
            res = delword(*args)
            self.assertEqual(expected, res,
                             'delword%s => %r != %r'
                             % (args, res, expected))

    testinput = '  follow   the white  rabbit   '
    testdata1 = (
                ((1,),
                 ('follow   the white  rabbit',     # subword
                  '  ',                             # delword
                  'follow',                         # word
                  3,                                # wordindex
                  )),
                ((2,),
                 ('the white  rabbit',
                  '  follow   ',
                  'the',
                  12,
                  )),
                ((3,),
                 ('white  rabbit',
                  '  follow   the ',
                  'white',
                  16,
                  )),
                ((4,),
                 ('rabbit',
                  '  follow   the white  ',
                  'rabbit',
                  23,
                  )),
                ((5,),
                 ('',
                  '  follow   the white  rabbit   ',
                  '',
                  0,
                  )),
            )

    def test_subword1(self):
        """\
        subword(s, start) yields expected results
        """
        s = self.testinput
        for tup in self.testdata1:
            args = (s,) + tup[0]
            expected = tup[1][0]
            res = subword(*args)
            self.assertEqual(expected, res,
                             'subword%s => %r != %r'
                             % (args, res, expected))

    def test_delword_unchanged(self):
        """\
        delword(s, start[, count]) returns input unchanged if start > words(s)
        """
        seq = (('follow the white rabbit', 5),
               ('follow the white rabbit', 5, 1),
               (' follow the white rabbit ', 5, 1),
               )
        for args in seq:
            expected = args[0]
            res = delword(*args)
            self.assertEqual(expected, res,
                             'delword%s => %r != %r'
                             % (args, res, expected))
            if args[2:] == (1,):
                import pdb
                pdb.set_trace()

    def _test_delword1(self):
        """\
        delword(s, start) yields expected results
        """
        s = self.testinput
        for tup in self.testdata1:
            args = (s,) + tup[0]
            expected = tup[1][1]
            res = delword(*args)
            self.assertEqual(expected, res,
                             'delword%s => %r != %r'
                             % (args, res, expected))

    def test_word(self):
        """\
        word(s, nr) yields expected results
        """
        s = self.testinput
        for tup in self.testdata1:
            args = (s,) + tup[0]
            expected = tup[1][2]
            res = word(*args)
            self.assertEqual(expected, res,
                             'word%s => %r != %r'
                             % (args, res, expected))

    def test_wordindex(self):
        """\
        wordindex(s, start) yields expected results
        """
        s = self.testinput
        for tup in self.testdata1:
            args = (s,) + tup[0]
            expected = tup[1][3]
            res = wordindex(*args)
            self.assertEqual(expected, res,
                             'wordindex%s => %r != %r'
                             % (args, res, expected))

    testdata2 = (
                ((1, 1),
                 ('follow',                     # subword
                  '  the white  rabbit   ',     # delword
                  )),
                ((1, 2),
                 ('follow   the',
                  '  white  rabbit   ',
                  )),
                ((1, 3),
                 ('follow   the white',
                  '  rabbit   ',
                  )),
                ((1, 4),
                 ('follow   the white  rabbit',
                  '  ',
                  )),
                ((1, 5),
                 ('follow   the white  rabbit',
                  '  ',
                  )),
                ((2, 1),
                 ('the',
                  '  follow   white  rabbit   ',
                  )),
                ((2, 2),
                 ('the white',
                  '  follow   rabbit   ',
                  )),
                ((2, 3),
                 ('the white  rabbit',
                  '  follow   ',
                  )),
                ((2, 4),
                 ('the white  rabbit',
                  '  follow   ',
                  )),
                ((2, 5),
                 ('the white  rabbit',
                  '  follow   ',
                  )),
                ((3, 1),
                 ('white',
                  '  follow   the rabbit   ',
                  )),
                ((3, 2),
                 ('white  rabbit',
                  '  follow   the ',
                  )),
                ((3, 3),
                 ('white  rabbit',
                  '  follow   the ',
                  )),
                ((3, 4),
                 ('white  rabbit',
                  '  follow   the ',
                  )),
                ((3, 5),
                 ('white  rabbit',
                  '  follow   the ',
                  )),
                ((4, 1),
                 ('rabbit',
                  '  follow   the white  ',
                  )),
                ((4, 2),
                 ('rabbit',
                  '  follow   the white  ',
                  )),
                ((4, 3),
                 ('rabbit',
                  '  follow   the white  ',
                  )),
                ((4, 4),
                 ('rabbit',
                  '  follow   the white  ',
                  )),
                ((4, 5),
                 ('rabbit',
                  '  follow   the white  ',
                  )),
                ((5, 1),
                 ('',
                  '  follow   the white  rabbit   ',
                  )),
                ((5, 2),
                 ('',
                  '  follow   the white  rabbit   ',
                  )),
                ((5, 3),
                 ('',
                  '  follow   the white  rabbit   ',
                  )),
                ((5, 4),
                 ('',
                  '  follow   the white  rabbit   ',
                  )),
                ((5, 5),
                 ('',
                  '  follow   the white  rabbit   ',
                  )),
                 )

    def test_subword2(self):
        """\
        subword(s, start, count) yields expected results
        """
        s = self.testinput
        for tup in self.testdata2:
            args = (s,) + tup[0]
            expected = tup[1][0]
            res = subword(*args)
            self.assertEqual(expected, res,
                             'subword%s => %r != %r'
                             % (args, res, expected))

    def _test_delword2(self):
        """\
        delword(s, start, count) yields expected results
        """
        s = self.testinput
        for tup in self.testdata2:
            args = (s,) + tup[0]
            expected = tup[1][1]
            res = delword(*args)
            self.assertEqual(expected, res,
                             'delword%s => %r != %r'
                             % (args, res, expected))


if __name__ == '__main__':
    unittest.main()
