# -*- coding: utf-8 -*-
'''
Copyright 2014 Yoshida Shin

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
'''


import threading

import error


class Future(object):
    """
    This class monitors associated callable progress and stores its return
    value or unhandled exception. Future.is_finished() returns whether the
    invoked callable is finished or not. Future.receive(timeout=None) blocks
    until timeout or invoked callable is finished and returns what the callable
    returns or raises its unhandled exception.

    The instance will be created by thread_utils.Pool.send method or callable
    decorated by thread_utils.async.
    """

    __slots__ = ('_result', '_is_error', '_is_finished', '__func', '__args',
                 '__kwargs')

    def __init__(self, func, *args, **kwargs):
        self._is_finished = threading.Event()
        self._is_finished.clear()
        self.__func = func
        self.__args = args
        self.__kwargs = kwargs

    def _run(self):
        try:
            self._result = self.__func(*self.__args, **self.__kwargs)
            self._is_error = False

        except BaseException as e:
            self._result = e
            self._is_error = True

        finally:
            self._is_finished.set()

    def is_finished(self):
        """
        Return True if invoked callable is finished. Otherwise, return False.
        """

        return self._is_finished.is_set()

    def receive(self, timeout=None):
        """
        Block until timeout or invoked callable is finished and returns what
        the callable returned or raises its unhandled exception.

        If the future object is generated by thread_utils.Pool.send method,
        and if the Pool instance is killed forcedly before the invoked task is
        started, this method raises DeadPoolError.

        When argument \`timeout\' is presend and is not None, it shoule be int
        or floating number. This method raises TimeoutError if task won't be
        finished before timeout.
        """

        self._is_finished.wait(timeout)

        if not self.is_finished():
            raise error.TimeoutError

        if self._is_error:
            raise self._result
        else:
            return self._result
