import os
from datetime import datetime
from urlparse import urljoin

from zope.interface import implements

from twisted.plugin import IPlugin
from twisted.internet import defer, threads
from twisted.python import log
from twisted.web.client import downloadPage

from imdb import IMDb

from apiserver.interfaces import IMetadata
from apiserver.db import metadata_db
from apiserver.metadata._simplemetadatahandler import SimpleMetadata
from apiserver import settings

class IMDBMetadata(SimpleMetadata):
    implements(IPlugin, IMetadata)
    currently_fetching = False
    id = 'imdb'
    
    trigger = 'plugin:imdb:id'
    
    def __init__(self):
        self._imdb = IMDb()
    
    @defer.inlineCallbacks
    def fetch_metadata(self, obj_id):
        imdb_info = yield threads.deferToThread(self._imdb.get_movie, str(obj_id))
        
        image_file = yield self.download_cover(obj_id, imdb_info.get('full-size cover url', None))

        movie = {
            'last_updated': datetime.now(),
            'title': imdb_info.get('title', None),
            'rating': imdb_info.get('rating', None),
            'votes': imdb_info.get('votes', None),
            'plot': imdb_info.get('plot', [''])[0].split('::')[0] or imdb_info.get('plot outline', ''),
            'year': imdb_info.get('year', None),
            'duration': imdb_info.get('runtimes', [''])[0].split('::')[0],
            'cover': image_file,
            'languages': imdb_info.get('languages', []),
            'genres': imdb_info.get('genres', []),
            'actors': [],
            'director': [],
            'writer': [],
        }
        
        for role in ['actors', 'director', 'writer']:
            movie[role] = [(int(person.personID.lstrip('0')), person['name']) for person in imdb_info.get(role, [])]
        
        defer.returnValue(movie)

    def actual_handle_item(self, user, item, raw_item, obj):
        for k in ['title', 'rating', 'votes', 'plot', 'year', 'duration', 'languages', 'genres']:
            if not obj.get(k, None):
                continue
            
            item[k] = obj[k]
        
        if obj['cover']:
            item['cover'] = urljoin(settings.BASE_URL, 'files/imdb/%s' % obj['cover'])
        
        for k in ['writer', 'director', 'actors']:
            item[k] = [actor[1] for actor in obj[k]]
        
        return item

imdbmetadata = IMDBMetadata()