import os
from copy import copy
from zope.interface import implements

import yaml

from twisted.application import internet
from twisted.python import usage, log
from twisted.plugin import IPlugin
from twisted.application.service import IServiceMaker
from twisted.web import server, guard, resource
from twisted.cred.checkers import FilePasswordDB
from twisted.cred.portal import IRealm, Portal
from twisted.plugin import getPlugins


class ConfigNotFoundException(Exception):
    """Configfile was not found"""

class ServerContextFactory:
    def __init__(self, cert_file):
        self._cert_file = cert_file
    
    def getContext(self):
        from OpenSSL import SSL
        ctx = SSL.Context(SSL.SSLv23_METHOD)
        ctx.use_certificate_file(self._cert_file)
        ctx.use_privatekey_file(self._cert_file)
        return ctx

class TidalStreamRealm(object):
    implements(IRealm)
    
    def __init__(self, tidalapiserver):
        self.tidalapiserver = tidalapiserver

    def requestAvatar(self, avatarId, mind, *interfaces):
        if resource.IResource in interfaces:
            return resource.IResource, self.tidalapiserver, lambda: None
        raise NotImplementedError()

class Options(usage.Options):
    optParameters = [['config', 'c', 'apiserver.yml', "Path to config file"]]

class TidalAPIServiceMaker(object):
    implements(IServiceMaker, IPlugin)
    tapname = "tidalapi"
    description = "Codename TidalStream API Server."
    options = Options

    def makeService(self, options):
        if not os.path.isfile(options['config']):
            raise ConfigNotFoundException()
        
        config = yaml.load(open(options['config']))
        
        use_ssl = config['usessl']
        
        from apiserver import settings
        settings.BASE_URL = '%s://%s:%s' % (use_ssl and 'https' or 'http', config['external_ip'], config['port'])
        
        settings.METADATA_FILE = config['metadata_database']
        settings.DB_FILE = config['database']
        if 'datapath' in config:
            if os.path.isdir(config['datapath']):
                settings.DATA_PATH = config['datapath']
            else:
                log.err("Datapath '%s' does not exist" % config['datapath'])
        
        from apiserver.interfaces import IBackend, ISection, IService
        
        import apiserver.backends
        backend_types = dict((p.name, p) for p in getPlugins(IBackend, apiserver.backends))
        
        import apiserver.sections
        section_types = dict((p.name, p) for p in getPlugins(ISection, apiserver.sections))
        
        import apiserver.services
        settings.SERVICES = list(getPlugins(IService, apiserver.services))
        
        known_nodes = {}
        for name, node in config['nodes'].iteritems():
            if node['backend'] not in backend_types:
                log.err("Unknown backend %s for node %s" % (node['backend'], name))
                continue
            node['type'] = backend_types[node['backend']]
            node['id'] = name
            known_nodes[name] = node
        
        for name, cfg in config['sections'].items():
            if 'nodes' not in cfg:
                log.err("No nodes found for section %s" % name)
                continue
            
            if 'type' not in cfg:
                log.err("Type missing for section %s" % name)
                continue
            
            if cfg['type'] not in section_types:
                log.err("Unknown section type %s for node %s" % (cfg['type'], name))
                continue
            
            section_type = section_types[cfg['type']]
            
            section = settings.SECTIONS[name] = section_type(name, cfg)
            
            for node_name, section_cfg in cfg['nodes'].items():
                if node_name not in known_nodes:
                    log.err("Unknown node %s for setion %s" % (node_name, name))
                    continue
                
                node_cfg = known_nodes[node_name]
                node_type = node_cfg['type']
                node = node_type(section, node_cfg, section_cfg)
                section.nodes.append(node)
        
        if 'plugins' in config:
            settings.PLUGINS.update(config['plugins'])
        
        from apiserver.resource import TidalStreamHTTPAuthSessionWrapper
        from apiserver.server import RootResource
        from apiserver.tokenauth import InternalTokenAuthFactory, TokenPasswordChecker
        
        settings.PASSWORD_CHECKER = password_checker = FilePasswordDB(config['userfile'])
        settings.TOKEN_PASSWORD_CHECKER = token_password_checker = TokenPasswordChecker()
        checkers = [password_checker, token_password_checker]
        
        wrapper = TidalStreamHTTPAuthSessionWrapper(
            Portal(TidalStreamRealm(RootResource()),
                   checkers), [guard.BasicCredentialFactory('TidalStream'), InternalTokenAuthFactory()])
        
        site = server.Site(wrapper)
        
        if use_ssl:
            from OpenSSL import SSL
            return internet.SSLServer(config['port'], site,
                                             ServerContextFactory(config['certfile']))
        else:
            return internet.TCPServer(config['port'], site)

serviceMaker = TidalAPIServiceMaker()
