import os
import urllib
import types
import base64
import json
from fnmatch import fnmatch
from datetime import datetime, timedelta

from zope.interface import implements
from twisted.plugin import IPlugin
from twisted.web import static, server, resource
from twisted.python import randbytes

from sdfs.interfaces import IURIHandler, ICrontab
from sdfs.exceptions import FileOrFolderNotFoundException
from sdfs.resource import PluginResource
from sdfs.site import predict_storage_path
from sdfs.fs import TEMP_FOLDER
from sdfs.utils import get_free_space

RETRIEVE = 0 # download a file
STORE = 1 # save files to a given folder

URL_LIFETIME = timedelta(days=1)


class FilelikeObjectResource(static.File):
    isLeaf = True
    contentType = None
    fileObject = None
    encoding = 'bytes'
    
    def __init__(self, fileObject, size, contentType='bytes'):
        self.contentType = contentType
        self.fileObject = fileObject
        self.fileSize = size
        resource.Resource.__init__(self)
    
    def _setContentHeaders(self, request, size=None):
        if size is None:
            size = self.getFileSize()
        
        if size:
            request.setHeader('content-length', str(size))
        if self.type:
            request.setHeader('content-type', self.type)
        if self.encoding:
            request.setHeader('content-encoding', self.encoding)
    
    def getFileSize(self):
        return self.fileSize
    
    def render_GET(self, request):
        """
        Begin sending the contents of this L{File} (or a subset of the
        contents, based on the 'range' header) to the given request.
        """
        request.setHeader('accept-ranges', 'bytes')

        producer = self.makeProducer(request, self.fileObject)

        if request.method == 'HEAD':
            return ''
        
        producer.start()
        # and make sure the connection doesn't get closed
        return server.NOT_DONE_YET
    render_HEAD = render_GET


class FileStoredOk(resource.Resource):
    isLeaf = True
    def render_POST(self, request):
        return json.dumps({
            'status': 'success',
            'message': 'The file has been saved successfully'
        })

class HTTPURIHandler(PluginResource):
    implements(IPlugin, IURIHandler, ICrontab)
    isLeaf = False
    
    name = 'http'
    scheme = 'http'
    ip = None
    port = None
    
    def serve_file(self, f):
        if isinstance(f['file'], types.StringTypes):
            f = static.File(f['file'], defaultType='application/octet-stream')
            f.isLeaf = True
        else:
            obj, size = f['file'].get_object()
            f = FilelikeObjectResource(obj, size)
        
        return f
    
    def store_file(self, request, f):
        filepath = os.path.abspath(os.path.join(f['path'], '/'.join(self.get_path(request).split('/')[1:])))
        if not filepath.startswith(f['path']):
            return resource.ErrorPage(500, 'Something messed with path', 'Seems like the actual path was tampered with')
        
        if os.path.isfile(filepath):
            return resource.ErrorPage(500, 'Already exist', 'The file already exists, not allowing overwrite')
        
        file2move = request.content.name
        request.content.close()
        
        path = os.path.split(filepath)[0]
        if not os.path.isdir(path):
            os.makedirs(path)
        
        os.rename(file2move, filepath)
        
        age = None
        try:
            age = int(request.args.get('age', ['No!'])[0])
        except ValueError:
            pass
        self.filesystem.add_file(filepath, age)
        return FileStoredOk()
    
    def generate_secure_token(self):
        return base64.urlsafe_b64encode(randbytes.RandomFactory().secureRandom(21, True))
    
    def prepare_serve(self, f, ip=None):
        id = self.generate_secure_token()
        self.filelist[id] = {
            'method': RETRIEVE,
            'expiration': datetime.now() + URL_LIFETIME,
            'ip': ip,
            'file': f
        }
        
        if isinstance(f, types.StringTypes):
            filename = os.path.split(f)[1]
        else:
            filename = f.filename
        
        return 'http%s://%s:%s/%s/%s/%s' % ((self.configfile.getboolean('general', 'usessl') and 's' or ''), self.ip, self.port, self.scheme, id, urllib.quote_plus(filename))
    
    def prepare_store(self, path, save_path, ip=None):
        id = self.generate_secure_token()
        self.filelist[id] = {
            'method': STORE,
            'expiration': datetime.now() + URL_LIFETIME,
            'ip': ip,
            'path': save_path
        }
        predict_storage_path[id] = path
        
        return 'http%s://%s:%s/%s/%s/' % ((self.configfile.getboolean('general', 'usessl') and 's' or ''), self.ip, self.port, self.scheme, id)
    
    def is_allowed_store_path(self, path):
        for p in self.allowed_store_paths:
            if fnmatch(path, p):
                return True
        
        return False
    
    def getChild(self, path, request):
        if path in self.filelist:
            if self.filelist[path]['ip'] and self.filelist[path]['ip'] != request.getClientIP():
                return resource.NoResource()
            
            if request.method in ['GET', 'HEAD'] and self.filelist[path]['method'] == RETRIEVE:
                return self.serve_file(self.filelist[path])
            
            if request.method == 'POST' and self.filelist[path]['method'] == STORE:
                return self.store_file(request, self.filelist[path])
        
        return self
    
    def render_GET(self, request):
        path = self.get_path(request)
        f = self.filesystem.get_file(path)
        if f is None:
            raise FileOrFolderNotFoundException("'%s' not found" % path)
        
        if 'ip' in request.args:
            ip = request.args['ip'][0]
        else:
            ip = None
        
        return {
            'status': 'success',
            'message': 'serving url',
            'url': self.prepare_serve(f, ip)
        }
    
    def render_POST(self, request): # store files to a given folder, avoids fragmentation
        if not self.allowed_store:
            return {
                'status': 'error',
                'message': 'storage not allowed',
            }

        
        path = '%s/' % self.get_path(request).rstrip('/')
        
        size = 0
        if 'size' in request.args:
            try:
                size = request.args['size'][0]
            except ValueError:
                pass
        
        usable_paths = []
        for p, prefix in self.paths:
            if not path.startswith('%s/' % prefix) or not self.is_allowed_store_path(p):
                continue
            
            if get_free_space(p) + 10*1024*1024 < size: # must be at least 10mb more space than needed
                continue
            
            save_path = os.path.join(p, path[len(prefix)+1:])
            check_path = os.path.join(p, path[len(prefix)+1:])
            depth = 0
            while check_path:
                if os.path.isdir(check_path):
                    break
                
                check_path = os.path.split(check_path)[0]
                depth += 1
            
            usable_paths.append((depth, p, save_path))
        
        if not usable_paths:
            return {
                'status': 'error',
                'message': 'failed to find proper place with sorte rights and sufficient disk space'
            }
        
        _, store_path, save_path = sorted(usable_paths)[0]
        store_path = os.path.join(store_path, TEMP_FOLDER)
        
        if 'ip' in request.args:
            ip = request.args['ip'][0]
        else:
            ip = None
        
        if 'ip' in request.args:
            ip = request.args['ip'][0]
        else:
            ip = None
        
        return {
            'status': 'success',
            'message': 'serving url folder',
            'url': self.prepare_store(store_path, save_path, ip),
        }
    
    def cleanup_old_urls(self):
        for k, v in self.filelist.items():
            if v['expiration'] < datetime.now():
                del self.filelist[k]
                if k in predict_storage_path:
                    del predict_storage_path[k]
    
    def schedule(self):
        return '* * * * 0', self.cleanup_old_urls
    
    def initialize(self):
        self.filelist = {}
        self.ip = self.configfile.get('general', 'externalip')
        self.port = self.configfile.getint('general', 'controlport')
        self.allowed_store = self.configfile.has_option(self.config_section, 'allow_store') and self.configfile.getboolean(self.config_section, 'allow_store') or False
        
        if self.allowed_store:
            self.allowed_store_paths = []
    
            i = 1
            while self.configfile.has_option(self.config_section, 'storepath%s' % i):
                self.allowed_store_paths.append(self.configfile.get(self.config_section, 'storepath%s' % i).rstrip('/'))
                i += 1
    
httpurihandler = HTTPURIHandler()