import json
import urllib

from twisted.web import resource, server, guard
from twisted.web._auth.wrapper import UnauthorizedResource
from twisted.web.resource import IResource
from twisted.internet import defer
from twisted.python import log

from sdfs.utils import parse_accept_header
from sdfs import portal

class HTTPAuthSessionWrapper(guard.HTTPAuthSessionWrapper):
    def __init__(self, portal, credentialFactories, cb_defer):
        guard.HTTPAuthSessionWrapper.__init__(self, portal, credentialFactories)
        self.cb_defer = cb_defer
        
    def _login(self, credentials):
        d = self._portal.login(credentials, None, IResource)
        d.addCallbacks(self.cb_defer.callback, self.cb_defer.errback)
        return d

class PluginResource(resource.Resource):
    isLeaf = True
    
    def get_path(self, request):
        return urllib.unquote_plus('/'.join(request.path.split('/')[2:]))
    
    def serialize_json(self, request, data):
        request.setHeader('content-type', 'application/vnd.tidalstream+json')
        return json.dumps(data)
    
    def serialize(self, request, data):
        serializer = self.serialize_json
        if request.requestHeaders.hasHeader('accept'):
            accept_headers = parse_accept_header(request.requestHeaders.getRawHeaders('accept')[0])
            for header, _ in accept_headers:
                serialize_type = header.split('/')[1]
                if hasattr(self, 'serialize_%s' % serialize_type):
                    serializer = getattr(self, 'serialize_%s' % serialize_type)
                    break
        
        return serializer(request, data)
    
    def render(self, request):
        """
        Adds support for deferred render methods
        """
        
        def login_successfully(ignored):
            m = getattr(self, 'render_' + request.method, None)
            if not m:
                # This needs to be here until the deprecated subclasses of the
                # below three error resources in twisted.web.error are removed.
                from twisted.web.error import UnsupportedMethod
                allowedMethods = (getattr(self, 'allowedMethods', 0) or
                                  resource._computeAllowedMethods(self))
                raise UnsupportedMethod(allowedMethods)
            
            result = defer.maybeDeferred(m, request)
            
            def write_rest(defer_result, request):
                if defer_result:
                    request.write(self.serialize(request, defer_result))
                request.finish()
            
            def err_rest(defer_result=None):
                log.err('Error while parsing request: %s' % defer_result)
                request.finish()
            
            result.addCallback(write_rest, request)
            result.addErrback(err_rest)
        
        def login_failed(ignored):
            UnauthorizedResource(portal.CHECKERS).render(request)
            request.finish()
        
        d = defer.Deferred()
        w = HTTPAuthSessionWrapper(portal.PORTAL, portal.CHECKERS, d)
        w._authorizedResource(request)
        d.addCallbacks(login_successfully, login_failed)
        
        return server.NOT_DONE_YET