"""
Momentet
"""

import os
import glob
from datetime import datetime

from twisted.web import resource
from twisted.python import log
from twisted.plugin import getPlugins
from twisted.internet import reactor

from txscheduling.task import ScheduledCall
from txscheduling.cron import CronSchedule

import sdfs.plugins
from sdfs.interfaces import IURIHandler, ISDFSPlugin, ICrontab, IMetadata
from sdfs.fs import FileSystem
from sdfs.exceptions import UnknownSchemeException
from sdfs.resource import PluginResource


class SDFSServer(object):
    """General outward facing filesharing API"""
    
    plugins = None
    filesystem = None
    configfile = None
    scanning = False
    last_scan = None
    scan_defer = None
    
    def __init__(self, configfile):
        self.configfile = configfile
        
        self.paths = paths = []
        i = 1
        while self.configfile.has_option('disks', 'disk%s' % i):
            prefix = self.configfile.get('disks', 'prefix%s' % i)
            for path in glob.glob(self.configfile.get('disks', 'disk%s' % i)):
                paths.append((path.rstrip(os.sep), prefix.strip('/')))
            i += 1
        
        reactor.suggestThreadPoolSize(len(paths) + 2)
        self.filesystem = FileSystem(paths, self.configfile.get('general', 'dbfile'),
                                     list(getPlugins(IMetadata, sdfs.plugins)))
        
        if not self.plugins:
            self.plugins = list(getPlugins(IURIHandler, sdfs.plugins))
            for plugin in self.plugins:
                log.msg('Found plugin for scheme %s' % plugin.scheme)
        
        self.initialize_all_plugins()
        
        self.rescan()
        
    def initialize_all_plugins(self):
        for plugin in list(getPlugins(ISDFSPlugin, sdfs.plugins)):
            log.msg('Found plugin %s, initializing' % plugin.name)
            self.initialize_plugin(plugin)
    
    def shutdown(self):
        self.uninitialize_plugins()
        return self.filesystem.close()
    
    def rescan(self):
        """
        Rescans the whole filesystem, looking from changes compared
        to how the database currently matches the actual filesystem.
        
        Returns a deferred, fired when scan is done.
        """
        
        self.scanning = True
        def done_scanning(ignored):
            self.scanning = False
            self.last_scan = datetime.now()
        
        self.scan_defer = self.filesystem.rescan().addCallback(done_scanning)
    
    def server_status(self):
        """
        Returns current server status which is only affected by
        an ongoing scan.
        """
        if self.scanning:
            status = 'scanning'
        else:
            status = 'idle'
        
        return {
            'date': datetime.now().isoformat(),
            'status': status
        }
    
    def uninitialize_plugins(self):
        for plugin in list(getPlugins(ISDFSPlugin, sdfs.plugins)):
            if hasattr(plugin, 'is_initialized'):
                delattr(plugin, 'is_initialized')
    
    def initialize_plugin(self, plugin):
        if not hasattr(plugin, 'is_initialized'):
            setattr(plugin, 'filesystem', self.filesystem)
            setattr(plugin, 'is_initialized', True)
            setattr(plugin, 'paths', self.paths)
            setattr(plugin, 'configfile', self.configfile)
            setattr(plugin, 'config_section', 'plugin-%s' %
                plugin.name)
            if hasattr(plugin, 'initialize'):
                plugin.initialize()
            
            if ICrontab.providedBy(plugin):
                result = plugin.schedule()
                if result:
                    crontab_syntax, f = result
                    crontab = ScheduledCall(f)
                    crontab.start(CronSchedule(crontab_syntax))
                    
    
    def get_plugin(self, scheme):
        """
        Takes a protocol scheme and returns the plugin matching it.
        The function ensures that the plugin is initalized.
        """
        for plugin in self.plugins:
            if plugin.scheme == scheme:
                self.initialize_plugin(plugin)
                return plugin
        
        raise UnknownSchemeException()
    
class SDFSWebServer(PluginResource):
    isLeaf = False
    
    def __init__(self, sdfsserver):
        self.sdfsserver = sdfsserver
        PluginResource.__init__(self)
    
    def serialize_html(self, request, data):
        request.setHeader('content-type', 'text/html')
        return data
    
    def render_GET(self, request):
        return 'Welcome to the main menu!'
    
    def getChild(self, path, request):
        if not path:
            return self
        
        try:
            p = self.sdfsserver.get_plugin(path)
        except UnknownSchemeException:
            return resource.NoResource('Unknown plugin %s' % path)
        
        return p