import os
import platform
import hashlib
import ctypes

__all__ = [
    'cleanup_path',
    'hash_path',
    'keyify',
    'parse_accept_header',
    'merge_set_dicts',
    'get_free_space',
]


def cleanup_path(path):
    if type(path) is list:
        return [cleanup_path(p) for p in path]
    else:
        return path.strip('/')

def hash_path(path):
    return hashlib.sha1(path).digest()

def keyify(key_type, path):
    return '%s%s' % (key_type, hash_path(cleanup_path(path)))

def parse_accept_header(accept_header):
    accept_types = []
    for h in accept_header.split(','):
        h = h.split(';')
        ch = h[0]
        quality = 1
        if len(h) > 1:
            h = h[1].split('=')
            if h[0].strip() == 'q':
                quality = max(min(float(h[1]), 1), 0)
        accept_types.append((ch.strip(), quality))
    return sorted(accept_types, key=lambda x:x[1], reverse=True)

def merge_set_dicts(result, merge_with):
    for k, v in merge_with.iteritems():
        result[k] |= v

# thanks to http://stackoverflow.com/questions/51658/cross-platform-space-remaining-on-volume-using-python
def get_free_space(folder):
    """Return folder/drive free space (in bytes)"""
    if platform.system() == 'Windows':
        free_bytes = ctypes.c_ulonglong(0)
        ctypes.windll.kernel32.GetDiskFreeSpaceExW(ctypes.c_wchar_p(folder), None, None, ctypes.pointer(free_bytes))
        return free_bytes.value
    else:
        return os.statvfs(folder).f_bfree * os.statvfs(folder).f_frsize