import ConfigParser
import signal
import os
from zope.interface import implements

from sdfs import portal

from twisted.application import internet
from twisted.cred import credentials
from twisted.python import usage
from twisted.plugin import IPlugin
from twisted.application.service import IServiceMaker, MultiService, Service
from twisted.web import server, guard, resource
from twisted.internet import protocol, reactor
from twisted.cred.checkers import FilePasswordDB
from twisted.cred.portal import IRealm, Portal

class ServerContextFactory:
    def __init__(self, cert_file):
        self._cert_file = cert_file
    
    def getContext(self):
        from OpenSSL import SSL
        ctx = SSL.Context(SSL.SSLv23_METHOD)
        ctx.use_certificate_file(self._cert_file)
        ctx.use_privatekey_file(self._cert_file)
        return ctx

class Options(usage.Options):
    optParameters = [['config', 'c', 'sdfs.conf', "Path to config file"]]

class KillerService(Service):
    def __init__(self, sdfsserver):
        self.sdfsserver = sdfsserver
        
    def stopService(self):
        self.running = 0
        return self.sdfsserver.shutdown()

class SDFSServiceMaker(object):
    implements(IServiceMaker, IPlugin)
    tapname = "sdfs"
    description = "SDFS instance."
    options = Options

    def makeService(self, options):
        if not os.path.isfile(options['config']):
            from sdfs.exceptions import ConfigNotFoundException
            raise ConfigNotFoundException('No configuration file found at %s' % options['config'])
        
        config = ConfigParser.ConfigParser()
        config.read(options['config'])
        
        from sdfs.server import SDFSServer, SDFSWebServer
        ms = MultiService()
        
        sdfsserver = self.sdfsserver = SDFSServer(configfile=config)
        
        checkers = [FilePasswordDB(config.get('general', 'userfile'))]
        portal.PORTAL = Portal(portal.SDFSRealm(), checkers)
        portal.CHECKERS = [guard.BasicCredentialFactory('SDFS')]
        
        from sdfs.site import Site
        site = Site(SDFSWebServer(sdfsserver))
        
        sdfsport = config.getint('general', 'controlport')
        if config.getboolean('general', 'usessl'):
            from OpenSSL import SSL
            ms.addService(internet.SSLServer(sdfsport, site,
                                             ServerContextFactory(config.get('general', 'certfile'))))
        else:
            ms.addService(internet.TCPServer(sdfsport, site))
        
        ms.addService(KillerService(sdfsserver))
        reactor.callLater(0.0, sdfsserver.initialize_all_plugins)
        return ms

serviceMaker = SDFSServiceMaker()
