"""
wikklytext.wiki.layout.py: Wiki page layout.

Copyright (C) 2007,2008 Frank McIngvale

Contact: fmcingvale@gmail.com

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
"""
from wikklytext.base import ifelse
from wikklytext.wiki.render import render_text_inner_html, get_item_skiplist
from urllib import urlencode
from wikklytext import WikklyItem
from wikklytext.port import *
import defcontent
from wikklytext.wiki.util import itemText
from wikklytext.buildHTML import escapeURL

def safe_nowiki(text):
	"""
	Make 'text' safe to place inside <nowiki>.
	
	Usage:
		s = '<nowiki>%s</nowiki>' % safe_nowiki(text)
	"""
	if text is None:
		return 'None'
		
	while 1:
		text2 = text.replace('</nowiki>', '')
		if len(text2) == len(text):
			return text2
			
		text = text2
	
def safe_code(text):
	"""
	Make 'text' safe to place inside {{{...}}} or {{class{...}}}
	
	Usage:
		s = '{{{%s}}}' % safe_code(text)
	"""
	if text is None:
		return 'None'
	
	while 1:
		text2 = text.replace('}}}', '}}&#x200b;}')
		if len(text2) == len(text):
			return text2
			
		text = text2
		
def safehtml(text):
	"""
	Escape any tag-like things in text.
	"""
	return text.replace('&','&amp;').replace('<','&lt;').replace('>','&gt;')

def utf8(s):
	"If s is unicode, return as UTF-8. Else return as-is."
	if isinstance(s, unicode):
		return s.encode('utf-8')
	else:
		return s
		
from util import content_type_for_xhtml

def HEAD(encoding='utf-8', title=None, stylesheet='css/wikklytext.css',
			styletext=u'', pre_head=u'', post_head=u'', baseurl=u'', sitename=u'',
			reqheaders={}):
	head = u''
	#head += u"""<?xml version="1.0" encoding="UTF-8"?>\n"""
	head += u"""<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">
<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en" lang="en">
<head>
%s
<!-- Generated by WikklyText, http://wikklytext.com -->
<meta http-equiv="Content-Type" content="%s; charset=%s"/>
""" % (pre_head, content_type_for_xhtml(reqheaders), encoding)

	# add stylesheet
	head += u'<link href="%s" rel="stylesheet" type="text/css"/>' % \
			ifelse(len(baseurl), '%s/%s' % (baseurl,stylesheet), stylesheet)
	
	# add extra styling
	head += u'<style type="text/css">%s</style>' % styletext
	
	# add favicon
	head += u'<link rel="shortcut icon" href="%sfavicon.png" type="image/png"/>' % \
			ifelse(len(baseurl), '%s/' % baseurl, '')
	head += u'<link rel="icon" href="%sfavicon.png" type="image/png"/>' % \
			ifelse(len(baseurl), '%s/' % baseurl, '')

	# add RSS feed pointer
	head += '<link rel="alternate" type="application/rss+xml" title="%s RSS" href="%srss.xml" />' % \
			(sitename, ifelse(len(baseurl), '%s/' % baseurl, ''))
	
	# <title> required, even if empty, for XHTML
	head += u'<title>%s</title>' % (title or '')
		
	head += post_head
	head += u'</head>'
	return head.encode(encoding)

def render(wiki, text, UID):
	"Convenience"
	do_cache = wiki.cacheable()
	return render_text_inner_html(wiki, text, UID=UID)
	
def escape_camelword(word):
	"If word is a CamelWord, add a ~ to it."
	from wikklytext.wikwords import is_camelword
	
	if is_camelword(word):
		return '~'+word
	else:
		return word

def pageTitle(wiki):
	from wikklytext.wiki.core import SiteTitle, SiteSubtitle
	
	h = u'<span class="wikkly-site-title">%s</span> <span class="wikkly-site-subtitle">%s</span>' % \
				(unicode(SiteTitle(wiki), 'utf-8'), unicode(SiteSubtitle(wiki), 'utf-8'))
	return h

def itemTitle(wiki, itemname, can_edit=True):
	p = {'cmd': 'editItem', 'name': utf8(itemname)}
	
	# if itemname is a UID, translate to real username
	title = wiki.user_get_username(itemname)
	
	h = u'{{item-title{%s}}}' % safe_code(escape_camelword(title))
	if can_edit:
		h += u' [[edit|DoServerCmd?%s]]' % urlencode(p)
		
	h += u'<br>'
	return render(wiki, h, UID=None)
	
def authorTagline(wiki, item):
	from wikklytext.store.wikStore import makeFSname
	
	h = u''
	
	# item.author may be a UID, so try to lookup real name
	# (not an error to not be a UID here)
	name = wiki.user_get_username(item.author)
	
	# add modifier (with link if applicable)
	if wiki.getitem(item.author): # still want to lookup item by UID!
		h += u'[[%s|%s]]' % (name, item.author)
	else:
		h += u"%s" % name		

	# add dates
	h += u'~~{{grayout{, %s (created %s)}}}~~<br>' % \
			(item.mtime.to_display(), item.ctime.to_display())
	
	return render(wiki, h, UID=None)
	
def tagline(wiki, item):
	if not len(item.tag_list()):
		return ''
		
	h = u"''Tags:'' "
	for tag in item.tag_list():
		h += u'[[%s|index-Tags#TAG_%s]] ' % (tag,tag)
		
	h += '<br>'
	
	return render(wiki, h, UID=None)
		
def tiddler_sort_desc_mtime(a, b):
	# sort by mtime, descending
	return cmp(b.mtime, a.mtime)

def makeNameIndex(wiki):
	"""
	Sort all items by name and create name index (like 'All' tab)
	as wikklytext.
	"""
	rlist = []
	skiplist = set_(get_item_skiplist(wiki))
	
	for item in wiki.getall():		
		# omit the pseudo page 'DoServerCmd' from index
		if item.name in skiplist or item.name == 'DoServerCmd':
			continue
			
		# add link (item.name may be UID, so lookup real name)
		name = wiki.user_get_username(item.name)
		rlist.append((item.name, u'<<indent <quote>[[%s|%s]]</quote>>>\n' % \
									(name, item.name)))
	
	rlist.sort(lambda a,b: cmp(a[0].lower(),b[0].lower()))
	txt_index = u''.join([b for a,b in rlist])
	return txt_index
	
def makeTimelineIndex(wiki):
	"""
	Sort all items by mtime and create Timeline index (like 'Timeline' tab)
	as wikklytext.
	"""
	allitems = wiki.getall()
	allitems.sort(tiddler_sort_desc_mtime)
	
	skiplist = set_(get_item_skiplist(wiki))
	
	txt_index = u''
	
	cur_YMD = None
	
	for item in allitems:
		# omit the pseudo page 'DoServerCmd' from index
		if item.name in skiplist or item.name == 'DoServerCmd':
			continue
			
		# when Y-M-D changes, write it
		if cur_YMD != item.mtime.to_YMD():
			txt_index += item.mtime.to_display()
			cur_YMD = item.mtime.to_YMD()
			
		# add link (item.name may be UID, so lookup real name)
		name = wiki.user_get_username(item.name)
		txt_index += u'<<indent <quote>[[%s|%s]]</quote>>>\n' % \
				(name, item.name)

	return txt_index
	
def tiddlers_by_tags(wiki):
	"""
	Return a map of tags to WikklyItems where:
		map[tag] = list of WikklyItems with tag
	"""
	tagmap = {}
	for item in wiki.getall():
		for tag in item.tag_list():
			if tagmap.has_key(tag):
				tagmap[tag].append(item)
			else:
				tagmap[tag] = [item]
				
	return tagmap

def makeTagIndex(wiki):
	"""
	Make index by tags (like "Tags" tab) as wikklytext.
	"""
	tagmap = tiddlers_by_tags(wiki)
	
	skiplist = set_(get_item_skiplist(wiki))

	txt_index = u''
		
	tags = list(tagmap.keys())
	tags.sort()
	
	# make index at top
	for tag in tags:
		txt_index += '* [[%s|#TAG_%s]] (%d)\n' % (tag, tag, len(tagmap[tag]))
		
	# now list each item by tag
	for tag in tags:
		txt_index += '!!![[#TAG_%s]] %s\n' % (tag, tag)
		for item in tagmap[tag]:
			if item.name in skiplist:
				continue
				
			# item.name may be a UID, so get displayed name
			name = wiki.user_get_username(item.name)
			txt_index += '* [[%s|%s]]\n' % (name, item.name)
	
	return txt_index
	
def pageHeader(wiki, focus, itemname=None, loginUID=None):
	"Return page header as UTF-8"
	from wikklytext.wiki.core import SiteTitleText
	site = SiteTitleText(wiki)
		
	if itemname is None:
		title = site
	else:
		title = site + ': ' + itemname
		
	# get styles defined by plugins
	plugins = wiki.load_plugins()
	styletext = plugins.css
	
	# get user-defined styles from wiki (these override
	# any defined in the plugins)
	styletext += itemText(wiki, 'StyleSheet')
		
	# load pre/post HEAD and pre BODY markup
	preHead = itemText(wiki, 'MarkupPreHead')
	postHead = itemText(wiki, 'MarkupPostHead')
		
	# generate as UTF-8
	h = ''
	h += HEAD(styletext=styletext, title=title, pre_head=preHead, post_head=postHead, 
				baseurl=wiki.getRT_baseurl(), sitename=site, reqheaders=wiki.getRT_wsgienv())
	h += '<body class="wikkly-body">'
	h += '%s' % utf8(itemText(wiki, 'MarkupPreBody'))
	
	# add spacer at top
	h += '<div class="wikkly-top-spacer"></div>'
	
	# create outer container <div>
	h += '<div class="wikkly-outside-container">'
				
	# add title <div>
	h += '<div class="wikkly-title-container">'
	h += utf8(pageTitle(wiki))
	h += '</div>'
	
	# add navmenu <div>
	h += '<div class="wikkly-navmenu-container">'
	h += mainmenu(wiki, focus, loginUID)
	h += '</div>'
		
	# leave content <div> open
	h += '<div class="wikkly-content-container">'
	
	return h
	
def pageFooter(wiki, focus):
	"Return page footer as UTF-8"
	from wikklytext.base import get_version
	# close content-container
	h = '</div>'
	
	# close outer container
	h += '</div>'

	# add footer div
	h += '<div class="wikkly-footer-container">'
	h += 'Powered by <a class="wikkly-footer-content" title="%s" href="http://wikklytext.com">WikklyText</a>' % \
			('WikklyText %s' % get_version(),)
	h += '</div>'
	
	h += utf8(itemText(wiki, 'MarkupPostBody'))
	h += '</body>'
	h += '</html>'
	return h

def default_mainmenu():
	"Used when MainMenu is missing"
	return defcontent.MainMenu
	
def mainmenu(wiki, focus, loginUID):
	item = wiki.getitem('MainMenu')
	if item is not None:
		t = item.content
	else:
		t = default_mainmenu()
		
	# if it looks like the standard menu items are missing,
	# add the default menu. this is nice when rendering
	# TiddlyWikis since their menus won't have links for
	# these items (since they are provided on tabs)
	if t.find('DoServerCmd') < 0:
		t += u'\n----\n'
		t += default_mainmenu()
		
	# add box for logged-in users
	if wiki.user_valid_UID(loginUID):
		t += u'\n----\n'
		
		t += u'{{navmenu{\n'
		name = wiki.user_get_username(loginUID)
		
		t += u'* Logged in as [[%s|%s]]' % \
				(utf8(name), utf8(loginUID))
		
		if loginUID == '0':
			t += u' (root user)'
		
		t += u'\n'
		
		p = {'cmd': 'editUser', 'uid': loginUID}
		t += u'* [[Edit|DoServerCmd?%s]]\n' % urlencode(p)
		
		p = {'cmd': 'logout'}
		t += u'* [[Logout|DoServerCmd?%s]]\n' % urlencode(p)
		
	# always render as UID=0 so doesn't show up strangely for other users
	# (assume only a trusted user can edit mainmenu)
	return render(wiki, t, UID='0')
		
#-----------------------------------------------------------------------------------
#
# This is the public API that is called by the wiki layer.
#
#-----------------------------------------------------------------------------------

def layoutPage(wiki, item, loginUID):
	"""
	Layout a complete wiki page for the given item.
	
	wiki: The WikklyWiki
	item: The WikklyItem to render.
	loginUID: The logged-in user, or None for no user.
	
	Returns complete page as HTML (encoded as UTF-8).
	"""
	from wikklytext.wiki.render import render_inner_html
	import re	
	from wikklytext.store import tags_split
	
	# get list of items to render
	allitems = []		
	if item.name == 'DefaultTiddlers':
		h = pageHeader(wiki, 'home', 'Home', loginUID)
		# for 'DefaultTiddlers', pull in content from all named items
		#
		# as I understand it, DefaultTiddlers can only contain names
		# of tiddlers, possibly like "[[name]]", so just do a wordsplit
		# pull in those tiddlers
		for name in tags_split(item.content):
			allitems.append((name,wiki.getitem(name)))
	else:
		# else, render single item
		h = pageHeader(wiki, None, item.name, loginUID)
		allitems.append((item.name,item))
		
	for name,item in allitems:
		if item is None:
			h += '<div class="wikkly-highlight">ERROR: No such item "%s"</div>' % utf8(name)
			continue
			
		# make an outside container to catch double-clicks to edit this item. 
		# Doing this per-item means that if you click an item on index.html
		# (DefaultTiddlers) you will edit the clicked-on item, *NOT* DefaultTiddlers.
		# This is by design (99% of the time I want to edit the item, NOT DefaultTiddlers itself).
		p = {'cmd': 'editItem', 'name': utf8(item.name)}
		h +='''<div class="wikkly-item-container" ondblclick="window.location='%s'">''' % \
			('DoServerCmd.html?%s' % escapeURL(urlencode(p)))
			
		h += '<div class="wikkly-item-header">'
		h += utf8(itemTitle(wiki, item.name))
		h += utf8(authorTagline(wiki, item))
		h += tagline(wiki, item)
		h += '</div>'
		do_cache = wiki.cacheable(item)
			
		# render with UID of *author*, not logged-in user
		# (add 'safe_mode' setting to digest so caching will depend on it as well)
		inner = render_inner_html(wiki, item.name, item.author)
					
		# watch for programming error
		if isinstance(h, unicode):
			raise Exception("INTERNAL ERROR: This should be UTF-8!!")
			
		h += '<div class="wikkly-item-content">' + inner + '</div>'
		
		# close outside container
		h += '</div>'
		
	h += pageFooter(wiki, 'none')
	return h

def layoutTimelinePage(wiki, loginUID):
	return layoutIndexPage(wiki, 'Timeline', 'timeline', makeTimelineIndex, loginUID)

def layoutNameIndexPage(wiki, loginUID):
	return layoutIndexPage(wiki, 'Items by Name', 'names', makeNameIndex, loginUID)

def layoutTagIndexPage(wiki, loginUID):
	return layoutIndexPage(wiki, 'Tags', 'tags', makeTagIndex, loginUID)
	
def layoutIndexPage(wiki, title, focus, factory, loginUID):
	"""
	Layout the timeline page.
	
	wiki: the WikklyWiki
	
	Returns complete page as HTML (encoded as UTF-8).
	"""
	h = pageHeader(wiki, focus, title, loginUID=loginUID)
	
	h += '<div class="wikkly-item-header">'
	h += utf8(itemTitle(wiki, title, can_edit=False))
	h += '</div>'
	txt = factory(wiki)
	h += '<div class="wikkly-item-content">' + render(wiki, txt, UID=None) + '</div>'
	
	h += pageFooter(wiki, focus)
	return h
	
def layoutAdminPage(wiki, loginUID, need_restart=False):
	"""
	Layout main admin page.
	
	Page must provide a form with actions as shown below.
	"""
	import wikklytext.version as VER
	import boodebr.version as BVER
	import wsgifront.version as WSVER
	import sys

	h = pageHeader(wiki, 'admin', 'Administration', loginUID=loginUID)
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Wiki Administration</span></div>'
	h += '<div class="wikkly-item-content">'
	
	# note: All unicode strings will be converted to UTF-8 in the process of wikifying,
	# so no explicit encoding is needed here. the server has to remember to decode them.
	t = u''

	t += u'!!!Site Configuration\n'
	
	# provide links for commonly edited configuration items
	for name in ['DefaultTiddlers', 'MainMenu', 'SiteTitle', 'SiteSubtitle',
			'MarkupPreHead', 'MarkupPostHead', 'MarkupPreBody', 'MarkupPostBody',
			'StyleSheet']:
		if wiki.getitem(name) is None:
			extra = '<br>^^@@No ~%s currently defined. Click above link to create it.@@^^' % name
		else:
			extra = ''
		t += u'* [[%s|DoServerCmd?%s]]%s\n' % \
			(name, urlencode({'cmd':'editItem', 'name': name}), extra)			
	
	t += '!!!Logs\n'
	t += '* [[Recent hits|DoServerCmd?cmd=logRecentHits]] '
	t += '^^'
	p = {'cmd': 'logRecentHits', 'type': metadata.LOG_PAGEVIEW} 
	t += '([[Content|DoServerCmd?%s]]) ' % urlencode(p)
	p = {'cmd': 'logRecentHits', 'type': metadata.LOG_ERRORMSG} 
	t += '([[Errors|DoServerCmd?%s]]) ' % urlencode(p)
	p = {'cmd': 'logRecentHits', 'type': metadata.LOG_SETUID} 
	t += '([[Set UID|DoServerCmd?%s]]) ' % urlencode(p)
	p = {'cmd': 'logRecentHits', 'type': metadata.LOG_FEED} 
	t += '([[RSS|DoServerCmd?%s]]) ' % urlencode(p)
	t += '^^\n'
	
	t += '* [[Hits by page|DoServerCmd?cmd=logHitsByPage]]\n'
	t += '* [[Hits by days|DoServerCmd?cmd=logHitsByDates]]\n'
	t += '\n'
	
	t += u'!!!!!User-defined filters\n\n'
	t += u'* [[Hits on user filters|DoServerCmd?cmd=logHitsByUserFilters]]\n'
	t += u'Enter filters, one per line, into the following box. Filters are SQL "{{tt{LIKE}}}" clauses to '
	t += u'be matched against server requests. For example, the filter {{{%.png%}}} would match '
	t += u'the requests: {{{\n'
	t += u'GET /favicon.png HTTP/1.1\n'
	t += u'GET /files/image.png HTTP/1.1\n'
	t += u'}}}\n'
	t += u'<html><form action="DoServerCmd.html" method="post">'
	t += u'<fieldset class="wikkly-form">'
	t += u'<textarea name="filters" rows="10" cols="40">%s</textarea><br/>' % \
			'\n'.join(wiki.get_request_log_filters())
	t += u'<input type="hidden" name="cmd" value="setUserLogFilters"/>'
	t += u'<input type="submit" name="Set" value="Save Filters"/>'
	t += u'</fieldset>'
	t += u'</form></html>\n'
	
	t += '!!!Users\n'
	
	p = {'cmd': 'editUser', 'uid': ''}
	t += '* [[Create new user|DoServerCmd?%s]]\n\n' % urlencode(p)
	
	users = [(uid,wiki.user_get_username(uid)) for uid in wiki.user_all_UIDs()]
	users.sort(lambda a,b: cmp(a[1],b[1]))
	
	t += "|Existing Users|c\n"
	t += '|!Username|!Can login?|!Safe mode?|!Email|!Edit|\n'		
	for uid,name in users:
		p = {'cmd': 'editUser', 'uid': uid}
		if wiki.user_get_safemode(uid):
			safe = 'True'
		else:
			safe = '@@False@@' # highlight users who run in full mode
			
		if uid == '0':
			fname = '!{{{%s}}}<br>{{grayout{^^(Root User)^^}}}' % safe_code(name)
		else:
			fname = '!{{{%s}}}' % safe_code(name)
			
		if wiki.user_can_login(uid):
			canlog = '@@Yes@@'
		else:
			canlog = 'No'
			
		t += '|%s | %s | %s | {{{%s}}} | [[edit|DoServerCmd?%s]] |\n' % \
					(fname, canlog, safe, safe_code(wiki.user_get_email(uid)), urlencode(p))

	t += u"<<set storage_form <quote>"
	t += u'<html><form action="DoServerCmd.html" method="post">'
	t += u'<fieldset class="wikkly-form">'
	t += u'<select name="storage_type">'
	t += u'<option %s value="ram">ram</option>' % \
			ifelse(wiki.get_session_storage() == 'ram', 'selected="selected"', '')
			
	t += u'<option %s value="file">file</option>' % \
			ifelse(wiki.get_session_storage() == 'file', 'selected="selected"', '')
			
	t += u'</select>' + '&nbsp;'*2
	t += u'<input type="hidden" name="cmd" value="setSessionStorage"/>'
	t += u'<input type="submit" name="Set" value="Set Storage Type"/>'
	t += u'</fieldset>'
	t += u'</form></html>\n'
	t += u'</quote>>>\n'
	
	t += u"<<set timeout_form <quote>"
	t += u'<html><form action="DoServerCmd.html" method="post">'
	t += u'<fieldset class="wikkly-form">'
	t += u'<input type="text" name="timeout" value="%s"/>' % str(wiki.get_session_timeout())
	t += '&nbsp;'*2
	t += u'<input type="hidden" name="cmd" value="setSessionTimeout"/>'
	t += u'<input type="submit" name="Set" value="Set Timeout"/>'
	t += u'</fieldset>'
	t += u'</form></html>\n'
	t += u'</quote>>>\n'
	
	t += u"<<set timelimit_form <quote>"
	t += u'<html><form action="DoServerCmd.html" method="post">'
	t += u'<fieldset class="wikkly-form">'
	t += u'<input type="text" name="timelimit" value="%s"/>' % str(wiki.get_safemode_timelimit())
	t += '&nbsp;'*2
	t += u'<input type="hidden" name="cmd" value="setSafemodeTimelimit"/>'
	t += u'<input type="submit" name="Set" value="Set Time Limit"/>'
	t += u'</fieldset>'
	t += u'</form></html>\n'
	t += u'</quote>>>\n'
	
	t += u'<<set debug_form <quote>'
	t += u'<html><form action="DoServerCmd.html" method="post">'
	t += u'<fieldset class="wikkly-form">'
	t += u'<select name="debug_flag">'
	t += u'<option %s value="off">Off</option>' % \
			ifelse(wiki.get_debug_flag(), '', 'selected="selected"')
	t += u'<option %s value="on">On</option>' % \
			ifelse(wiki.get_debug_flag(), 'selected="selected"', '')
	t += '</select>' + '&nbsp;'*2
	t += u'<input type="submit" name="Set" value="Set debug flag"/>'
	t += u'<input type="hidden" name="cmd" value="setDebugFlag"/>'
	t += u'</fieldset>'
	t += u'</form></html>'
	t += u'</quote>>>\n'
	
	t += u'<<set cache_form <quote>'
	t += u'<html><form action="DoServerCmd.html" method="post">'
	t += u'<fieldset class="wikkly-form">'
	t += u'Enable? <select name="cache_flag">'
	t += u'<option %s value="off">Off</option>' % \
			ifelse(wiki.get_cache_flag(), '', 'selected="selected"')
	t += u'<option %s value="on">On</option>' % \
			ifelse(wiki.get_cache_flag(), 'selected="selected"', '')
	t += '</select>' + '&nbsp;'*2
	t += u'<br/>Expiration (seconds): <input type="text" name="expire_time" value="%s" size="6"/>' % str(wiki.get_cache_expire_time())
	t += u'<br/>Clean interval (accesses): <input type="text" name="clean_interval" value="%s" size="6"/><br/>' % str(wiki.get_cache_clean_interval())
	t += u'<input type="submit" name="Set" value="Save cache settings"/>'
	t += u'<input type="hidden" name="cmd" value="setCacheConfig"/>'
	t += u'</fieldset>'
	t += u'</form></html>'
	t += u'</quote>>>\n'
	
	t += u'<<set cache_help <quote>'
	t += u'<<div "" "font-size: .8em" "" """\n'
	t += u'* //Enable//: Turn cache on/off\n'
	t += u'* //Expiration//: Expiration time (seconds) for items added to the cache\n'
	t += u'* //Clean interval//: Clear out expired items (approximately) once per this many cache accesses\n'
	t += u'You may want to turn caching off when running on a USB drive, to minimize writes.\n'
	t += u'""">>'
	t += u'</quote> >>'
	
	t += u'<<set metadb_form <quote>'
	t += u'<html><form action="DoServerCmd.html" method="post">'
	t += u'<fieldset class="wikkly-form">'
	t += u'<select name="metadb_flag">'
	t += u'<option %s value="off">Off</option>' % \
			ifelse(wiki.get_metadb_flag(), '', 'selected="selected"')
	t += u'<option %s value="on">On</option>' % \
			ifelse(wiki.get_metadb_flag(), 'selected="selected"', '')
	t += '</select>' + '&nbsp;'*2
	t += u'<input type="submit" name="Set" value="Set metadb flag"/>'
	t += u'<input type="hidden" name="cmd" value="setMetadbFlag"/>'
	t += u'</fieldset>'
	t += u'</form></html>'
	t += u'</quote>>>\n'
	
	t += u'<<set unknown_camelword_form <quote>'
	t += u'<html><form action="DoServerCmd.html" method="post">'
	t += u'<fieldset class="wikkly-form">'
	t += u'<select name="unknown_camelword_flag">'
	t += u'<option %s value="off">Off</option>' % \
			ifelse(wiki.get_link_unknown_camelwords(), '', 'selected="selected"')
	t += u'<option %s value="on">On</option>' % \
			ifelse(wiki.get_link_unknown_camelwords(), 'selected="selected"', '')
	t += '</select>' + '&nbsp;'*2
	t += u'<input type="submit" name="Set" value="Set camelwords flag"/>'
	t += u'<input type="hidden" name="cmd" value="setUnknownCamelwordsFlag"/>'
	t += u'</fieldset>'
	t += u'</form></html>'
	t += u'</quote>>>\n'
	
	t += u'!!!Plugins\n'
	
	t += '[[Plugins|%s/api/plugins]]\n' % wiki.getRT_baseurl()
	
	t += u'<html><form action="DoServerCmd.html" method="post">'
	t += u'<fieldset class="wikkly-form">'
	t += u'Enter additional directories to search for plugins, in addition to '
	t += u'the default location (<tt>%s</tt>). Enter one directory per line.<br/>' % wiki.get_plugin_path()
	t += u'<textarea name="plugindirs" rows="10" cols="40">%s</textarea><br/>' % \
		('\n'.join(wiki.get_user_plugin_dirs()))
	t += u'<input type="hidden" name="cmd" value="setUserPluginDirs"/>'
	t += u'<input type="submit" name="Set" value="Save Plugin Directories"/>'
	t += u'</fieldset>'
	t += u'</form></html>\n'
	
	t += u'!!!Server Configuration\n'
	if need_restart:
		t += '<<warnbox "Need Restart" "You must restart the server for changes to take effect.">>\n'
		
	t += '|!Parameter|!Current Value|!Adjust|\n'
	t += '|!Session storage<br>~~@@CAUTION@@: I do not recommend using "file" sessions if your wiki is stored on a network drive.<br>For USB drives, you may want to set this to {{{ram}}} to minimize writes.~~| %s |<<get storage_form>>|\n' % wiki.get_session_storage()
	t += '|!Session timeout<br>~~(minutes)~~| %s |<<get timeout_form>>|\n' % wiki.get_session_timeout()
	t += '|!Rendering time limit<br>~~(seconds)<br>This affects restricted users only; full users have no limit. Use -1 for no limit.~~| %d |<<get timelimit_form>>|\n' % wiki.get_safemode_timelimit()
	t += '|!Rendering Cache<br><<get cache_help>>| %s |<<get cache_form>>|\n' % \
				ifelse(wiki.get_cache_flag(), 'On', 'Off')
	t += '|!~MetaDb flag<br>~~You may want to turn this off when running on a USB drive, to minimize writes. Currently, this disables logging, but more features will be turned on/off in the future.~~| %s |<<get metadb_form>>|\n' % \
				ifelse(wiki.get_metadb_flag(), 'On', 'Off')
	t += '|!Make links to unknown ~CamelWords| %s |<<get unknown_camelword_form>>|\n ' % \
				ifelse(wiki.get_link_unknown_camelwords(), 'On', 'Off')
	t += '|!Debugging flag<br>~~@@I recommend leaving this OFF since it exposes server information.@@<br>To use, pass {{{cmd=debug}}}.~~| %s |<<get debug_form>>|\n' % \
				ifelse(wiki.get_debug_flag(), 'On', 'Off')
	
	t += "!!!Statistics\n"
	st = wiki.rendercache().stats()
	t += '|>|>|>|!Caching|\n'
	t += '|!type|>|>| %s |\n' % st['type']
	t += '|!puts|>|>| %d |\n' % st['puts']
	t += '|!gets|>|>| %d |\n' % st['gets']
	t += '|!hits|>|>| %d (%.1f%%) |\n' % (st['hits'], (st['hits']*100.0/st['gets']))
	t += '|!objects|>|>| %d |\n' % st['objects']
	t += '|!bytes|>|>| %d |\n' % st['bytes']
	t += '|!expiration_time|>|>| %d |\n' % st['expiration_time']
	t += '|!clean_interval|>|>| %d |\n' % st['clean_interval']
	
	t += u'<html><form action="DoServerCmd.html" method="post">'
	t += u'<fieldset class="wikkly-form">'
	t += u'<input type="submit" name="Set" value="Clear cache (no prompting!)"/>'
	t += u'<input type="hidden" name="cmd" value="clearRenderCache"/>'
	t += u'</fieldset>'
	t += u'</form></html>\n'
	
	t += u'!!!Server Information\n'
	t += u";~WikklyText version\n:{{{%s}}}\n" % VER.VSTR
	t += u";boodebr version\n:{{{%s}}}\n" % BVER.VSTR
	t += u";wsgifront version\n:{{{%s}}}\n" % WSVER.VSTR
	t += u";Python version\n:{{{%s}}}\n" % sys.version
	# this module is (in general) not allowed to touch cherrypy.*, but this 
	# is just to get the version string.
	import cherrypy
	t += u';~CherryPy version\n:{{{%s}}}\n' % cherrypy.__version__
	t += u';Server command\n:{{{'
	t += u' '.join(sys.argv)
	t += '}}}\n'
	t += u';sys.path\n:{{{'
	for p in sys.path:
		t += p + '\n'
	
	t += '}}}\n'
	
	h += render(wiki, t, UID='0')
	
	h += '</div>'
	
	h += pageFooter(wiki, None)
	return h
	
def layoutDeleteItem(wiki, itemname, loginUID):
	"""
	Layout a page for confirmation on deleting the named item.
	
	Page must provide a form with actions as shown below.
	"""
	h = pageHeader(wiki, None, 'Delete Item', loginUID=loginUID)
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Confirmation</span></div>'
	h += '<div class="wikkly-item-content">'
	
	# note: All unicode strings will be converted to UTF-8 in the process of wikifying,
	# so no explicit encoding is needed here. the server has to remember to decode them.
	f = u'<html>'
	f += u'<form action="DoServerCmd.html" method="post">'
	f += u'<fieldset class="wikkly-form">'
	f += u'Are you sure you want to delete "<b>%s</b>"?<p><i>(This cannot be undone!)</i><p>' % \
				safehtml(itemname)
	f += u'<input type="hidden" name="itemName" value="%s"/>' % safehtml(itemname)
	f += u'<input type="hidden" name="cmd" value="deleteItem"/>'
	f += u'<input type="submit" name="btn_yes" value="Yes"/> '
	f += u'<input type="submit" name="btn_no" value="No"/>'
	f += u'</fieldset>'
	f += u'</form>'
	f += u'</html>'
	
	h += render(wiki, f, UID="0")
	
	h += '</div>'
	
	h += pageFooter(wiki, None)
	return h
			
def layoutEditItem(wiki, itemname, defaultItem, loginUID):
	"""
	Layout a page for editing the named item.
	
	Page must provide a form with action as shown below.
	"""
	from wikklytext.store import tags_join
	
	item = wiki.getitem(itemname)
	if item is None:
		item = defaultItem
		oldname = ''
	else:
		oldname = item.name
		
	if item is defaultItem:
		h = pageHeader(wiki, 'newitem', 'New Item', loginUID=loginUID)
	else:
		h = pageHeader(wiki, None, 'Editing Item', loginUID=loginUID)
		
	if wiki.user_valid_UID(itemname):
		name = wiki.user_get_username(itemname)
		h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Editing: %s</span></div>' % \
					utf8(safehtml(name))		
	else:
		h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Editing: %s</span></div>' % \
					utf8(safehtml(itemname))
					
	h += '<div class="wikkly-item-content">'
	
	# note: All unicode strings will be converted to UTF-8 in the process of wikifying,
	# so no explicit encoding is needed here. the server has to remember to decode them.
	f = u'<html>'
	f += u'<form action="DoServerCmd.html" method="post">'
	f += u'<fieldset class="wikkly-form">'
	# if name is a UID, show author name instead (and make it non-editable)
	if wiki.user_valid_UID(itemname):
		name = wiki.user_get_username(itemname)
		f += u'Name: <b>%s</b><br/>' % safehtml(name)
		# put name in hidden item instead
		f += u'<input type="hidden" name="itemName" value="%s"/>' % safehtml(itemname)
	else:
		f += u'Name: <input class="wikkly-edit-input" type="text" name="itemName" value="%s"/><br/>' % \
				safehtml(itemname)
	f += u'<textarea class="wikkly-edit-textarea" name="content" cols="80" rows="25">'
	txt = item.content
	# escape any </html> tags so won't end early
	txt = txt.replace('</html>', '&lt;/html&gt;')
	f += txt
	f += u'</textarea><br/>'
	f += u'Tags: <input class="wikkly-edit-input" type="text" name="tags" value="%s"/><br/>' % \
			safehtml(tags_join(item.tag_list()))
	f += u'&nbsp;'*10 + '<sup>Enter space-separate tags, using [[braces if needed]].</sup><br/>'
	f += u'Markup style: <select name="content_type">'
	f += u'   <option %s value="WikklyText">WikklyText</option>' % \
				ifelse(item.content_type=='WikklyText', u'selected="selected"', u'')
	f += u'   <option %s value="TiddlyWiki">TiddlyWiki</option>' % \
				ifelse(item.content_type=='TiddlyWiki', u'selected="selected"', u'')
	f += u'</select><br/><br/>'
	f += u'<input type="hidden" name="oldItemName" value="%s"/>' % oldname
	f += u'<input type="hidden" name="cmd" value="completeEdit"/>'
	f += u'<input type="hidden" name="ctime" value="%s"/>' % item.ctime.to_store()
	f += u'<input type="submit" name="btn_save" value="Save"/> '
	if len(oldname): # don't show delete button on new items
		f += u'<input type="submit" name="btn_delete" value="Delete Item"/><br/>'
		
	f += u'</fieldset>'
	f += u'</form>'
	f += u'</html>'
	
	h += render(wiki, f, UID='0')
	
	h += '</div>'
	
	h += pageFooter(wiki, None)
	return h
		
def layoutEditUser(wiki, loginUID, editUID=None):
	"""
	Layout a page to edit a user.
	
	loginUID = UID of logged-in user
	editUID = UID to create/edit (if None, create new user)
	
	Page must provide a form with actions as shown below.
	"""
	h = pageHeader(wiki, None, 'Edit User', loginUID=loginUID)
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Editing User</span></div>'
	h += '<div class="wikkly-item-content">'
	
	# note: All unicode strings will be converted to UTF-8 in the process of wikifying,
	# so no explicit encoding is needed here. the server has to remember to decode them.
	f = u''
	if '0' not in wiki.user_all_UIDs():
		f += u'@@You are the first user, and will therefore be the root user.<br><br>You must set a username and password before continuing.<br><br>@@\n'
				
	f += u'<html>'
	f += u'<form action="DoServerCmd.html" method="post">'
	f += u'<fieldset class="wikkly-form">'
	if editUID is None or not wiki.user_valid_UID(editUID):
		name = u''
		email = u''
	else:
		name = wiki.user_get_username(editUID)
		email = wiki.user_get_email(editUID)

	f += u'<table>'		
	f += u'<tr><td>Username:</td><td colspan="2"><input class="wikkly-edit-input" type="text" name="username" value="%s"/></td></tr>' % \
				safehtml(name)
	f += u'<tr><td>Email address:</td><td colspan="2"><input class="wikkly-edit-input" type="text" name="email" value="%s"/></td></tr>' % \
				safehtml(email)
	f += u'<tr><td>Password:</td><td colspan="2"><input class="wikkly-edit-input" type="password" name="password1" value=""/></td></tr>'
	f += u'<tr><td>Confirm Password:</td><td colspan="2"><input class="wikkly-edit-input" type="password" name="password2" value=""/></td></tr>'
	# only root user can change safe_mode settings
	if loginUID == '0':
		f += u'<tr><td>Layout mode:</td><td>'
		f += u'<select name="safe_mode">'
		f += u'  <option %s value="Safe">Safe Mode</option>' % \
					ifelse(wiki.user_get_safemode(editUID), u'selected="selected"', u'')
		f += u'  <option %s value="Full">Full Mode</option>' % \
					ifelse(not wiki.user_get_safemode(editUID), u'selected="selected"', u'')
		f += u'</select></td></tr>'
	
	f += u'</table>'
	f += u'<input type="hidden" name="cmd" value="saveUser"/>'
	# this is checked vs. logged-in UID, but this is needed here so the root user can
	# edit other users profiles
	if editUID is not None:
		f += u'<input type="hidden" name="uid" value="%s"/>' % editUID
	else:
		f += u'<input type="hidden" name="uid" value=""/>'
	
	if editUID is None or not wiki.user_valid_UID(editUID):
		f += u'<input type="submit" value="Create User"/>'
	else:
		f += u'<input type="submit" value="Save Settings"/>'
		
	f += u'</fieldset>'
	f += u'</form>'
	f += u'</html>'
	
	# need to run w/safe_mode=False so use 'AutoContent' user
	# (I escaped text above to make sure it was safe)
	h += render(wiki, f, UID=wiki.user_getUIDs('AutoContent')[0])
	
	h += '</div>'
	
	h += pageFooter(wiki, None)
	return h

def layoutErrorPage(wiki, message, loginUID):
	h = pageHeader(wiki, None, 'Error Page', loginUID=loginUID)
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Error</span></div>'
	h += '<div class="wikkly-item-content">'
	
	h += utf8(safehtml(message))
	h += '</div>'
	
	return h
	
def layoutLoginPage(wiki):
	"""
	Layout a login page.
	
	Page must provide a form with actions as shown below.
	"""
	h = pageHeader(wiki, None, 'Login') # don't care about passing loginUID here
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Login</span></div>'
	h += '<div class="wikkly-item-content">'
	
	# note: All unicode strings will be converted to UTF-8 in the process of wikifying,
	# so no explicit encoding is needed here. the server has to remember to decode them.
	f = u''
	f += u'<html>'
	f += u'<form action="DoServerCmd.html" method="post">'
	f += u'<fieldset class="wikkly-form">'
	f += u'Username: <input class="wikkly-edit-input" type="text" name="username" value=""/><p>'
	f += u'Password: <input class="wikkly-edit-input" type="password" name="password" value=""/><p>'			
	f += u'<input type="hidden" name="cmd" value="doLogin"/>'
	f += u'<input type="submit" value="Login"/> '
	f += u'</fieldset>'
	f += u'</form>'
	f += u'</html>'
	
	h += render(wiki, f, UID='0')
	
	h += '</div>'
	
	h += pageFooter(wiki, None)
	return h

def layoutSearchResults(wiki, query):
	items = wiki.search(query)
	
	h = pageHeader(wiki, None, 'Search Results')
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Search Results</span></div>'
	h += '<div class="wikkly-item-content">'
	
	t = u''
	t += query.explain() + '<br><br>'
	
	if len(items) == 0:
		t += u"''Results'': No items found."
	else:
		t += u"''Results'':\n"
		for item in items:
			t += '* [[%s]]\n' % item.name
			
	h += render(wiki, t, UID=None)
	
	h += '</div>'
	
	h += pageFooter(wiki, None)
	return h
	
import metadata
import socket
	
def layoutLogHitsByPage(wiki, sql):
	h = pageHeader(wiki, None, 'Search Results')
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Search Results</span></div>'
	h += '<div class="wikkly-item-content">'

	pages = metadata.get_hits_by_page(sql)
	items = list(pages.items())
	items.sort(lambda a,b: cmp(b[1],a[1]))
	t = u'|!Page|!Hits|\n'
	for name,hits in items:
		t += u'|%s|%d|\n' % (name, hits)
	
	h += render(wiki, t, UID='0')
		
	h += '</div>'	
	h += pageFooter(wiki, None)
	return h

def layoutLogRecentHits(wiki, sql, logtype):
	h = pageHeader(wiki, None, 'Search Results')
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Search Results</span></div>'
	h += '<div class="wikkly-item-content">'
	
	t = u''
	
	t += '|!Date/Time|!Type|!Resource|!UID|!Src IP|!Request line|\n'
	
	# use <nowiki> on all fields where a user can inject text
	for o in metadata.get_recent_logs(sql, max_nr=200, logtype=logtype):
		t += '|%s|%s|<nowiki>%s</nowiki>|%s|{{{%s}}}|<nowiki>%s</nowiki>|\n' % \
				(o.datetime, metadata.LOG_TYPES.get(o.logtype,'*UNKNOWN*'), 
					safe_nowiki(o.resource), o.uid, 
					safe_code(o.remote_addr),
					safe_nowiki(o.request_line))
				
	h += render(wiki, t, UID='0')
		
	h += '</div>'	
	h += pageFooter(wiki, None)
	return h

def layoutLogHitsByDates(wiki, sql):
	h = pageHeader(wiki, None, 'Search Results')
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Search Results</span></div>'
	h += '<div class="wikkly-item-content">'
	
	t = u''
	
	t += '|!Date|!All|!Page<br>views|!RSS|!Static<br>Files|!Server<br>Cmds|!Logins|!Errors|\n'
	
	for datestr,counts in metadata.get_hits_by_dates(sql):
		t += '|!%s| %s| %s| %s| %s| %s| %s| %s|\n' % \
			(datestr, counts['ALL'], counts['PAGEVIEWS'], counts['FEED'],
				counts['STATICFILES'], counts['SERVERCMDS'], 
				counts['SETUID'], counts['ERRORS'])
				
	h += render(wiki, t, UID='0')
		
	h += '</div>'	
	h += pageFooter(wiki, None)
	return h

def layoutLogHitsByUserFilters(wiki, sql):
	h = pageHeader(wiki, None, 'Search Results')
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Search Results</span></div>'
	h += '<div class="wikkly-item-content">'

	t = u''
	for like in wiki.get_request_log_filters():
		t += '!!!!{{{%s}}}: %d hits\n\n' % (safe_code(like), metadata.get_requests_count_by_LIKE(sql, like))
		t += 'Recent hits:\n'
		t += '|!Request|!Date|\n'
		for o in metadata.get_requests_by_LIKE(sql,like):
			t += '|{{{%s}}}|%s|\n' % (safe_code(o.request_line), o.datetime.split()[0])
			
	h += render(wiki, t, UID='0')
		
	h += '</div>'	
	h += pageFooter(wiki, None)
	return h

from textwrap import wrap
	
def debug_show_dict(d):
	t = u''
	keys = d.keys()
	keys.sort(lambda a,b: cmp(a.lower(),b.lower()))
	t += u'//Keys://\n{{{ %s }}}\n' % u'\n'.join(wrap(str(keys)))
	for k in keys:
		kstr = str(d[k]).replace('}', '}&#x200b;').replace('{', '{&#x200b;')
		kstr = u'\n'.join(wrap(kstr))
		t += u';%s\n:{{{%s}}}\n' % (k, kstr)
		
	return t
	
def layoutDebugPage(wiki):
	# this is the only layout routine that is allowed to touch 'cherrypy.*' directly,
	# and its only for debugging purposes. the rest of this module is completely independent
	# of cherrypy
	import cherrypy
	
	h = pageHeader(wiki, None, 'Debug Info')
	
	h += '<div class="wikkly-item-header"><span class="wikkly-item-title">Debug Info</span></div>'
	h += '<div class="wikkly-item-content">'
	
	t = u''
	
	req = cherrypy.request
	
	t += u'!!Request ({{{cherrypy.request}}})\n'
	
	t += u';request.base\n:{{{%s}}}\n' % req.base
	t += u';request.body\n:{{{%s}}}\n' % req.body
	t += u';request.method\n:{{{%s}}}\n' % req.method
	t += u';request.params\n:<<indent <quote>'
	t += debug_show_dict(req.params)
	
	t += u'</quote>>>\n'
	
	t += u';request.pathinfo\n:{{{%s}}}\n' % req.path_info
	t += u';request.protocol\n:{{{%s}}}\n' % str(req.protocol)
	t += u';request.query_string\n:{{{%s}}}\n' % req.query_string
	t += u';request.request_line\n:{{{%s}}}\n' % req.request_line
	t += u';request.scheme\n:{{{%s}}}\n' % req.scheme
	t += u';request.script_name\n:{{{%s}}}\n' % req.script_name
	t += u';request.server_protocol\n:{{{%s}}}\n' % req.server_protocol
	t += u';request.show_tracebacks\n:{{{%s}}}\n' % req.show_tracebacks
	
	t += u'!!WSGI Environment ({{{cherrypy.wsgi_environ}}})\n'
	wsgi = wiki.getRT_wsgienv()
		
	t += debug_show_dict(wsgi)
	
	t += u'!!Config ({{{cherrypy.request.config}}})\n'
	for k,v in req.config.items():
		t += ';%s\n:{{{%s}}}\n' % (k,v)
		
	t += u'!!Headers ({{{cherrypy.request.headers}}})\n'
	for k,v in req.headers.items():
		t += ';%s\n:{{{%s}}}\n' % (k,v)
	
	t += u'!!Remote Host ({{{cherrypy.request.remote}}})\n'
	t += ';IP Address\n:{{{%s}}}\n' % (req.remote.ip)
	t += ';Port\n:{{{%s}}}\n' % (req.remote.port)
	t += ';Name\n:{{{%s}}}\n' % (req.remote.name)
	
	t += u'!!{{{cherrypy.__dict__}}}\n'
	t += debug_show_dict(cherrypy.__dict__)
	
	t += u'!!{{{cherrypy.request.__dict__}}}\n'
	t += debug_show_dict(cherrypy.request.__dict__)
	
	t += u'!!{{{cherrypy.server.__dict__}}}\n'
	t += debug_show_dict(cherrypy.server.__dict__)
	
	t += u'!!{{{cherrypy.session.__dict__}}}\n'
	t += debug_show_dict(cherrypy.session.__dict__)
	
	h += render(wiki, t, UID='0')
	
	h += '</div>'	
	h += pageFooter(wiki, None)
	return h
	
def layoutWikiWasUpdated(wiki):
	
	# generate as UTF-8
	h = ''
	h += HEAD(title='WikklyText: Metadata Updated', baseurl=wiki.getRT_baseurl(),
				reqheaders=wiki.getRT_wsgienv())
	h += '<body class="wikkly-body">'
	
	# add spacer at top
	h += '<div class="wikkly-top-spacer"></div>'
	# create outer container <div>
	h += '<div class="wikkly-outside-container">'

	# generate by hand -- since the wiki/db needs to be updated, it may
	# not be safe to call any rendering routines
	
	h += '<h1 class="wikkly-h1">Wiki Updated</h1>'
	
	h += '<p>You have installed an update to WikklyText that required a metadata update. '
	h += '<p>You should be able to simply reload this page and continue.'
	h += '<hr class="wikkly-separator">If you continue to see this message, then you may need to run the update manually.'
	h += '<br/>From your wiki folder, run:'
	h += "<div class='wikkly-code-block'>"
	h += "wik updatemeta<br/>"
	h += '</div>'
		
	h += '<hr class="wikkly-separator">If you are still seeing this message after trying the above steps, please report '
	h += 'this to the <a href="http://groups.google.com/group/wikklytext">mailing list</a>.'
	
	h += '</div>'
	h += '</body>'
	h += '</html>'
	
	return h
	
