"""
wikklytext.wikwords.py - Wiki-word-ification

Copyright (C) 2007,2008 Frank McIngvale

Contact: fmcingvale@gmail.com

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.
"""
from wikklytext import WikklyText_to_XML
from wikklytext.base import load_wikitext, ElementTree, StringIO, ElementList, Text
import re

def is_camelword(text):
	"Does text *begin* with a CamelWord?"
	text, word, extra = camelpair(text)
	return len(word) > 0	
	
def camelpair(text):
	"""
	Split next piece of text into a pair ('CamelWord', 'non camel text').
	Either or both piece may be empty. If both are empty, there is no
	more text to split.
	
	Returns:
		(text, camel_word, non_camel_text)
		
	Where:
		text = Remainder of text after match
		camel_word = CamelWord text
		non_camel_text = Text after CamelWord
	"""
	if len(text) == 0:
		return ('', '', '') # no more text

	# CamelCase word match, with possible '~' at the start
	# (must have at least two capitals seperated by lowercase)

	# handle: 'CamelWord non camel text CamelWord'
	#         'CamelWord non camel text'
	#         'CamelWord'
	m = re.match('([~]?[A-Z][a-z]+[A-Z][a-zA-Z]+)(\s+([^A-Z~]+))?', text)
	if m:
		return (text[len(m.group(0)):], m.group(1), m.group(2) or '')
		
	# if it begins with non-camel char, eat entire word + trailing whitespace
	m = re.match('([^A-Z~]\S*\s*)', text)
	if m:
		return (text[len(m.group(0)):], '', m.group(1))
		
	# leading text is something like ABC that isn't a camelword, yet doesn't
	# match the above rule. eat this word and trailing whitespace as-is.
	m = re.match('(\S+\s*)', text)
	if m:
		return (text[len(m.group(0)):], '', m.group(1))
	
	# should be impossible
	raise Exception("Internal error")
	
def camelsplit(text):
	"""
	Split a piece of text into wiki (CamelCase) words and
	non-wiki words.
	
	Returns:
		(wordlist, fraglist)
		
	Where:
		wordlist = List of CamelCase words
		fraglist = List of all other text fragments.
		
	The lists are padded as needed so that text can always
	be reassembled as:
		
		text = wordlist[0] + fraglist[0] + ... + wordlist[N] + fraglist[N]
		
	This may require that an empty word be added to the beginning of wordlist.
	"""
	wordlist = []
	fraglist = []
	while len(text):
		text, camel, noncamel = camelpair(text)
		wordlist.append(camel)
		fraglist.append(noncamel)
		
	return (wordlist, fraglist)
		
def cameljoin(wordlist, fraglist):
	"""
	Take the output from camelsplit an reassemble input text.
	
	Returns text.
	"""
	if len(wordlist) != len(fraglist):
		raise Exception("Bad inputs")
		
	text = u''
	for word,frag in zip(wordlist,fraglist):
		text += word + frag
		
	return text
	
def on_unknown_camelword(word):
	"Default policy on unknown camelwords -- treat as normal text."
	return (word, False)
	
def wikiwordify_text(text, wikiwords, on_unknown=on_unknown_camelword):
	"""
	Where:
		text = Text to wikify, usually from a <Text> node, but
		       can be any arbitrary text.
		wikiwords = Dict of wikiwords as:
			wikiwords[word] = URL
		on_unknown_camelword = callback when unknown camelwords are found.
		    called as:
			   (text, needs_eval) = on_unknown(word)
			   
			Where: 
				text = replacement text.
			    needs_eval = True/False if returned text is wikitext that needs
				             to be evaluated.
	Returns:
		(wikitext, needs_eval)
		
	Where:
		wikitext: replacement for text.
		needs_eval: True/False if text needs evaluation.
	"""
	wordlist,fraglist = camelsplit(text)
	wikitext = u''
	#print "WORDLIST",wikiwords
	needs_eval = False
	
	for word,frag in zip(wordlist,fraglist):
		#print "*** WORD",word
		if not len(word):
			wikitext += frag
			continue
			
		#print "WORD",word
		# look for escaped wikiwords
		if word[0] == '~':
			# remove ~
			wikitext += word[1:]
		elif wikiwords.has_key(word):
			# its a link
			wikitext += u'[[%s|%s]]' % (word,wikiwords[word])
			needs_eval = True
		else:
			#print "*** CALLBACK WORD",word
			# unknown camelword - let callback decide what to do
			wtext,neval = on_unknown(word)
			if neval:
				needs_eval = True
				
			wikitext += wtext
			
		wikitext += frag
		
	#print "MADE WIKITEXT",repr(wikitext)
	return (wikitext, needs_eval)		
	
def wikiwordify(node, wcontext, wikiwords, on_unknown=on_unknown_camelword):
	"""
	Given an ElementTree generated by wikklytext.buildXML, turn all
	wikiwords into links.
	
	node = ElementTree node (typically the rootnode from the parser).
	wcontext = WikContext
	wikiwords = Dict of wikiwords as:
		wikiwords[word] = URL
		
	on_unknown_camelword = callback when unknown camelwords are found.
		called as:
		   (text, needs_eval) = on_unknown(word)
		   
		Where: 
			text = replacement text.
			needs_eval = True/False if returned text is wikitext that needs
						 to be evaluated.

	Modifies the tree in place.
	"""
	from wikklytext.eval import eval_wiki_text
	from wikklytext.base import xmltrace, ElementList

	if node.tag == 'Text':
		raise Exception("<Text> cannot be at toplevel here.")
		
	# ignore <Text> nodes under these tags
	if node.tag in ('Link', 'Image','CreateAnchor'):
		return
		
	for i in range(len(node)):
		subnode = node[i]
		
		#print '<%s>' % subnode.tag
		# only <Text> nodes are eligible for wikiword substitution,
		# don't look in <TextCode>, <TextHTML>, etc.
		if subnode.tag == 'Text':
			#print "CONSIDER:",subnode.text
			wtext,needs_eval = wikiwordify_text(subnode.text or '', wikiwords, on_unknown)
			if needs_eval:
				nodes = eval_wiki_text(wcontext, wtext)
			else:
				# saves a lot of time to only eval as needed
				nodes = [Text(wtext)]
				
			#print "GOT",xmltrace(nodes)
			# place all under a single ElementList
			elist = ElementList()
			for n in nodes:
				elist.append(n)
				
			#print "GOT",xmltrace(elist)
			
			# replace <Text> with <ElementList>
			node[i] = elist
			
		else:
			wikiwordify(subnode, wcontext, wikiwords, on_unknown)

def testsplit():
	textlist = [
		"""Regular words, CamelCase
		Not an escape: ~notescape
		An escape: ~CamelEscape
		WikiWord94 ~NonWikiWord86"""
		]
	
	for text in textlist:
		print "TEXT",repr(text)
		words,frags = camelsplit(text)
		print "WORDS",repr(words)
		print "FRAGS",repr(frags)
		text2 = cameljoin(words,frags)
		print "OK?",text==text2

def testwikify():
	text = """
	Hello world, HelloWorld.
	Here are OneWord, TwoWord, ThreeWord.
	"""
	wwords = {'HelloWorld':'http://helloworld',
				'OneWord':'http://oneword'}
	wikiwordify_text(text, wwords)

def testwikify2():
	from wikklytext.base import xmltrace
	from wikklytext import loadxml
	
	wwords = {'WikiWord':'http://wikiword.com'}

	buf = u"""
Hello world [[boo|http://boodebr.org]]. Here is a WikiWord, there.
Escape this ~WikiWord.
"""

	xml, context = WikklyText_to_XML(buf, 'utf-8', False)
	
	#sio = StringIO(xml)
	#tree = ElementTree(None, sio)
	root = loadxml(xml)
	
	print "ORIGINAL TREE",xmltrace(root)
	wikiwordify(root, context, wwords)
	#print "GOT",xmltrace(e)
	print "TREE AFTER WIKIFICATION",xmltrace(root)

if __name__ == '__main__':	
	#testsplit()	
	testwikify2()
