#!/usr/bin/python
# -*- coding: utf8 -*-
# -*- Mode: Python; py-indent-offset: 4 -*-

"""Globals and function needs for this project.

.. module:: utils
    :platform: Unix
    :synopsis: This module contains all useful globals and function needed.

.. moduleauthor:: Pierre Leray <pierreleray64@gmail.com>

"""

import argparse
import logging
import ConfigParser
import os
import json
from datetime import datetime
from xml.etree import ElementTree
from messager.utils import CURRENTCOST, ERROR, CC_INCORRECT_MESSAGE
from messager.utils import CURRENTCOST_UNICODE_ERROR, CURRENTCOST_TIMEOUT

# Logger initialization
LOGGER = logging.getLogger("currentcost")

INCORRECT_XML_MESSAGE = "Incorrect XML message => %s"
INCORRECT_CURRENTCOST_MISSING_TMPR = "Missing currentcost temperature value\
 in %s"
INCORRECT_CURRENTCOST_MISSING_WATTS = "Missing currentcost watts value in %s"
LOG_FILE_ERROR = "Error reading log config file, please verify file \"%s\"."
ACCESS_RIGHT_ERROR = "Error reading \"%s\". " \
    "You have not right to read/write on it."
VALUE = "value"
TIMESTAMP = "date"


def argument_parser():
    """Method that parse arguments from command line, return an error in
    case of missing parameter or return arguments with their value.

    :returns:  dict -- Dict containing arguments.

    """
    # Get command line arguments
    parser = argparse.ArgumentParser()
    # Define expected arguments
    parser.add_argument(
        "-v", "--verbose", help="activate verbose mode", action="store_true")
    parser.add_argument(
        "-c", "--config-file", help="path to configuration file")
    # Return list of argument passed in command line
    return parser.parse_args()


def init_message(variable_id, site_id, tty_port, log_conf):
    """Create log message starting current cost.

    :param variable_id: Name of the Variable.
    :type variable_id: str.

    :param site_id: Name of the Site.
    :type site_id: str.

    :param tty_port: TTY port path.
    :type tty_port: str.

    :param log_conf: Path to log configuration.
    :type log_conf: str.
    """
    # Create init message
    message = "Starting current cost application\n"
    message += "Current time: %s\n" % datetime.now()
    message += "Variable ID: %s\n" % variable_id
    message += "Site ID: %s\n" % site_id
    message += "TTY port: %s\n" % tty_port
    message += "Log configuration: %s\n" % log_conf
    # We log this message
    LOGGER.info(message)


def verbose_mode(verbose):
    """Active verbose mode.

    :param verbose: Boolean if true active verbose mode.
    :type variable_id: bool.

    """
    # If verbose mode is activated
    if verbose:
        # Create an handler to console and display log message
        sth = logging.StreamHandler()
        sth.setLevel(logging.INFO)
        # Add this handler to current logger
        LOGGER.addHandler(sth)


def data_validator(data, variable_id, site_id):
    """Analyse data from currentcost and return according TOPIC and MESSAGE.

    :param data: XML string that contain data.
    :type variable_id: str.

    :param variable_id: Name of the Variable.
    :type variable_id: str.

    :param site_id: Name of the Site.
    :type site_id: str.

    :returns:  str -- Topic of the message (error or success) and
        Message containing error description or data sent by CC.

    """
    # Initialization of variable
    topic = ERROR
    message = None
    # If data is empty, that means we reach a Timeout
    if data == "":
        message = CURRENTCOST_TIMEOUT % (variable_id, site_id)
    # Else we retrieve a string
    else:
        # We remove useless end-line and new-line caracteres
        data = data.replace("\n", "").replace("\r", "")
        try:
            # We expect a valid XML
            ElementTree.fromstring(data)
        except ElementTree.ParseError:
            try:
                # If it's not a valid XML, we expected it's a good string
                # and return an error
                message = CC_INCORRECT_MESSAGE % (
                    variable_id,
                    site_id,
                    data)
            except UnicodeDecodeError:
                # If it's not a good string, we send according message
                message = CC_INCORRECT_MESSAGE % (
                    variable_id,
                    site_id,
                    CURRENTCOST_UNICODE_ERROR)
        else:
            # Else, we return our message with according topic
            topic = CURRENTCOST
            message = data

    return topic, message


def create_series(value, variable_id, message):
    """Transform param into a series Object

        :param value: Value of the series
        :type value: str.

        :param variable_id: Variable identifiant
        :type variable_id: str.

        :param message: Information about value date
        :type message: dict.
    """
    return {
        "variableID": variable_id,
        "siteID": message["siteID"],
        "value": value,
        "date": message["date"],
        "dstTimezone": message["dstTimezone"],
        "nonDstTimezone": message["nonDstTimezone"]
    }


def get_last_series(site_id, variable_id, data_folder):
    """Read last value of a variable.
    """
    config = ConfigParser.ConfigParser()
    config.read("%s/%s.ini" % (data_folder, site_id))
    try:
        value = float(config.get(variable_id, VALUE))
        timestamp = config.get(variable_id, TIMESTAMP)
    except ConfigParser.NoSectionError:
        value = None
        timestamp = None
    return value, timestamp


def set_new_series(site_id, variable_id, value, timestamp, data_folder):
    """Fake read last value of a variable.
    """
    config = ConfigParser.ConfigParser()
    config.read("%s/%s.ini" % (data_folder, site_id))

    filename = "%s/%s.ini" % (data_folder, site_id)
    if not os.path.exists(data_folder):
        os.makedirs(data_folder)

    cfgfile = open(filename, "w")

    if variable_id not in config.sections():
        config.add_section(variable_id)

    config.set(variable_id, VALUE, value)
    config.set(variable_id, TIMESTAMP, timestamp)

    config.write(cfgfile)
    cfgfile.close()


def currentcost_kwh(energy_variable_id, power_variable_id, message, data_fold):
    """Function that compute kwh consumption from watts

        :param energy_variable_id: Energy variable id.
        :type energy_variable_id: str.

        :param watts_variable_id: Power variable id.
        :type watts_variable_id: str.

        :param message: Message from RabbitMQ.
        :type message: dict.
    """
    site_id = message["siteID"]
    last_energy_value, last_energy_date = get_last_series(
        site_id, energy_variable_id, data_fold)
    last_power_value, last_power_date = get_last_series(
        site_id, power_variable_id, data_fold)

    if (last_energy_value is not None and
            last_energy_date is not None and
            last_power_value is not None):

        last_energy_value = float(last_energy_value)
        last_power_value = float(last_power_value)
        last_energy_date = datetime.strptime(
            last_energy_date,
            "%Y-%m-%dT%H:%M:%S.%f")
        actual_date = datetime.strptime(
            message["date"],
            "%Y-%m-%dT%H:%M:%S.%f")
        delta = actual_date - last_energy_date
        ws = last_power_value * delta.total_seconds()
        wh = ws / 3600
        kwh = wh / 1000
        last_energy_value += kwh
        return round(last_energy_value, 6)
    else:
        return 0.0


def extract_w_and_wh(watts, key, old_value, message, variable_id, data_folder):
    """Extract watts and watts_hour
    """
    value = "old"
    value_kwh = None
    if watts != old_value:
        value = int(watts)
        value_kwh = currentcost_kwh(
            variable_id + "_kwh",
            variable_id,
            message,
            data_folder)
        set_new_series(
            message["siteID"],
            variable_id,
            value,
            message["date"],
            data_folder)
        set_new_series(
            message["siteID"],
            variable_id + "_kwh",
            value_kwh,
            message["date"],
            data_folder)
    return value, value_kwh


def check_currentcost_xml(xml_string):
    """Take a current cost XML string in output
    and extract according value or an error
    """
    ch1_w = None
    ch2_w = None
    ch3_w = None
    weather = None
    error = None
    hist = False
    try:
        xml = ElementTree.fromstring(xml_string)
        weather = xml.findtext("tmpr")
        hist = xml.findtext("hist")
        if hist is None:
            if weather is None:
                error = INCORRECT_CURRENTCOST_MISSING_TMPR % (
                    xml_string)
            else:
                weather = float(weather)
                if xml.find("ch1") is not None:
                    ch1_w = float(xml.find("ch1").findtext("watts"))
                if xml.find("ch2") is not None:
                    ch2_w = float(xml.find("ch2").findtext("watts"))
                if xml.find("ch3") is not None:
                    ch3_w = float(xml.find("ch3").findtext("watts"))
                if ch1_w is None and ch2_w is None and ch3_w is None:
                    error = INCORRECT_CURRENTCOST_MISSING_WATTS % (
                        xml_string)
        else:
            hist = True
    except ElementTree.ParseError:
        error = INCORRECT_XML_MESSAGE % xml_string

    return ch1_w, ch2_w, ch3_w, weather, error, hist


def currentcost_series(message, currencost_values, ts_args, data_folder):
    """Transform currentcost XML message into according series.

        :param message: Message from RabbitMQ.
        :type message: dict.
    """
    series = None
    ch1_w = currencost_values["ch1_w"]
    ch2_w = currencost_values["ch2_w"]
    ch3_w = currencost_values["ch3_w"]
    weather = currencost_values["tmpr"]

    series = []
    for key in ts_args:
        if ts_args[key]:
            value = None
            value_kwh = None
            old_value = get_last_series(
                message["siteID"], ts_args[key], data_folder)
            old_value = old_value[0]
            if key == "ch1_w" and ch1_w is not None:
                value, value_kwh = extract_w_and_wh(
                    ch1_w,
                    key,
                    old_value,
                    message,
                    ts_args[key],
                    data_folder)
            elif key == "ch2_w" and ch2_w is not None:
                value, value_kwh = extract_w_and_wh(
                    ch2_w,
                    key,
                    old_value,
                    message,
                    ts_args[key],
                    data_folder)
            elif key == "ch3_w" and ch3_w is not None:
                value, value_kwh = extract_w_and_wh(
                    ch3_w,
                    key,
                    old_value,
                    message,
                    ts_args[key],
                    data_folder)
            elif key == "tmpr" and weather is not None:
                value = "old"
                if weather != old_value:
                    value = weather
                    set_new_series(
                        message["siteID"],
                        ts_args[key],
                        value,
                        message["date"],
                        data_folder)

            if value is not None and value != "old":
                value = float(value)
                new_series = create_series(
                    value,
                    ts_args[key],
                    message)
                series.append(new_series)
                if value_kwh is not None:
                    value_kwh = float(value_kwh)
                    new_series_kwh = create_series(
                        value_kwh,
                        ts_args[key] + "_kwh",
                        message)
                    series.append(new_series_kwh)
    return series


def get_currentcost_message(topic, message, data, ts_a, data_folder):
    """Send currentcost message validated or according error
    """
    # If CurrentCost message
    series = []
    first_condition = ts_a["ch1_w"] or ts_a["ch2_w"]
    second_condition = ts_a["ch3_w"] or ts_a["tmpr"]
    if topic == CURRENTCOST:

        ch1_w, ch2_w, ch3_w, weather, error, hist = check_currentcost_xml(
            message["value"])

        # If XML message is corrupted
        if error:
            # We send this error
            topic = ERROR
            message["value"] = error
        elif (not hist and (first_condition or second_condition)):
            # We compute timeseries messages
            currencost_values = {
                "ch1_w": ch1_w,
                "ch2_w": ch2_w,
                "ch3_w": ch3_w,
                "tmpr": weather,
            }
            series = currentcost_series(
                message, currencost_values, ts_a, data_folder)

    return topic, json.dumps(message), series
