#!/usr/bin/python
# -*- coding: utf8 -*-
# -*- Mode: Python; py-indent-offset: 4 -*-
# pylint: disable=C0103

"""Db adapter for timevortex project"""

import logging
from time import tzname
from os import listdir, makedirs
from os.path import isfile, join, exists
from datetime import datetime
from dateutil import tz
from messager.utils import ERROR


class FileStorage(object):
    """Class that help us to store and load data over several file"""

    def __init__(self, folder_path):
        """Constructor"""
        self.folder_path = folder_path
        self.logger = logging.getLogger("filestorage")

    def insert_series(self, series):
        """Insert series in DB

            :param series: Representation of a series
            :type series: dict.
        """
        self.logger.error(series["value"])
        self.insert(series)

    def insert(self, message):
        """Insert data in file"""
        from_zone = tz.tzutc()
        to_zone = tz.tzlocal()
        file_folder = "%s/%s" % (self.folder_path, message["siteID"])
        file_date = datetime.strptime(
            message["date"], "%Y-%m-%dT%H:%M:%S.%f")
        file_date = file_date.replace(tzinfo=from_zone)
        file_date = file_date.astimezone(to_zone)
        file_date = file_date.strftime("%Y-%m-%d")

        if not exists(self.folder_path):
            makedirs(self.folder_path)

        if not exists(file_folder):
            makedirs(file_folder)

        raw_file = "%s/%s.tsv.%s" % (
            file_folder, message["variableID"], file_date)
        extracted = open(raw_file, "a+")
        extracted.write("%s\t%s\t%s\t%s\n" % (
            message["value"],
            message["date"],
            message["dstTimezone"],
            message["nonDstTimezone"]))
        extracted.close()

    def insert_error(self, message):
        """Function that store error in errors collection and in log

            :param message: Error to insert in DB
            :type message: str.
        """
        self.logger.error(message["value"])
        message["variableID"] = ERROR
        self.insert(message)

    def store_error(self, error):
        """Function that create valid error message

            :param error: Mal formed message
            :type error: str.
        """
        message = {
            "value": error,
            "variableID": ERROR,
            "siteID": "system",
            "date": datetime.utcnow().isoformat('T'),
            'dstTimezone': tzname[1],
            'nonDstTimezone': tzname[0]
        }
        self.logger.error(error)
        self.insert(message)

    def get_last_series(self, site_id, variable_id):
        """Retrieve last value of variable_id in site_id
        """
        element = variable_id
        file_prefix = "%s.tsv." % element
        site_folder = "%s/%s" % (self.folder_path, site_id)
        if exists(site_folder):
            old_date = ""
            last_filename = ""
            for filename in listdir(site_folder):
                is_file = isfile(join(site_folder, filename))
                if is_file and file_prefix in filename:
                    date = filename.replace(file_prefix, "")
                    try:
                        date = datetime.strptime(date, "%Y-%m-%d")
                        if (old_date == "" or date > old_date):
                            old_date = date
                            last_filename = filename
                    except ValueError:
                        self.logger.error("Not right file")

            last_filename = "%s/%s" % (site_folder, last_filename)
            with open(last_filename, "rb") as f:
                for last in f:
                    pass

            last = last.replace("\n", "")

            return {
                "variableID": element,
                "siteID": site_id,
                "value": last.split("\t")[0],
                "date": last.split("\t")[1],
                "dstTimezone": last.split("\t")[2],
                "nonDstTimezone": last.split("\t")[3]
            }

        return None

    def get_last_error(self, site_id):
        """Retrieve last error of a site_id file storage
        """
        return self.get_last_series(site_id, ERROR)

    def get_number_of_error(self, site_id, day_date):
        """This method retrieve number of error published for a day_date
        """
        element = ERROR
        site_folder = "%s/%s" % (self.folder_path, site_id)
        filename = "%s.tsv.%s" % (element, day_date)
        file_path = "%s/%s" % (site_folder, filename)
        if exists(site_folder) and exists(file_path):
            return sum(1 for line in open(file_path))
        return 0

    def get_number_of_series(self, site_id, day_date):
        """This method retrieve number of series published for a day_date
        """
        site_folder = "%s/%s" % (self.folder_path, site_id)
        series = {}

        if exists(site_folder):
            for filename in listdir(site_folder):
                if "%s.tsv" % ERROR not in filename and day_date in filename:
                    file_path = "%s/%s" % (site_folder, filename)
                    var_id = filename.replace(".tsv.%s" % day_date, "")
                    series_numbers = sum(1 for line in open(file_path))
                    series[var_id] = {"value": series_numbers}

        return series

    def set_data_location(self, folder_path):
        """Set data folder space"""
        self.folder_path = folder_path
