#!/usr/bin/python
# -*- coding: utf8 -*-
# -*- Mode: Python; py-indent-offset: 4 -*-

"""Globals and function needs for this project.

.. module:: utils
    :platform: Unix
    :synopsis: This module contains all useful globals and function needed.

.. moduleauthor:: Pierre Leray <pierreleray64@gmail.com>

"""

import argparse
import logging
from datetime import datetime

LOGGER = logging.getLogger("timeserieslogger")


def argument_parser():
    """Method that parse arguments from command line, return an error in
    case of missing parameter or return arguments with their value.

    :returns:  dict -- Dict containing arguments.

    """
    # Get command line arguments
    parser = argparse.ArgumentParser()
    # Define expected arguments
    parser.add_argument(
        "-v", "--verbose", help="activate verbose mode", action="store_true")
    parser.add_argument(
        "-c", "--config-file", help="path to configuration file")
    # Return list of argument passed in command line
    return parser.parse_args()


def init_message(log_conf):
    """Create log message starting current cost.

    :param log_conf: Path to log configuration.
    :type log_conf: str.
    """
    # Create init message
    message = "Starting timeserieslogger application\n"
    message += "Current time: %s\n" % datetime.now()
    message += "Log configuration: %s\n" % log_conf
    # We log this message
    LOGGER.info(message)


def verbose_mode(verbose):
    """Active verbose mode.

    :param verbose: Boolean if true active verbose mode.
    :type variable_name: bool.

    """
    # If verbose mode is activated
    if verbose:
        # Create an handler to console and display log message
        sth = logging.StreamHandler()
        sth.setLevel(logging.INFO)
        # Add this handler to current logger
        LOGGER.addHandler(sth)


def format_message(series_information, y_date, free_space):
    """This method take information from database and format it in according
    template

    :param series_information: Dict containing series informations.
    :type series_information: str.

    :returns: string -- E-mail containing formatting series informations.
    """
    subject = "[TimeVortex] Daily report of %s" % y_date

    series_str = "Series numbers for <b>%s</b>: <b>%s</b><br/>"
    error_str = "Error numbers: <b>%s</b><br/>"

    body = "Hello, <br/><br/>"
    body += "Here is your timevortex daily report for %s.<br/><br/>" % y_date

    for series in series_information:
        body += "Site <b>%s</b>: <br/>" % series["siteID"]
        if "errorNumbers" in series:
            body += error_str % series["errorNumbers"]
        if "seriesNumbers" in series:
            for variable, value in sorted(series["seriesNumbers"].items()):
                body += series_str % (
                    variable, value["value"])

        body += "<br/>"

    body += "%s<br/><br/>" % free_space

    body += "Thank you for your time."
    body = "" + body + ""

    return subject, body
