# -*- coding: utf-8 -*-

import time
import functools
from tornado.tcpserver import TCPServer
from tornado.ioloop import IOLoop

import kola
from kola import safe_call, logger
from .connection import Connection


class Kola(kola.Kola, TCPServer):
    def __init__(self, conn_class=None, *args, **kwargs):
        kola.Kola.__init__(self)
        TCPServer.__init__(self, *args, **kwargs)

        self.conn_class = conn_class or Connection

        self.repeat_timer_func_list = []
        self._configure_exc_handler()

    def handle_stream(self, stream, address):
        self.conn_class(self, stream, address).process()

    def run(self, host, port):
        self._start_repeat_timers()
        self.listen(port, host)
        IOLoop.instance().start()

    def _configure_exc_handler(self):
        """
        因为tornado的默认handler的log不一定会被配置
        """
        def exc_handler(callback):
            logger.error("Exception in callback %r", callback, exc_info=True)

        IOLoop.instance().handle_callback_exception = exc_handler

    def _start_repeat_timers(self):
        """
        把那些repeat timers启动
        """
        for func in self.repeat_timer_func_list:
            safe_call(func)
        # 执行blueprints
        for name, bp in self.blueprints.items():
            for func in bp.repeat_app_timer_func_list:
                safe_call(func)

    def repeat_timer(self, interval):
        """
        每隔一段时间执行(秒)
        """
        def inner_repeat_timer(func):
            @functools.wraps(func)
            def func_wrapper(*args, **kwargs):
                # 每次也要再加入

                result = safe_call(func, *args, **kwargs)
                IOLoop.instance().add_timeout(time.time() + interval, func_wrapper)

                return result

            self.repeat_timer_func_list.append(func_wrapper)
            return func_wrapper
        return inner_repeat_timer

