import os
import sys
import argparse

from .version import VERSION
from .validators import *
from .tokens import Tokens


def main():
    parser = argparse.ArgumentParser(prog='tokens',
                                     formatter_class=argparse.RawDescriptionHelpFormatter,
                                     epilog="")
    parser.add_argument('--version', action='version', version='tokens.io '+VERSION+" - https://tokens.io")
    subparsers = parser.add_subparsers(title='Token Commands')

    def subparse(endpoint, ignore=[], **more):
        new_parser = subparsers.add_parser(endpoint, help=getattr(Tokens, endpoint).__doc__)
        new_parser.set_defaults(func=endpoint)
        # always
        new_parser.add_argument('--no-color', action='store_true', help="remove terminal color")
        new_parser.add_argument('--json', action='store_true', help="json export")

        # sometimes
        if 'url' not in ignore: 
            new_parser.add_argument('--url', default=os.getenv("TOKENS_IO_URL", "https://tokens.io"), help="change endpoint url for **Elite** accounts", type=str)
        if 'env' not in ignore: 
            new_parser.add_argument('--env', default="development", help="(future) choose environment to manage tokens with", type=str)
        if 'slug' not in ignore: 
            new_parser.add_argument('--slug', help="repo slug (:service/:owner/:repo), default `git remote show origin`", type=str)
        if 'auth' not in ignore: 
            new_parser.add_argument('--auth', default=os.getenv("TOKENS_IO_AUTH"), help="your authorization token (user:secret)", type=str)

        #custom
        for arg, kwargs in more.items():
            arg = ("--%s"%arg[1:]) if arg[0]=='_' else arg
            new_parser.add_argument(*[arg, kwargs.pop('_')] if '_' in kwargs else [arg], **kwargs)

        return new_parser
        
    
    subparse('get', keys=dict(nargs="*", help="", type=str), 
                    _ref=dict(help="lookup value at ref (sha, @user or branch)", type=str))
    
    subparse('set', tokens=dict(nargs="+", help="list of KEY=VALUE", type=str))
    
    subparse('delete', keys=dict(nargs="+", help="", type=str))
    
    subparse('meta', key=dict(nargs=1, help="", type=str), 
                     meta=dict(nargs="+", help="meta values :key=:value", type=str))
    
    subparse('pull', ref=dict(nargs="?", help="pull tokens from ref (sha, @user or branch)", type=str))
    
    subparse('push', _c=dict(_="--comment", help="optional comment to push with key changes", type=str))
    
    subparse('diff', diff=dict(nargs="*", help="", type=str))
    
    subparse('share')
    
    subparse('flush', ignore=['url', 'env', 'slug', 'auth'])
    
    subparse('compare', compare=dict(nargs="+", help="compare cross environment (:env :ref :env :ref)", type=str))
    
    if len(sys.argv) > 1:
        # argparse sys args
        args = parser.parse_args()
        # build token api object
        tokens = Tokens(args)
        # call the tokens object
        data = getattr(tokens, args.func)()
        # write details out
        tokens.write(data)
    else:
        parser.print_help()

    sys.stdout.write("\n") # for a pretty shell


if __name__ == '__main__':
    main()
