import os
import sys
import valideer
import commands
import requests
import functools
from json import dumps
from urllib import urlencode
from mako.template import Template
from valideer import Nullable as null
from mako.lookup import TemplateLookup
from valideer import HomogeneousSequence

from .version import VERSION
from .validators import List
from .validators import Object
from .variables import Variables


def endpoint(method):
    @functools.wraps(method)
    def wrapper(self, **kwargs):
        try:
            # not used in the cli
            self._user_args.update(kwargs)
            # valudate the user_args provided in construct
            json = method(self) or "success"
            if type(json) is str:
                json = dict(meta=dict(status=200, result=True, message=json))
            json.setdefault('meta', dict(status=200, result=True, message="success"))

        except AssertionError as e:
            json = dict(meta=dict(status=400, result=False, message=str(e)))

        except valideer.ValidationError as e:
            json = dict(meta=dict(status=400, result=False, message=str(e), context=e.context))

        except Exception as e:
            json = dict(meta=dict(status=500, result=False, message=str(e)))

        # add default data
        if self._linked:
            json.setdefault('linked', {}).update(self._linked)
        json['locals'] = self.locals
        json['meta']['action'] = method.func_name
        return json

    return wrapper


class Tokens(object):
    schema = dict(auth="auth", url="url", env="env", key="?key", keys=null(HomogeneousSequence("key")), slug="?slug", comment="?string",
                  commit="?sha", tokens="?token-sequence", meta="?token-sequence", ref="?ref", json="bool", color="bool", 
                  compare=null(List("env", "ref", "env", "ref")), diff=null(List("?ref", "?ref")))

    def __init__(self, user_args):
        self._linked = {}
        self._locals, self._slug, self.results, self._args = (None, None, None, None)
        if type(user_args) is dict:
            user_args = Object(**user_args)
        self._user_args = dict(auth=(user_args.auth if "auth" in user_args else None) or os.getenv("TOKENS_IO_AUTH"), 
                               url=(user_args.url if "url" in user_args else None) or os.getenv("TOKENS_IO_URL", "https://tokens.io"),
                               env=(user_args.env if "env" in user_args else None) or "development",
                               slug=(user_args.slug if "slug" in user_args else None) or os.getenv("TOKENS_IO_SLUG", None),
                               diff=user_args.diff if 'diff' in user_args else None,
                               compare=user_args.compare if 'compare' in user_args else None,
                               json=user_args.json or False,
                               color=not user_args.no_color,
                               ref=user_args.ref if 'ref' in user_args else None,
                               key=user_args.key[0] if 'key' in user_args else None,
                               keys=user_args.keys if 'keys' in user_args else None,
                               tokens=user_args.tokens if 'tokens' in user_args else None,
                               comment=user_args.comment if 'comment' in user_args else None,
                               meta=user_args.meta if 'meta' in user_args else None)

    @property
    def args(self):
        "validated user args"
        if not self._args:
            self._args = valideer.parse(self.schema).validate(self._user_args)
        return self._args

    @property
    def ref(self):
        return (self['ref'] or dict(branch=self.branch)).values()[0]

    @property
    def user(self):
        return self.auth.split(':', 1)[0]

    @property
    def secret(self):
        return self.auth.split(':', 1)[1]

    @property
    def branch(self):
        branch = commands.getstatusoutput('git rev-parse --abbrev-ref HEAD')[1]
        return None if branch == 'HEAD' else branch

    @property
    def head(self):
        return commands.getstatusoutput('git rev-parse HEAD')[1]

    @property
    def slug(self):
        "returns (service, owner, repo)"
        if self._slug:
            return self._slug
        elif self['slug']:
            return self['slug']
        try:
            remote = commands.getstatusoutput('git remote show origin | grep "Fetch URL"')[1].split('Fetch URL: ')[-1]
            if 'github.com' in remote:
                if '@github.com:' in remote:
                    # git@github.com:owner/repo.git
                    owner, repo = tuple(remote.split('@github.com:')[-1][:-4].split('/', 1))
                else:
                    # https://github.com/owner/repp.git
                    owner, repo = tuple(remote.split('github.com/')[-1][:-4].split('/', 1))
                return Object(service="github", owner=owner, repo=repo)

            elif 'bitbucket.org' in remote:
                # https://user@bitbucket.org/owner/repo.git
                owner, repo = tuple(remote.split('@bitbucket.org/')[-1][:-4].split('/', 1))
                return Object(service="bitbucket", owner=owner, repo=repo)

        except:
            sys.exit("failed to parse your git origin using $ git remote show origin")

        else:
            sys.exit("failed to parse your git origin using $ git remote show origin (origin url required)")    

    @property
    def service(self):
        return self.slug.service

    @property
    def owner(self):
        return self.slug.owner
    
    @property
    def repo(self):
        return self.slug.repo

    @property
    def clean(self):
        "clean working directory"
        return (commands.getstatusoutput('git status -s')[1] == "")

    @property
    def spot(self):
        "returns the most reliable reference point"
        return dict([("commit", self.head) if self.clean else ("branch", self.branch)])

    @property
    def locals(self):
        """return dict of local .env variables
        """
        if not self._locals:
            self._locals = Variables.from_env()
        return self._locals

    def __getitem__(self, index):
        return self.args[index]

    def __getattr__(self, index):
        return self.args[index]

    def _url(self, more=None, urlargs=None):
        url = "/".join((self.url, self.service, self.owner, self.repo) + tuple(more or ()))
        if urlargs:
            url = "%s?%s" % (url, urlencode(urlargs))
        return url

    def _fetch(self, method, *more, **body):
        body = dict([(k,str(v)[0].lower() if v in (True, False) else v) for k,v in body.items() if v is not None])
        result = getattr(requests, method)(self._url(filter(lambda a: a, more), body if method in ('get', 'delete') else None),
                                           headers={"User-Agent": "tokens@v%s"%VERSION},
                                           auth=(self.user, self.secret),
                                           data=dumps(body) if method in ('post', 'put', 'patch') else None)
        json = result.json() if result.status_code != 204 else dict(meta=dict(result=True, status=204))
        json['meta'].update(dict(url=result.url,
                                 owner=self.slug.owner,
                                 repo=self.slug.repo,
                                 ref=self.ref))
        self._linked.update(json.get('linked', {}))
        return json['meta']['result'], json

    def write(self, data):
        assert type(data) is dict, "can only write dict results"
        if self.json:
            sys.stdout.write(dumps(data, indent=2, separators=(',', ': ')))
        else:
            mako = Template(#module_directory=os.path.join(os.path.dirname(__file__), "./templates"),
                            filename=os.path.join(os.path.dirname(__file__), "./templates/%s.mako"%(data['meta']['action'] if data['meta']['result'] else 'error')),
                            lookup=TemplateLookup(directories=[os.path.join(os.path.dirname(__file__), "./templates")]))
            sys.stdout.write(mako.render(this=self, t=self.table, c=self._color, **data))

    def _color(self, text, color):
        if self.color:
            colors = dict(grey="90", red="91", green="92", yellow="93", blue="94", pink="95", teal="96")
            if text:
                return "\033[%sm%s\033[0m" % (colors.get(color, '99'), str(text))
            else:
                return "\033[90mnull\033[0m"
        else:
            return str(text or 'null')

    def table(self, headers, rows, **colors):
        """
        Thanks to http://stackoverflow.com/questions/5909873/python-pretty-printing-ascii-tables
        headers = ["col", "col"]
        rows = [{"col1": 1, "col2": 2}]
        colors = {"col1": "red", "col2": "green}
        """
        if len(rows) > 1:
            lens = dict([(h, len(max([x[h] for x in rows] + [h],key=lambda x:len(str(x))))) for h in headers])
            formats = [("%%-%ds" % lens[h]) for h in headers]
            pattern = " | ".join(formats)
            separator = "-+-".join(['-' * lens[h] for h in headers])
            results = [pattern % tuple(headers), separator]
            results.extend([(pattern % tuple([self._color(line[k], colors.get(k)) for k in headers])) for line in rows])
            return "\n".join(results)
        elif len(rows) == 1:
            row = rows[0]
            hwidth = len(max(row._fields,key=lambda x: len(x)))
            return "\n".join(["%*s = %s" % (hwidth,row._fields[i],row[i]) for i in range(len(row))])

    @endpoint
    def get(self):
        "get value of token(s) from server"
        ok, json = self._fetch('get', self.env, ",".join(self.keys) if self.keys else None, ref=self.ref)
        return json

    @endpoint
    def set(self):
        "set tokens, list of KEY=value"
        # update the local .env
        self.locals.update(self.tokens)
        if self.clean:
            self.share()
        return dict(new=self.tokens)

    @endpoint
    def delete(self):
        "remove/delete token"
        # delete local env keys
        self.locals.remove(*self.keys)
        return dict(deleted=self.keys)

    @endpoint
    def flush(self):
        "remove all token locally"
        # delete local env keys
        self.locals.clear()

    @endpoint
    def meta(self):
        "set meta data for a token, ex. SERVICE=heroku URL=..."
        ok, json = self._fetch('patch', self.env, self.key, meta=self['meta'])
        return json

    @endpoint
    def pull(self):
        "fetch and override local tokens"
        ok, json = self._fetch('get', self.env, ref=self.ref)
        if ok:
            self.locals.clear()
            self.locals.update(json['tokens'])
        return json

    @endpoint
    def push(self):
        "commit token changes and push to server"
        assert self.clean, "you must be on a clean branch to push"
        ok, json = self._fetch('post', self.env, tokens=self.locals, branch=self.branch, sha=self.head, comment=self.comment)
        if ok: self.share()
        return json

    @endpoint
    def share(self):
        "share your tokens, helps for debugging"
        ok, json = self._fetch('put', "@%s"%self.user, tokens=self.locals, **self.spot)
        return json

    @endpoint
    def diff(self):
        "show token differences from local to remote ref"
        d = self['diff'] or []
        if len(d) == 2:
            ok, json = self._fetch('get', self.env, 'diff', "...".join(d[0]))
        else:
            ok, json = self._fetch('get', self.env, ref=d[0] if d else None)
            json['diff'] = [key for key in set(tuple(self.locals.keys()) + tuple(json['tokens'].keys()))\
                            if self.locals.get(key)!=json['tokens'].get(key)]
        return json

    @endpoint
    def compare(self):        
        "compare key differences between two environments references"
        ok, json = self._fetch('get', 'compare', ("%s@%s...%s@%s" % (self.env, self.ref, self.env2, self.ref2)))
        return json
