import os
import re
import valideer

class Object(dict):
    def __getattr__(self, index):
        return self.get(index)

class RegexpGroup(valideer.String):
    def validate(self, value, adapt=True):
        super(RegexpGroup, self).validate(value, adapt=adapt)
        matches = self.regexp.match(value)
        if not matches:
            self.error(value)
        return Object(**matches.groupdict()) if adapt else value

class key(valideer.Pattern):
    name = "key"    
    regexp = re.compile(r"^[A-Z\-\_]{1,45}$")

class value(valideer.Pattern):
    name = "value"
    regexp = re.compile(r"^.{,255}$")

class tokens(valideer.Mapping):
    name = "tokens"
    _key_validator = valideer.parse("key")
    _value_validator = valideer.parse("value")
    def __init__(self): pass

class token(RegexpGroup):
    name = "token"
    regexp = re.compile(r"^(?P<key>[A-Z\-\_]{1,45})=(?P<value>.{1,255})$")

class token_seq(valideer.HomogeneousSequence):
    """
    "KEY=value HERO=batman" => {"KEY": "value", "HERO": "batman"}
    """
    name = "token-sequence"
    _item_validator = valideer.parse("token")
    _min_length, _max_length = None, None
    def __init__(self): pass
    def validate(self, value, adapt=True):
        value = super(token_seq, self).validate(value)
        if adapt:
            return dict(map(lambda g: (g.key, g.value), value))
        return value

class List(valideer.Validator):
    schemas = None
    validator = None
    _min_length, _max_length = None, None

    def __init__(self, *schemas):
        if schemas:
            self.schemas = schemas
        self.schemas = valideer.parse(dict([(str(i), s) for i, s in enumerate(self.schemas)]), 
                                      additional_properties=False)

    def validate(self, value, adapt=True):
        if type(value) not in (tuple, list):
            self.error(value)
        try:
            nv = self.schemas.validate(dict([(str(i), s) for i, s in enumerate(value)]))
        except valideer.ValidationError as e:
            self.error(e.context)
        if adapt:
            return nv
        return value

class uuid(valideer.Pattern):
    name = "uuid"    
    regexp = re.compile(r"^[0-9a-f]{8}(-?[0-9a-f]{4}){3}-?[0-9a-f]{12}$")

class auth(valideer.Pattern):
    name = "auth"    
    regexp = re.compile(r"^(?P<user>[a-z\d\-]{1,25}):(?P<secret>[\w]{20})$")

class env(valideer.Enum):
    name = "env"    
    values = ("development", "staging", "production")

class user(valideer.Pattern):
    name = "user"
    regexp = re.compile(r"^(?P<user>\@[a-z\d\-\_]{1,25})$")

class slug(RegexpGroup):
    name = "slug"
    regexp = re.compile(r"^(?P<service>github|bitbucket)/(?P<owner>[\w\-\.]{1,45})/(?P<repo>[\w\-\.]{1,45})")

class sha(valideer.Pattern):
    name = "sha"
    regexp = re.compile(r"^(?P<sha>[a-z\d]{7,40})$")

class handler(valideer.Pattern):
    name = "handler"
    regexp = re.compile(r"^[\w\-\.]{1,45}$")
    def validate(self, value, adapt=True):
        super(handler, self).validate(value)
        return str(value).lower()

class branch(valideer.Pattern):
    name = "branch"
    regexp = re.compile(r"^(?P<branch>[a-z\_\-\.]{1,45})$")

class ref(RegexpGroup):
    name = "ref"
    regexp = re.compile(r"^(?P<ref>%s)$" % "|".join((sha.regexp.pattern[1:-1], user.regexp.pattern[1:-1], branch.regexp.pattern[1:-1])))
    def validate(self, value, adapt=True):
        value = super(ref, self).validate(value, adapt=adapt)
        if adapt:
            return dict([(k,v) for k,v in value.items() if v is not None and k!='ref'])
        return value

class url(valideer.Pattern):
    name = "url"
    regexp = re.compile(r"^https?\:\/\/[\w\.\-]+(\:\d+)?$" if os.getenv('DEBUG')=="TRUE" else r"^https\:\/\/[\w\.\-]+(\:\d+)?$")

class service(valideer.Enum):
    name = "service"
    values = ('github', 'bitbucket')

class version(valideer.Pattern):
    name = "version"
    regexp = re.compile(r"^v?\d+\.\d+\.\d+$")

class boolean(valideer.Validator):
    name = "bool"
    others = {
        "y": True, "yes": True, "1": True, "t": True, "true": True,
        "no": False, "n": False, "0": False, "f": False, "false": False
    }
    def validate(self, value, adapt=True):
        if type(value) is bool:
            return value
        result = self.others.get(str(value).lower(), None)
        if result is not None:
            return result if adapt else value
        else:
            raise valideer.ValidationError("bool is not valid")
