from functools import wraps
from tornado.web import HTTPError

import cgitb
import json
import logging
import sys

logger = logging.getLogger(__name__)


def handle_exceptions(method):
    """
    Catches exceptions, logs them, and returns a generic error.

    This decorator will actually handle HTTPError exceptions properly and
    return the appropriate error code for this, along with the message json
    wrapped.

    This decorator **should** precede all others so as to also handle
    exceptions these might raise.
    """

    @wraps(method)
    def wrapper(self, *args, **kwargs):
        try:
            return method(self, *args, **kwargs)
        except HTTPError as e:
            logger.debug(e.log_message)
            self.set_header("Content-Type", "application/json")
            self.write(json.dumps({'message': e.log_message}, indent=2,
                                  cls=self.Encoder))
            self.set_status(e.status_code)
            self.finish()
        except Exception as e:
            logger.debug("Caught exception")
            logger.exception(cgitb.text(sys.exc_info()))

            self.set_header("Content-Type", "application/json")
            self.write(json.dumps({'message': "Internal server error"}))
            self.set_status(500)
            self.finish()

    return wrapper
